//-----------------------------------------------------------------------------
// AvatarChooser.cs
// (C) A. Schiffler, 2010-2011
//-----------------------------------------------------------------------------

namespace AvatarChooser
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Audio;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.GamerServices;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Input;
    using Microsoft.Xna.Framework.Media;
    using Microsoft.Xna.Framework.Net;
    using Microsoft.Xna.Framework.Storage;

    using NewGamePhysics.GraphicalElements;
    using NewGamePhysics.Utilities;

    /// <summary>
    /// This is the main type for your game
    /// </summary>
    public class AvatarChooser : Microsoft.Xna.Framework.Game
    {
        GraphicsDeviceManager graphics;
        SpriteBatch spriteBatch;
        Effect invertEffect;
        Screenshot screenshot;

        const int numFractals = 9;
        const int gridstep = 3;
        const int gridsize = 300;
        GumowskiMiraFractal[] fractals = new GumowskiMiraFractal[numFractals];
        private int mouseScroll = 0;
        private int currentIndex = -1;
        private Boolean released = false;
        private Boolean iterated = false;

        // a,mu,x1,y2,Scale,Iterations
        double[] parameters = new double[]
        {
            0.008, -0.7, 0, 0.5, 0.03,  3000000,
            0.008, -0.8, 0, 0.5, 0.024, 3000000,
            0.008, -0.9, 0, 0.5, 0.019, 3000000,

            0, -0.15, 0, 0.5, 0.03, 10000000,
            0, -0.2, 0.5, 0.0, 0.03, 10000000,
            0, -0.22, 0, 0.5, 0.016, 10000000,

            0, -0.31, 0, 0.5, 0.008, 5000000,
            0, -0.55, 0, 0.5, 0.02, 5000000,
            0, -0.23, 0, 0.5, 0.02, 4000000,
        }; 

        public AvatarChooser()
        {
            graphics = new GraphicsDeviceManager(this);
            Content.RootDirectory = "Content";
            graphics.PreferredBackBufferWidth = gridsize * gridstep;
            graphics.PreferredBackBufferHeight = gridsize * gridstep;
            this.IsMouseVisible = true;
        }

        /// <summary>
        /// Allows the game to perform any initialization it needs to before starting to run.
        /// This is where it can query for any required services and load any non-graphic
        /// related content.  Calling base.Initialize will enumerate through any components
        /// and initialize them as well.
        /// </summary>
        protected override void Initialize()
        {            
            base.Initialize();
            this.screenshot = new Screenshot();
        }

        /// <summary>
        /// LoadContent will be called once per game and is the place to load
        /// all of your content.
        /// </summary>
        protected override void LoadContent()
        {
            // Create a new SpriteBatch, which can be used to draw textures.
            this.spriteBatch = new SpriteBatch(GraphicsDevice);
            invertEffect = Content.Load<Effect>(@"Effects\Invert");

            for (int i = 0; i < numFractals; i++)
            {
                fractals[i] = new GumowskiMiraFractal(GraphicsDevice, gridsize, gridsize, 
                    parameters[0 + 6 * i], 
                    parameters[1 + 6 * i], 
                    parameters[2 + 6 * i], 
                    parameters[3 + 6 * i]);
                fractals[i].Scale = (float)parameters[4 + 6 * i];
            }
        }

        /// <summary>
        /// UnloadContent will be called once per game and is the place to unload
        /// all content.
        /// </summary>
        protected override void UnloadContent()
        {
            // TODO: Unload any non ContentManager content here
        }

        /// <summary>
        /// Allows the game to run logic such as updating the world,
        /// checking for collisions, gathering input, and playing audio.
        /// </summary>
        /// <param name="gameTime">Provides a snapshot of timing values.</param>
        protected override void Update(GameTime gameTime)
        {
            // Allows the game to exit
            if (GamePad.GetState(PlayerIndex.One).Buttons.Back == ButtonState.Pressed)
                this.Exit();

            // Handle mouse and keyboard
            KeyboardState keyboardState = Keyboard.GetState();
            MouseState mouseState = Mouse.GetState();
            int mouseX = mouseState.X;
            int mouseY = mouseState.Y;
            double relX = (2.0 * (double)((mouseX % gridsize) - gridsize / 2)) / gridsize;
            double relY = (2.0 * (double)((mouseY % gridsize) - gridsize / 2)) / gridsize;
            int index = (mouseX / gridsize) + (mouseY / gridsize) * gridstep;
            this.currentIndex = -1;
            if ((index >= 0) && (index < numFractals))
            {
                this.currentIndex = index;
                if (this.released)
                {
                    if ((mouseState.LeftButton == ButtonState.Pressed) ||
                        (mouseState.RightButton == ButtonState.Pressed))
                    {
                        if (mouseState.LeftButton == ButtonState.Pressed)
                        {
                            fractals[index].Mu = fractals[index].Mu + 0.001;
                        }
                        else
                        {
                            fractals[index].Mu = fractals[index].Mu - 0.001;
                        }
                        fractals[index].Reset(relX, relY);
                        this.iterated = false;
                        this.released = false;
                    }

                    // Scrollwheel to zoom
                    if (this.mouseScroll != mouseState.ScrollWheelValue)
                    {
                        fractals[index].Scale = fractals[index].Scale + 0.00001f * (this.mouseScroll - mouseState.ScrollWheelValue);
                        this.mouseScroll = mouseState.ScrollWheelValue;
                        fractals[index].Reset(relX, relY);
                        this.iterated = false;
                        this.released = false;
                    }

                    // Keypress commands
                    // Handle keyboard
                    if (keyboardState.IsKeyDown(Keys.PrintScreen))
                    {
                        screenshot.TakeScreenshot(GraphicsDevice);
                        this.released = false;
                    }
                }
                else
                {
                    if ((mouseState.LeftButton == ButtonState.Released) &&
                        (mouseState.RightButton == ButtonState.Released) &&
                        (keyboardState.GetPressedKeys().Length == 0))
                    {
                        this.released = true;
                        this.mouseScroll = mouseState.ScrollWheelValue;
                    }
                }
            }


            // Update fractals
            if (!this.iterated)
            {
                this.iterated = true;
                for (int i = 0; i < numFractals; i++)
                {
                    int toIterate = (int)parameters[5 + 6 * i];
                    if (fractals[i].Iterations < toIterate)
                    {
                        this.iterated = false;
                        fractals[i].Iterate(10000);
                    }
                }
            }

            base.Update(gameTime);
        }

        /// <summary>
        /// This is called when the game should draw itself.
        /// </summary>
        /// <param name="gameTime">Provides a snapshot of timing values.</param>
        protected override void Draw(GameTime gameTime)
        {            
            GraphicsDevice.Clear(Color.CornflowerBlue);

            this.spriteBatch.Begin();
            for (int i = 0; i < numFractals; i++)
            {
                Color currentColor = Color.White; 
                this.spriteBatch.Draw(fractals[i].ImageTexture, new Vector2(gridsize * (i % gridstep), gridsize * (i / gridstep)), currentColor);
            }
            this.spriteBatch.End();

            base.Draw(gameTime);
        }
    }
}
