﻿//-----------------------------------------------------------------------------
// GeigerRandom.cs
// (A. Schiffler, 2009-2011)
//-----------------------------------------------------------------------------

namespace GeigerRandom
{
    using System;
    using System.IO;
    using NewGamePhysics.Devices;
    using NewGamePhysics.Mathematics;
    using NewGamePhysics.Utilities;
    
    /// <summary>
    /// Sample program that collects random bits from an Aware geiger counter
    /// using the a device helper class, unbiases them and then 
    /// and stores the data in data.txt (ASCII) and data.dat (binary)
    /// for statistical analysis.
    /// </summary>
    class GeigerRandom
    {
        private const int arraySize = 1024;
        private static int arrayPos = 0;

        private static double lastTime;
        private static double[] data = new double[arraySize];
        private static ValueUnbiasAlgorithm unbiasAlgorithm = ValueUnbiasAlgorithm.Partition;
        private static ValueUnbiaser unbiaser = new ValueUnbiaser(unbiasAlgorithm);
        private static int totalBits = 0;

        private static string dataAscii = "data.txt";
        private static string dataBytes = "data.dat";

        static void Main(string[] args)
        {
            System.Console.Error.WriteLine("=== Aware Geiger Random Generator ===");
            System.Console.Error.WriteLine(" Unbias algorithm: " + unbiasAlgorithm);
            System.Console.Error.WriteLine(" Block size: " + arraySize);

            if (File.Exists(dataAscii))
            {
                File.Delete(dataAscii);
            }

            if (File.Exists(dataBytes))
            {
                File.Delete(dataBytes);
            }

            try
            {
                AwareGeigerCounter awareCounter = new AwareGeigerCounter(args[0]);
                lastTime = HighResolutionTimer.Seconds();
                awareCounter.EventHandler = CounterEventHandler;
            }
            catch (Exception e)
            {
                System.Console.Error.WriteLine("Error: " + e.Message);
                System.Console.Error.WriteLine(e);
                return;
            }

            System.Console.Error.WriteLine("Press key to stop ...");
            System.Console.ReadKey(false);            
        }

        /// <summary>
        /// Event handler for counter events
        /// </summary>
        /// <param name="relativeTime"></param>
        private static void CounterEventHandler(double relativeTime)
        {
            // Capture relative time
            data[arrayPos] = relativeTime;
            arrayPos++;
            if (arrayPos == arraySize)
            {
                string bits = unbiaser.Extract(data);
                
                // Write bytes
                byte accumulator = 0;
                int bitcount = 0;
                FileStream fs = File.Open(dataBytes, FileMode.OpenOrCreate | FileMode.Append);
                BinaryWriter br = new BinaryWriter(fs);
                foreach (char bit in bits)
                {
                    accumulator *= 2;
                    accumulator |= (bit == '0') ? (byte)0 : (byte)1;                    
                    bitcount++;
                    if (bitcount == 8)
                    {
                        br.Write(accumulator);
                        bitcount = 0;
                        accumulator = 0;
                    }
                }
                br.Close();
                fs.Close();

                // Write ASCII
                fs = File.Open(dataAscii, FileMode.OpenOrCreate | FileMode.Append);
                TextWriter sr = new System.IO.StreamWriter(fs);
                sr.Write(bits);
                sr.Close();
                fs.Close();

                totalBits += bits.Length;
                System.Console.Error.WriteLine("Total Bits = {0}   Current Bits = {1} from {2} samples    Yield = {3}%", totalBits, bits.Length, arraySize, 100.0 * (double)bits.Length / (double)arraySize);
                arrayPos = 0;
            }
        }
    }
}
