//-----------------------------------------------------------------------------
// GravityChooserMainMenuScreen.cs
// (A. Schiffler, 2009-2011)
//-----------------------------------------------------------------------------

namespace GravityChooser
{
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Graphics;
    using NewGamePhysics.StateManager;
    using NewGamePhysics.Physics;
    using NewGamePhysics.GraphicalElements;
    using System;
    using System.IO;
    using System.Text;

    /// <summary>
    /// The main menu screen is the first thing displayed when the game starts up.
    /// </summary>
    class GravityChooserMainMenuScreen : MenuScreen
    {
        /// <summary>
        /// Text scroller for informational message.
        /// </summary>
        private ScrollingText scrollingInfoText;

        /// <summary>
        /// Menu entry for choosing gravity
        /// </summary>
        private MenuEntry chooseValueMenuEntry;

        /// <summary>
        /// Constructor fills in the menu contents.
        /// </summary>
        public GravityChooserMainMenuScreen()
            : base("Gravity Chooser")
        {
            // Create our menu entries.
            this.chooseValueMenuEntry = new MenuEntry("*"); // text will be set later
            MenuEntry optionsMenuEntry = new MenuEntry("Change Celestial Body");
            MenuEntry helpMenuEntry = new MenuEntry("Module Controls Help");
            MenuEntry infoMenuEntry = new MenuEntry("Module Information");
            MenuEntry doneMenuEntry = new MenuEntry("Done");

            // Hook up menu event handlers.
            chooseValueMenuEntry.Selected += ChooseValueMenuEntrySelected;
            optionsMenuEntry.Selected += OptionsMenuEntrySelected;
            infoMenuEntry.Selected += InfoMenuEntrySelected;
            helpMenuEntry.Selected += HelpMenuEntrySelected; 
            doneMenuEntry.Selected += OnCancel;

            // Add entries to the menu.
            MenuEntries.Add(chooseValueMenuEntry);
            MenuEntries.Add(optionsMenuEntry);
            MenuEntries.Add(infoMenuEntry);
            MenuEntries.Add(helpMenuEntry);
            MenuEntries.Add(doneMenuEntry);
        }

        public override void LoadContent()
        {
            // Create scrollers
            SpriteFont font = ScreenManager.Fonts["menu"];
            GravityChooser game = (GravityChooser)ScreenManager.Game;
            int width = game.GraphicsDevice.Viewport.Width;
            scrollingInfoText = new ScrollingText("-", font, width, 400); // text will be set later           

            base.LoadContent();
        }


        /// <summary>
        /// Update the main menu screen.
        /// </summary>
        /// <param name="gameTime">The current game time.</param>
        /// <param name="otherScreenHasFocus">
        /// Flag indicating of another screen has the focus.
        /// </param>
        /// <param name="coveredByOtherScreen">
        /// Flag indicating of the screen is covered by another screen.
        /// </param>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus, bool coveredByOtherScreen)
        {
            if (WaitForUncover)
            {
                if ((!coveredByOtherScreen) || (!otherScreenHasFocus))
                {
                    // Update the texts
                    UpdateAllTexts();

                    // Update menu text
                    WaitForUncover = false;
                }
            }

            // Update scrollers
            scrollingInfoText.Update(gameTime);

            base.Update(gameTime, otherScreenHasFocus, coveredByOtherScreen);
        }

        /// <summary>
        /// Draw the main menu screen.
        /// </summary>
        /// <param name="gameTime">The current game time.</param>
        public override void Draw(GameTime gameTime)
        {
            base.Draw(gameTime);

            // Draw scroller
            SpriteBatch spriteBatch = ScreenManager.SpriteBatch;
            scrollingInfoText.Draw(gameTime, spriteBatch);
        }

        #region Handle Input

        /// <summary>
        /// Event handler for when the Play Game menu entry is selected.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        void ChooseValueMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            GravityChooserLoadingScreen.Load(ScreenManager, true, e.PlayerIndex,
                               new GravityChooserGameplayScreen());

            // Set flag to update text when we return
            WaitForUncover = true;
        }

        /// <summary>
        /// Event handler for when the Options menu entry is selected.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        void OptionsMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            ScreenManager.AddScreen(new GravityChooserOptionsMenuScreen(), null);
            
            // Reset to play entry
            this.SelectedEntry = 0;

            // Set flag to update text when we return
            WaitForUncover = true;
        }

        /// <summary>
        /// When the user quits the main menu, returns code that sets gravity ENV variable.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        protected override void OnCancel(PlayerIndex playerIndex)
        {
            // Set environment batch code
            string batchCode = "set GRAVITY=" + GravityChooser.state.CurrentGravity.ToString();
            File.WriteAllText("gravity.bat", batchCode);
            
            ScreenManager.Game.Exit();
        }

        #endregion

        /// <summary>
        /// Event handler for when the Info menu entry is selected.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        void InfoMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            // Text to display
            string[] messages = { 
            "Information on The Gravity Chooser", 
            "",
            "The Gravity Chooser is an implementation of a hypothetical game",
            "component which allows players to select gravity (hence the name).",
            "",
            "The gravity is not a constant value, but specific to the location",
            "due to the non-uniform shape and density of any celestial body.",
            "",
            "Geodesy research has generated very high quality models of the",            
            "gravity of earth, some of which are available through the chooser.",
            "Other celestial bodies have significant different gravitatation",
            "which can be explored with the gravity chooser.",
            "",
            "The chooser returns the selected gravity for use by another game.",
            };

            ScreenManager.AddScreen(new GravityChooserHelpScreen(messages), null);

            // Reset back to play entry
            this.SelectedEntry = 0;

            // Set flag to update text when we return
            WaitForUncover = true;
        }

        /// <summary>
        /// Event handler for when the Help menu entry is selected.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        void HelpMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            // Text to display
            string[] messages = { 
            "Controls for The Gravity Chooser", 
            "",
            " [Up]/[Down] cursor/pad",
            " [Left]/[Right] cursor/pad",
            "            rotate the celestial body",
            " [Space] or (A)",
            "            continue or select in menu, game or help screens",
            " [Escape] or (Back)",
            "            pause game or quit to main menu",
            };

            ScreenManager.AddScreen(new GravityChooserHelpScreen(messages), null);

            // Reset back to play entry
            this.SelectedEntry = 0;

            // Set flag to update text when we return
            WaitForUncover = true;
        }

        /// <summary>
        /// Update menu and scroller text.
        /// </summary>
        private void UpdateAllTexts()
        {
            // Menu text
            chooseValueMenuEntry.Text =
                "Choose Your Gravity on " + GravityChooser.state.CurrentCelestialObject.ToString();

            // Scroller text
            if (null != scrollingInfoText)
            {
                scrollingInfoText.Text =
                        "Your selected gravity on " + GravityChooser.state.CurrentCelestialObject.ToString() + ": ";
                if (GravityChooser.state.CurrentGravity > 0.0)
                {
                    scrollingInfoText.Text = scrollingInfoText.Text +
                        "g = " + GravityChooser.state.CurrentGravity.ToString() + " m/s^-2";
                }
                else
                {
                    scrollingInfoText.Text = scrollingInfoText.Text +
                        "not set yet.";
                }
            }
        }
    }
}
