//-----------------------------------------------------------------------------
// GravityChooserOptionsMenuScreen.cs
// (A. Schiffler, 2009-2011)
//-----------------------------------------------------------------------------

namespace GravityChooser
{
    using System;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Graphics;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Utilities;
    using NewGamePhysics.Physics;
    using NewGamePhysics.GraphicalElements;
    
    /// <summary>
    /// The options for the gravity chooser.
    /// </summary>
    class GravityChooserOptionsMenuScreen : MenuScreen
    {
        /// <summary>
        /// Text scroller for informational message.
        /// </summary>
        private ScrollingText scrollingInfoText;

        /// <summary>
        /// The celestial object chooser option.
        /// </summary>
        private MenuEntry celestialObjectMenuEntry;

        /// <summary>
        /// The gravity model chooser option.
        /// </summary>
        private MenuEntry earthGravityModelMenuEntry;

        #region constructor

        /// <summary>
        /// Constructor for the options menu screen of the gravity chooser.
        /// </summary>
        public GravityChooserOptionsMenuScreen()
            : base("Gravity Chooser Options")
        {
            // Create our menu entries
            celestialObjectMenuEntry = new MenuEntry(string.Empty);
            celestialObjectMenuEntry.Selected += CelestialObjectMenuEntrySelected;
            MenuEntries.Add(celestialObjectMenuEntry);

            earthGravityModelMenuEntry = new MenuEntry(string.Empty);
            earthGravityModelMenuEntry.Selected += EarthGravityModelMenuEntrySelected;
            MenuEntries.Add(earthGravityModelMenuEntry);

            MenuEntry backMenuEntry = new MenuEntry("Back to Main Menu");
            backMenuEntry.Selected += OnCancel;
            MenuEntries.Add(backMenuEntry);
        }

        #endregion

        /// <summary>
        /// Load content and create graphical elements needed for this screen.
        /// </summary>
        public override void LoadContent()
        {
            base.LoadContent();

            // Create scroller
            SpriteFont font = ScreenManager.Fonts["menu"];
            int width = ScreenManager.GraphicsDevice.Viewport.Width;
            scrollingInfoText = new ScrollingText("-", font, width, 400); // text will be set later
            scrollingInfoText.ScrollerSpeed = 200.0f;
        }

        /// <summary>
        /// Update the main menu screen.
        /// </summary>
        /// <param name="gameTime">The current game time.</param>
        /// <param name="otherScreenHasFocus">
        /// Flag indicating of another screen has the focus.
        /// </param>
        /// <param name="coveredByOtherScreen">
        /// Flag indicating of the screen is covered by another screen.
        /// </param>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus, bool coveredByOtherScreen)
        {
            if (WaitForUncover)
            {
                if ((!coveredByOtherScreen) || (!otherScreenHasFocus))
                {
                    // Update the texts
                    UpdateAllTexts();

                    // Update menu text
                    WaitForUncover = false;
                }
            }

            // Update scrollers
            scrollingInfoText.Update(gameTime);

            base.Update(gameTime, otherScreenHasFocus, coveredByOtherScreen);
        }

        /// <summary>
        /// Draw the options screen.
        /// </summary>
        /// <param name="gameTime">The current game time.</param>
        public override void Draw(GameTime gameTime)
        {
            base.Draw(gameTime);

            // Draw scroller
            SpriteBatch spriteBatch = ScreenManager.SpriteBatch;
            scrollingInfoText.Draw(gameTime, spriteBatch);
        }

        #region private_methods

        /// <summary>
        /// Fills in the options screen texts from the game state.
        /// </summary>
        private void UpdateAllTexts()
        {
            UpdateMenuText();
            UpdateInfoText();
        }

        /// <summary>
        /// Fills in the options screen menu text.
        /// </summary>
        private void UpdateMenuText()
        {
            // Menu text
            celestialObjectMenuEntry.Text = 
                "Celestial Object: " + GravityChooser.state.CurrentCelestialObject.ToString();

            earthGravityModelMenuEntry.Text =
                    "Gravity Model: ";
            if (GravityChooser.state.CurrentCelestialObject == CelestialObject.Earth)
            {
                earthGravityModelMenuEntry.Text +=
                    GravityChooser.state.CurrentEarthGravityModel.ToString();
            }
            else
            {
                earthGravityModelMenuEntry.Text +=
                    "- not available -";
            }
        }

        /// <summary>
        /// Fills in the options screen info text.
        /// </summary>
        private void UpdateInfoText()
        {
            // Scroller text
            if (null != scrollingInfoText)
            {
                CelestialBody celestialBody = new CelestialBody(GravityChooser.state.CurrentCelestialObject);
                string infoText =
                    GravityChooser.state.CurrentCelestialObject.ToString() +
                    ": " +
                    celestialBody.GetBodyInfo();
                infoText = infoText.Replace("\n", "  *  ");
                scrollingInfoText.Text = infoText;
            }
        }

        #endregion

        #region event_handlers

        /// <summary>
        /// Event handler for the selection of the celestial object menu entry.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        private void CelestialObjectMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            // Select next item form the celestial body enumeration
            int currentIndex = EnumConvert.ChangeTo<int>(GravityChooser.state.CurrentCelestialObject);
            int maxIndex = Enum.GetValues(typeof(CelestialObject)).Length;
            currentIndex++;
            if (currentIndex == maxIndex)
            {
                currentIndex = 0;
            }

            GravityChooser.state.CurrentCelestialObject = EnumConvert.ChangeTo<CelestialObject>(currentIndex);

            // Reset gravity selection
            GravityChooser.state.CurrentGravity = 0.0;

            // Update menu
            UpdateAllTexts();
        }

        /// <summary>
        /// Event handler for the selection of the celestial object menu entry.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        private void EarthGravityModelMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            // Select next item form the celestial body enumeration
            int currentIndex = EnumConvert.ChangeTo<int>(GravityChooser.state.CurrentEarthGravityModel);
            int maxIndex = Enum.GetValues(typeof(EarthGravityModel)).Length;
            currentIndex++;
            if (currentIndex == maxIndex)
            {
                currentIndex = 0;
            }

            GravityChooser.state.CurrentEarthGravityModel = EnumConvert.ChangeTo<EarthGravityModel>(currentIndex);

            // Reset gravity selection
            GravityChooser.state.CurrentGravity = 0.0;

            // Update menu
            UpdateMenuText();
        }

        #endregion
    }
}
