﻿//-----------------------------------------------------------------------------
// TransmitterForm.cs
// (A. Schiffler, 2009-2011)
//-----------------------------------------------------------------------------

namespace InfoTransmitter
{
    using System;
    using System.Drawing;
    using System.IO;
    using System.Text;
    using System.Windows.Forms;
    using NewGamePhysics.Networking;

    /// <summary>
    /// UI for the InfoLink transmitter tool.
    /// </summary>
    public partial class TransmitterForm : Form
    {
        InfoLinkTransmitter infoLinkTransmitter;

        string userMessage = string.Empty;

        /// <summary>
        /// Creates an instance for the transmitter form,
        /// initializes the info link transmitter and starts
        /// the status timer.
        /// </summary>
        public TransmitterForm()
        {
            InitializeComponent();

            this.infoLinkTransmitter = new InfoLinkTransmitter();
            this.infoLinkTransmitter.StartTransmitter();

            this.timer1.Enabled = true;
            UpdateTransmitButton();
        }

        /// <summary>
        /// Handles the transmit button click event.
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">The event arguments.</param>
        private void buttonTransmit_Click(object sender, EventArgs e)
        {
            if (!string.IsNullOrEmpty(textBoxUrl.Text))
            {
                string url = textBoxUrl.Text;

                // Add http
                if (!url.Equals("about:blank") &&
                    !url.StartsWith("http://") &&
                    !url.StartsWith("https://"))
                {
                    url = "http://" + url;
                }

                Uri uri;
                try
                {
                    uri = new Uri(url, UriKind.Absolute);
                }
                catch (System.UriFormatException)
                {
                    this.userMessage = "Invalid URL ...";
                    this.textBoxUrl.ForeColor = Color.Red;
                    this.textBoxUrl.Focus();
                    return;
                }

                this.userMessage = "Sending ...";
                InfoLink infoLink = new InfoLink(
                    this.textBoxCategory.Text,
                    this.textBoxSubCategory.Text,
                    this.textBoxTitle.Text,
                    uri);               
                this.infoLinkTransmitter.Transmit(infoLink);

                this.textBoxTitle.Text = string.Empty;
                this.textBoxUrl.Text = string.Empty;
                this.textBoxTitle.Focus();
            }
        }

        /// <summary>
        /// Enables/disables the transmit button based the form completeness.
        /// </summary>
        private void UpdateTransmitButton()
        {
            this.buttonTransmit.Enabled =
                (!string.IsNullOrEmpty(this.textBoxCategory.Text)) &&
                (!string.IsNullOrEmpty(this.textBoxSubCategory.Text)) &&
                (!string.IsNullOrEmpty(this.textBoxTitle.Text)) &&
                (!string.IsNullOrEmpty(this.textBoxUrl.Text));
        }

        /// <summary>
        /// Track the URL entry changes.
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">The event arguments.</param>
        private void textBoxUrl_TextChanged(object sender, EventArgs e)
        {
            UpdateTransmitButton();
        }

        /// <summary>
        /// Track the category entry changes.
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">The event arguments.</param>
        private void textBoxCategory_TextChanged(object sender, EventArgs e)
        {
            UpdateTransmitButton();
        }

        /// <summary>
        /// Track the title entry changes.
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">The event arguments.</param>
        private void textBoxTitle_TextChanged(object sender, EventArgs e)
        {
            UpdateTransmitButton();
        }

        /// <summary>
        /// Track category keyboard entry.
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">The event arguments.</param>
        private void textBoxCategory_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                this.textBoxSubCategory.Focus();
            }
        }

        /// <summary>
        /// Track sub-category keyboard entry.
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">The event arguments.</param>
        private void textBoxSubCategory_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                this.textBoxTitle.Focus();
            }
        }

        /// <summary>
        /// Track title keyboard entry.
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">The event arguments.</param>
        private void textBoxTitle_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                this.textBoxUrl.Focus();
            }
        }

        /// <summary>
        /// Track url keyboard entry.
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">The event arguments.</param>
        private void textBoxUrl_KeyDown(object sender, KeyEventArgs e)
        {
            this.textBoxUrl.ForeColor = Color.Black;
            if (e.KeyCode == Keys.Enter)
            {
                this.buttonTransmit.Focus();
            }
        }

        /// <summary>
        /// Track form closures.
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">The event arguments.</param>
        private void TransmitterForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            this.infoLinkTransmitter.StopTransmitter();
            e.Cancel = false;
        }

        /// <summary>
        /// Timer callback for transmission status updates.
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">The event arguments.</param>
        private void timer1_Tick(object sender, EventArgs e)
        {
            long transmitted = this.infoLinkTransmitter.Transmitted;
            long errors = this.infoLinkTransmitter.Errors;

            string message = string.Empty;

            if (string.IsNullOrEmpty(this.userMessage))
            {
                if (transmitted > 0)
                {
                    message += string.Format("  Transmitted InfoLinks: {0}", transmitted);
                }

                if (errors > 0)
                {
                    message += string.Format("  Errors: {0}", errors);
                }
            }
            else
            {
                message = this.userMessage;
                this.userMessage = string.Empty;
            }

            this.labelSending.Text = message.Trim();
        }

        /// <summary>
        /// Menu handler to quick program.
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">The event arguments.</param>
        private void toolStripMenuItemExit_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        /// <summary>
        /// Menu handler for about box.
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">The event arguments.</param>
        private void aboutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            MessageBox.Show(
                "InfoLink Transmitter" + Environment.NewLine +
                "Transmitting UDP packets on port: " + this.infoLinkTransmitter.Port + Environment.NewLine +
                "(C) A. Schiffler, 2009-2011" + Environment.NewLine,
                "InfoTransmitter About");
        }

        /// <summary>
        /// Menu handler to load InfoLink .xml file.
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">The event arguments.</param>
        private void toolStripMenuItemLoad_Click(object sender, EventArgs e)
        {
            OpenFileDialog openFileDialog = new OpenFileDialog();
            openFileDialog.Filter = "InfoLink XML|*.xml|All|*.*";
            string startDir = Path.Combine(Environment.CurrentDirectory, "InfoLinkData");
            if (Directory.Exists(startDir))
            {
                openFileDialog.InitialDirectory = startDir;
            } 
            else
            {
                openFileDialog.InitialDirectory = Environment.CurrentDirectory;
            }

            if (openFileDialog.ShowDialog() == DialogResult.OK)
            {
                string fileContent = System.IO.File.ReadAllText(openFileDialog.FileName, Encoding.Unicode);
                InfoLink[] infoLinks = InfoLinkSerializer.DeserializeArray(fileContent);
                this.infoLinkTransmitter.Transmit(infoLinks);
                this.userMessage = "Sending " + infoLinks.Length + " ...";
            }
        }
    }
}
