﻿//-----------------------------------------------------------------------------
// MontyHallGameOverScreen.cs
// (C) A. Schiffler, 2009-2011
//-----------------------------------------------------------------------------

namespace MontyHallGame
{
    using System;
    using System.Threading;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using NewGamePhysics.StateManager;

    /// <summary>
    /// Shows the GameOver screen.
    /// </summary>
    class MontyHallGameOverScreen : MenuScreen
    {
        #region Initialization

        /// <summary>
        /// Constructor.
        /// </summary>
        public MontyHallGameOverScreen(string title)
            : base(title)
        {
            // Flag that there is no need for the game to transition
            // off when the pause menu is on top of it.
            IsPopup = true;

            // Create our menu entries.
            MenuEntry continueMenuEntry = new MenuEntry("Continue and Play Again");
            MenuEntry returnMenuEntry = new MenuEntry("Return to Main Menu");

            // Hook up menu event handlers.
            continueMenuEntry.Selected += OnContinue;
            returnMenuEntry.Selected += OnReturn;

            // Add entries to the menu.
            MenuEntries.Add(continueMenuEntry);
            MenuEntries.Add(returnMenuEntry);
        }

        #endregion

        #region Handle Input

        /// <summary>
        /// Event handler for Continue menu.
        /// </summary>
        void OnContinue(object sender, PlayerIndexEventArgs e)
        {
            MontyHallLoadingScreen.Load(ScreenManager, true, e.PlayerIndex,
                               new MontyHallGameplayScreen());
        }

        /// <summary>
        /// Event handler for Return menu.
        /// </summary>
        void OnReturn(object sender, PlayerIndexEventArgs e)
        {
            // Submit play time
            SubmitUsage(Convert.ToInt32(MontyHallGame.state.playTime));

            MontyHallLoadingScreen.Load(ScreenManager, false, null, new MontyHallBackgroundScreen(),
                                                           new MontyHallMainMenuScreen());
        }

        #endregion

        #region Draw

        /// <summary>
        /// Draws the menu screen. This darkens down the gameplay screen
        /// that is underneath us, and then chains to the base MenuScreen.Draw.
        /// </summary>
        public override void Draw(GameTime gameTime)
        {
            ScreenManager.FadeBackBufferToBlack(TransitionAlpha * 4 / 5);

            // Messages
            string[] messages = new string[4];
            messages[0] = "Statistics:";
            messages[1] = "  Games Won: " + MontyHallGame.state.Wins;
            messages[2] = "  Games Lost: " + MontyHallGame.state.Losses;
            messages[3] = "  Switched Doors: " + MontyHallGame.state.Switches + " times";

            int totalGames = MontyHallGame.state.Wins + MontyHallGame.state.Losses;
            if (totalGames > 0)
            {
                if (MontyHallGame.state.Wins > 0)
                {
                    messages[1] += "    (" +
                        string.Format("{0:.%}", (double)MontyHallGame.state.Wins / (double)totalGames) +
                        " of games)";
                }

                if (MontyHallGame.state.Losses > 0)
                {
                    messages[2] += "    (" +
                        string.Format("{0:.%}", (double)MontyHallGame.state.Losses / (double)totalGames) +
                        " of games)";
                }


                if (MontyHallGame.state.Switches > 0)
                {
                    messages[3] += "    (" +
                        string.Format("{0:.%}", (double)MontyHallGame.state.Switches / (double)totalGames) +
                        " of games)";
                }
            }

            // Draw score summary
            SpriteBatch spriteBatch = ScreenManager.SpriteBatch;
            SpriteFont font = ScreenManager.Fonts["retro"];
            Viewport viewport = ScreenManager.GraphicsDevice.Viewport;
            Color color = Color.White;
            Vector2 origin = new Vector2(0, 0);
            Vector2 position = new Vector2(viewport.Width * 0.25f, viewport.Height * 0.55f);
            spriteBatch.Begin();
            foreach (string message in messages)
            {
                spriteBatch.DrawString(
                    font,
                    message,
                    position,
                    color,
                    0,
                    origin,
                    1.0f,
                    SpriteEffects.None,
                    0);
                position.Y += (viewport.Height * 0.05f);
            }
            spriteBatch.End();

            base.Draw(gameTime);
        }

        #endregion

        /// <summary>
        /// Submit the usage duration in a background thread to webservice.
        /// </summary>
        /// <param name="usageSeconds">
        /// The seconds if usage on the gameplay screen.
        /// </param>
        private void SubmitUsage(int usageSeconds)
        {
            // Maybe submit
            Thread t;
            if (usageSeconds > 0)
            {
                t = new System.Threading.Thread(MontyHallGame.state.Rng.PlayTrulyRandom.OnlineUsage);
                t.Start(usageSeconds);
            }
        }

    }
}


