
using System;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;

using NewGamePhysics.Utilities;

namespace NewGamePhysics.GraphicalElements
{
    /// <summary>
    /// A collection of 2D points which are plotted.
    /// </summary>
    public class DotPlotter
    {
        /// <summary>
        /// The maximum number of dots in the plot.
        /// </summary>
        private int plotStackSize;

        /// <summary>
        /// The positions of each dot.
        /// </summary>
        Vector2[] plotPositions;

        /// <summary>
        /// The color of each segment.
        /// </summary>
        Color[] plotColors;

        /// <summary>
        /// The type of each point.
        /// </summary>
        int[] plotType;

        /// <summary>
        /// Head of position stack.
        /// </summary>
        private int headOfPositionStack;

        /// <summary>
        /// Tail of position stack.
        /// </summary>
        private int tailOfPositionStack;

        /// <summary>
        /// Number of points in stack.
        /// </summary>
        private int pointsInStack;

        /// <summary>
        /// Construct a new dotplotter
        /// </summary>
        /// <param name="count">The maximum number of dots in the plotter.</param>
        public DotPlotter(int count)
        {
            if (count <= 0)
            {
                throw new ArgumentOutOfRangeException("count");
            }

            // Keep size
            plotStackSize = count;
            pointsInStack = 0;

            // Pre-allocate
            plotPositions = new Vector2[plotStackSize];
            plotColors = new Color[plotStackSize];
            plotType = new int[plotStackSize];
            for (int i = 0; i < plotStackSize; i++)
            {
                plotPositions[i] = new Vector2(0.0f, 0.0f);
                plotColors[i] = new Color();
                plotType[i] = 0;
            }

            // Empty stack
            headOfPositionStack = 0;
            tailOfPositionStack = 0;
        }

        /// <summary>
        /// Draw a segment of the plot line.
        /// </summary>
        /// <param name="start">Start position.</param>
        /// <param name="finish">End position.</param>
        /// <param name="color">The color of the segment</param>
        /// <param name="emphasis">The emphasis type. 1=box.</param>
        /// <param name="primitiveBatch">The primitive batch object to use.</param>
        private void DrawSegment(Vector2 start, Vector2 finish, Color color, int emphasis, PrimitiveBatch primitiveBatch)
        {
            // Always draw line segment part
            primitiveBatch.AddVertex(start, color);
            primitiveBatch.AddVertex(finish, color);
            // Emphasis
            if (emphasis == 1)
            {
                float size = 2.0f;
                Vector2 ul = new Vector2(-size, -size);
                Vector2 ur = new Vector2( size, -size);
                Vector2 bl = new Vector2(-size,  size);
                Vector2 br = new Vector2( size,  size);
                primitiveBatch.AddVertex(finish + ul, color);
                primitiveBatch.AddVertex(finish + ur, color);
                primitiveBatch.AddVertex(finish + ur, color);
                primitiveBatch.AddVertex(finish + br, color);
                primitiveBatch.AddVertex(finish + br, color);
                primitiveBatch.AddVertex(finish + bl, color);
                primitiveBatch.AddVertex(finish + bl, color);
                primitiveBatch.AddVertex(finish + ul, color);
            }
        }

        /// <summary>
        /// Render the plotter.
        /// </summary>
        /// <param name="primitiveBatch">The primitive batch to use for drawing the lines.</param>
        /// <param name="dotColor">The color for the dots.</param>
        public void Draw(GameTime gameTime, PrimitiveBatch primitiveBatch)
        {
            if (primitiveBatch == null)
            {
                throw new ArgumentNullException("primitiveBatch");
            }

            // Do we have enough points
            if (pointsInStack > 1)
            {
                primitiveBatch.Begin(PrimitiveType.LineList);
                if (headOfPositionStack > tailOfPositionStack)
                {
                    for (int i = tailOfPositionStack; i < (headOfPositionStack - 1); i++)
                    {
                        DrawSegment(plotPositions[i], plotPositions[(i + 1) % plotStackSize],
                                    plotColors[i], plotType[i], primitiveBatch);
                    }
                }
                else if (headOfPositionStack < tailOfPositionStack)
                {
                    for (int i = tailOfPositionStack; i < plotStackSize; i++)
                    {
                        DrawSegment(plotPositions[i], plotPositions[(i + 1) % plotStackSize],
                                    plotColors[i], plotType[i], primitiveBatch);
                    }
                    if (headOfPositionStack > 0)
                    {
                        for (int i = 0; i < (headOfPositionStack - 1); i++)
                        {
                            DrawSegment(plotPositions[i], plotPositions[(i + 1) % plotStackSize],
                                        plotColors[i], plotType[i], primitiveBatch);
                        }
                    }
                }
                else
                {
                    // if head==tail there is nothing to do
                }
                primitiveBatch.End();
            }
        }

        /// <summary>
        /// Move all dots by the given amount.
        /// </summary>
        /// <param name="displacement"></param>
        public void MoveDots(Vector2 displacement)
        {
            for (int i=0; i<plotStackSize; i++) 
            {
                plotPositions[i] += displacement;
            }
        }

        /// <summary>
        /// Add a dot to the plot.
        /// </summary>
        /// <param name="targetPosition">The target position of the dot.</param>
        /// <param name="color">The color of the dot.</param>
        /// <param name="type">The emphasis type of the dot.</param>
        public void AddDot(Vector2 targetPosition, Color color, int type)
        {
            // Store plot position and attributes
            plotPositions[headOfPositionStack].X = targetPosition.X;
            plotPositions[headOfPositionStack].Y = targetPosition.Y;
            plotColors[headOfPositionStack] = color;
            plotType[headOfPositionStack] = type;

            // Advance head to next empty slot
            headOfPositionStack++;

            // Maybe wrap head
            if (headOfPositionStack == plotStackSize)
            {
                headOfPositionStack = 0;
            }

            // Maybe move tail
            if (headOfPositionStack == tailOfPositionStack)
            {
                // Also advance tail
                tailOfPositionStack++;

                // Maybe wrap tail
                if (tailOfPositionStack == plotStackSize)
                {
                    tailOfPositionStack = 0;
                }
            }

            // Count points
            if (pointsInStack < plotStackSize)
            {
                pointsInStack++;
            }
        }

        /// <summary>
        /// Set the type of the point at the top of the stack.
        /// </summary>
        /// <param name="type">The emphasis type of the dot.</param>
        public void SetTypeAtHead(int type)
        {
            // Do we have points?
            if (pointsInStack > 0)
            {
                if (headOfPositionStack > 0)
                {
                    // Previous position is last point added.
                    plotType[headOfPositionStack - 1] = type;
                }
                else
                {
                    // Last element in stack is the last point added.
                    plotType[plotStackSize - 1] = type;
                }
            }
        }
    }
}
