﻿using System;

using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;

using NewGamePhysics.StateManager;
using NewGamePhysics.Utilities;

namespace NewGamePhysics.GraphicalElements
{
    /// <summary>
    /// Base class for a graphical element in a screen managed by the
    /// screen manager.
    /// </summary>
    public class GraphicalElementBase
    {
        /// <summary>
        /// The screen manager.
        /// </summary>
        private ScreenManager screenManager;

        /// <summary>
        /// The global primitive batch obejct.
        /// </summary>
        private PrimitiveBatch primitiveBatch;

        /// <summary>
        /// The global sprite batch object.
        /// </summary>
        private SpriteBatch spriteBatch;

        /// <summary>
        /// The global viewport object;
        /// </summary>
        private Viewport viewport;

        /// <summary>
        /// Initializes a graphical element for a screen managed by a 
        /// screen manager.
        /// </summary>
        /// <param name="manager">The screen manager to use for this element.</param>
        public GraphicalElementBase(ScreenManager manager)
        {
            if (null == manager)
            {
                throw new ArgumentNullException(
                    "manager",
                    "Screen manager object cannot be null");
            }

            // Set manager and get drawing helpers
            this.screenManager = manager;
            this.primitiveBatch = this.screenManager.PrimitiveBatch;
            this.spriteBatch = this.screenManager.SpriteBatch;
            this.viewport = this.screenManager.GraphicsDevice.Viewport;
        }

        /// <summary>
        /// Gets the screen manager associated with the graphical element.
        /// </summary>
        public ScreenManager ScreenManager
        {
            get { return this.screenManager; }
        }

        /// <summary>
        /// Gets the sprite batch associated with the manager.
        /// </summary>
        public SpriteBatch SpriteBatch
        {
            get { return this.spriteBatch; }
        }

        /// <summary>
        /// Gets the primitive batch associated with the screen.
        /// </summary>
        public PrimitiveBatch PrimitiveBatch
        {
            get { return this.primitiveBatch; }
        }

        /// <summary>
        /// Gets the viewport associated with the screen.
        /// </summary>
        public Viewport Viewport
        {
            get { return this.viewport; }
        }
    }
}
