﻿// DrawableGravitySelector.cs
//

namespace NewGamePhysics.GraphicalElements
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Input;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Utilities;
    using NewGamePhysics.PhysicalElements;
    using NewGamePhysics.Mathematics;
    using NewGamePhysics.Physics;

    /// <summary>
    /// Represents a 3D animated gravity selector for XNA.
    /// </summary>
    public class GravitySelector
    {
        /// <summary>
        /// The screen manager for drawing.
        /// </summary>
        private ScreenManager screenManager;

        /// <summary>
        /// The direction the camera points without rotation: looking into the screen.
        /// </summary>
        private static Vector3 cameraReferenceVector = new Vector3(0.0f, 0.0f, -1.0f);

        /// <summary>
        /// The rotation of the camera: upright.
        /// </summary>
        private static Vector3 cameraUpVector = new Vector3(0.0f, 1.0f, 0.0f);

        /// <summary>
        /// The textured sphere representing the planet.
        /// </summary>
        private TexturedSphere planetSphere;

        /// <summary>
        /// The fixed projection matrix for viewer.
        /// </summary>
        private Matrix projectionMatrix;

        /// <summary>
        /// The fixed view matrix for viewer.
        /// </summary>
        private Matrix viewMatrix;

        /// <summary>
        /// Indicator showing currently selected gravity.
        /// </summary>
        private ValueIndicator gravityIndicator;

        /// <summary>
        /// Indicator showing currently selected latitude.
        /// </summary>
        private ValueIndicator latitudeIndicator;

        /// <summary>
        /// Indicator showing currently selected longitude.
        /// </summary>
        private ValueIndicator longitudeIndicator;

        /// <summary>
        /// Text scroller for the model and texture info.
        /// </summary>
        private ScrollingText scrollingInfoText;

        /// <summary>
        /// Current latitude value.
        /// </summary>
        private double latitude;

        /// <summary>
        /// Current longitude value.
        /// </summary>
        private double longitude;

        /// <summary>
        /// Current display size. 
        /// Range [0,2] 2 zoomed, 1 fully visible, 0 invisible.
        /// </summary>
        private float displaySize;

        /// <summary>
        /// The center of the screen.
        /// </summary>
        private Vector2 screenCenter;

        /// <summary>
        /// The color of the cross-hair indicator.
        /// </summary>
        private Color drawColor;

        /// <summary>
        /// The clestial body to display and select from.
        /// </summary>
        private CelestialBody celestialBody;

        /// <summary>
        /// Gravity calculator.
        /// </summary>
        private GravityCalculator gravityCalculator;

        /// <summary>
        /// Body surface texture.
        /// </summary>
        private Texture2D surfaceTexture;

        /// <summary>
        /// Current gravity value.
        /// </summary>
        private double gravity;

        /// <summary>
        /// Gets the current selected gravity value.
        /// </summary>
        public double Gravity
        {
            get { return this.gravity; }
        }

        /// <summary>
        /// Gets or sets the current display size. 
        /// 2 zoomed, 1 fully visible, 0 invisible;
        /// </summary>
        public float DisplaySize
        {
            get { return this.displaySize; }
            set 
            { 
                this.displaySize = value;

                // Lock to valid range
                if (this.displaySize < 0.0f)
                {
                    this.displaySize = 0.0f;
                }
                else if (this.displaySize > 2.0f)
                {
                    this.displaySize = 2.0f;
                }

                // Change display
                UpdateViewMatrix();
            }            
        }

        /// <summary>
        /// Constructor of the AnimatedGravitySelector instance.
        /// </summary>
        /// <param name="manager">The screen manager to use.</param>
        /// <param name="celestialObject">The celestial object to show and use for the selection.</param>
        /// <param name="gravityCalculator">The gravity calculator to use for determining the gravity.</param>
        public GravitySelector(
            ScreenManager manager, 
            CelestialObject celestialObject,
            GravityCalculator gravityCalculator)
        {
            // Keep reference to screen manager for drawing
            this.screenManager = manager;

            // Initialize body
            this.celestialBody = new CelestialBody(celestialObject);

            // Load texture (default variation)
            this.surfaceTexture = screenManager.Game.Content.Load<Texture2D>(
                this.celestialBody.GetTextureName(0));

            // Initialize gravity calculator and value
            this.gravityCalculator = gravityCalculator;
            this.gravity = this.gravityCalculator.Value;

            // Create textured sphere object
            this.planetSphere = new TexturedSphere(
                screenManager.GraphicsDevice, 
                36, 
                36, 
                1.0f, 
                this.surfaceTexture);

            // Center of screen
            this.screenCenter = new Vector2(
                (float)(screenManager.GraphicsDevice.Viewport.Width / 2),
                (float)(screenManager.GraphicsDevice.Viewport.Height / 2));

            // Aspect ratio of screen
            float aspectRatio = 
                (float)screenManager.GraphicsDevice.Viewport.Width /
                (float)screenManager.GraphicsDevice.Viewport.Height;

            // Set projection matrix for view
            this.projectionMatrix = Matrix.CreatePerspectiveFieldOfView(
                0.1f * MathHelper.PiOver4, 
                aspectRatio, 
                5.0f, 
                10000.0f);

            // Create a view matrix for the camera
            this.displaySize = 1.0f;
            UpdateViewMatrix();

            // Reset values
            this.latitude = 0.0;
            this.longitude = 0.0;

            // Create indicators and set initial value which will autoset ranges
            this.latitudeIndicator = new ValueIndicator(
                this.screenManager, "Latitude", "{0,20:####.##} deg", -90.0, 90.0);
            this.latitudeIndicator.SetPosition(new Vector2(20.0f, 20.0f));
            this.latitudeIndicator.SetValueInRange(this.latitude);

            this.longitudeIndicator = new ValueIndicator(
                this.screenManager, "Longitude", "{0,20:####.##} deg", 0.0, 360.0);
            this.longitudeIndicator.SetPosition(new Vector2(20.0f, 80.0f));
            this.longitudeIndicator.LowHighColoring = false;
            this.longitudeIndicator.SetValueInRange(this.longitude);

            this.gravityIndicator = new ValueIndicator(
                this.screenManager, "Gravity", "{0,20:###.######} N");
            this.gravityIndicator.SetPosition(new Vector2((float)screenManager.GraphicsDevice.Viewport.Width - 20.0f - ValueIndicator.Width, 20.0f));
            this.latitudeIndicator.LowHighColoring = false;
            this.gravityIndicator.SetValue(this.gravity);

            // Load fonts
            this.screenManager.AddFont("retro", "Fonts/retroMedium");

            // Create scrollers
            SpriteFont font = this.screenManager.Fonts["game"];
            int width = this.screenManager.GraphicsDevice.Viewport.Width;
            int yPos = this.screenManager.GraphicsDevice.Viewport.Height - 32;
            string infoText =
                "Model Info: " +
                gravityCalculator.GetModelInfo() +
                " --- Texture Info: " +
                celestialBody.GetTextureInfo() +
                "  ";
            infoText = infoText.Replace("\n", "  *  ");
            scrollingInfoText = new ScrollingText(infoText, font, width, yPos);
            scrollingInfoText.TextScale = 0.5f;

            // Set draw color
            this.drawColor = new Color(1.0f, 0.25f, 0.25f);
        }

        /// <summary>
        /// Gets or sets the currently shown latitude.
        /// Maintains value range of [-90,90].
        /// </summary>
        public double Latitude
        {
            get 
            { 
                return this.latitude; 
            }

            set 
            { 
                this.latitude = value; 

                // Fit into range
                while (this.latitude < -90.0)
                {
                    this.latitude = -90.0;
                }

                while (this.latitude > 90.0)
                {
                    this.latitude = 90.0;
                }

                // Set latitude and recalculate gravity
                this.gravityCalculator.Latitude = this.latitude;
                this.gravity = this.gravityCalculator.Value;

                // Update indicators
                this.latitudeIndicator.SetValueInRange(this.latitude);
                this.gravityIndicator.SetValue(this.gravity);
            }
        }

        /// <summary>
        /// Gets or sets the currently shown longitude.
        /// Maintains value range of [0,360].
        /// </summary>
        public double Longitude
        {
            get
            {
                return this.longitude;
            }

            set
            {
                this.longitude = value;

                // Fit into range
                while (this.longitude < 0.0)
                {
                    this.longitude += 360.0;
                }

                while (this.longitude > 360.0)
                {
                    this.longitude -= 360.0;
                }

                // Set longitude to recalculate gravity
                this.gravityCalculator.Longitude = this.longitude;
                this.gravity = this.gravityCalculator.Value;

                // Update indicators
                this.longitudeIndicator.SetValueInRange(this.longitude);
                this.gravityIndicator.SetValue(this.gravity);
            }
        }

        /// <summary>
        /// Update the game state.
        /// </summary>
        public void Update(GameTime gameTime)
        {
            scrollingInfoText.Update(gameTime);
        }

        /// <summary>
        /// Draw the animated gravity selector (sphere, value indicators, cross-hair).
        /// </summary>
        /// <param name="gameTime">Current game time.</param>
        public void Draw(GameTime gameTime)
        {
            // Earth is rotated by longitude and latitude
            Matrix sphereRollingMatrix = Matrix.CreateRotationY((float)(this.longitude * Math.PI / 180.0))
                                       * Matrix.CreateRotationX((float)(this.latitude * Math.PI / 180.0));
            planetSphere.Render(
                this.screenManager.GraphicsDevice,
                this.viewMatrix, 
                this.projectionMatrix, 
                sphereRollingMatrix,
                this.DisplaySize);

            // Overlay cross-hair
            PrimitiveBatch primitiveBatch = screenManager.PrimitiveBatch;
            float offset = (float)Math.Abs(Math.Sin(Math.PI * gameTime.TotalRealTime.TotalSeconds));
            offset = 4.0f * MathHelper.SmoothStep(0.0f, 1.0f, offset);
            float max_size = 12.0f + offset;
            float min_size =  1.0f + offset;

            // Lines
            primitiveBatch.Begin(PrimitiveType.LineList);

            primitiveBatch.AddVertex(screenCenter + new Vector2( max_size, 0.0f), this.drawColor);
            primitiveBatch.AddVertex(screenCenter + new Vector2( min_size, 0.0f), this.drawColor);

            primitiveBatch.AddVertex(screenCenter + new Vector2(-max_size, 0.0f), this.drawColor);
            primitiveBatch.AddVertex(screenCenter + new Vector2(-min_size, 0.0f), this.drawColor);

            primitiveBatch.AddVertex(screenCenter + new Vector2(0.0f,  max_size), this.drawColor);
            primitiveBatch.AddVertex(screenCenter + new Vector2(0.0f,  min_size), this.drawColor);

            primitiveBatch.AddVertex(screenCenter + new Vector2(0.0f, -max_size), this.drawColor);
            primitiveBatch.AddVertex(screenCenter + new Vector2(0.0f, -min_size), this.drawColor);

            primitiveBatch.End();

            // Draw indicators and scroller
            if (this.displaySize > 0.99f)
            {
                longitudeIndicator.Draw(gameTime);
                latitudeIndicator.Draw(gameTime);
                gravityIndicator.Draw(gameTime);

                SpriteBatch spriteBatch = this.screenManager.SpriteBatch;
                scrollingInfoText.Draw(gameTime, spriteBatch);
            }
        }

        /// <summary>
        /// Calculate new view matrix based on the current displaySize.
        /// </summary>
        private void UpdateViewMatrix()
        {
            // Camera distance
            double cameraDistance = 30.0f;
            if (this.displaySize <= 1.0f)
            {
                cameraDistance = cameraDistance + MathHelper.SmoothStep(0.0f, 1.0f, (1.0f - this.displaySize)) * 100.0f * cameraDistance;
            }
            else
            {
                cameraDistance = cameraDistance - MathHelper.SmoothStep(0.0f, 1.0f, this.displaySize - 1.0f) * cameraDistance;
            }

            // Right handed system, +ve z goes into screen.  
            // This places camera behind screen
            Vector3 cameraPosition = new Vector3(0, 0, (float)cameraDistance);

            // Copy the camera's reference vector.
            Vector3 cameraLookAtVector = cameraReferenceVector;

            // Calculate the position the camera is looking at.
            cameraLookAtVector += cameraPosition;

            // Create a view matrix for the camera
            // The third parameter is a vector which points up 
            // - this indicates the direction that "up" is in
            this.viewMatrix = Matrix.CreateLookAt(
                cameraPosition,
                cameraLookAtVector,
                cameraUpVector);
        }
    }
}
