﻿//-----------------------------------------------------------------------------
// LaplaceOverlay.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.GraphicalElements
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;

    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Utilities;
    using NewGamePhysics.Mathematics;

    /// <summary>
    /// 
    /// </summary>
    public class LaplaceOverlay : GraphicalElementBase
    {

        /// <summary>
        /// The effect to extract bright parts.
        /// </summary>
        private Effect extractEffect;

        /// <summary>
        /// The effect to apply laplace relaxation.
        /// </summary>
        private Effect laplaceEffect;

        /// <summary>
        /// The effect to combine the new and original image.
        /// </summary>
        private Effect combineEffect;

        /// <summary>
        /// The backbuffer.
        /// </summary>
        private ResolveTexture2D resolveTarget;

        /// <summary>
        /// Work texture 1
        /// </summary>
        private RenderTarget2D renderTarget1;

        /// <summary>
        /// Work texture 2
        /// </summary>
        private RenderTarget2D renderTarget2;
        
        /// <summary>
        /// Filter parameter to set tap offsets.
        /// </summary>
        EffectParameter offsetsParameter;

        /// <summary>
        /// Precalculated  tap offsets.
        /// </summary>
        Vector2[] tapOffsets;

        /// <summary>
        /// Creates a bloom overay.
        /// </summary>
        /// <param name="screenManager">The screen manager to use.</param>
        public LaplaceOverlay(
            ScreenManager screenManager)
            : base(screenManager)
        {
        }

        /// <summary>
        /// Load effects, create textures and prepare filters for filtering 
        /// from a specified source area on the screen.
        /// </summary>
        public void LoadContent()
        {
            this.extractEffect =
                this.ScreenManager.Game.Content.Load<Effect>("Effects/GreyscaleCutoff");
            this.laplaceEffect =
                this.ScreenManager.Game.Content.Load<Effect>("Effects/Laplace");
            this.combineEffect =
                this.ScreenManager.Game.Content.Load<Effect>("Effects/CombineContour");

            // Look up the resolution and format of our main backbuffer.
            PresentationParameters presentationParams =
                this.ScreenManager.GraphicsDevice.PresentationParameters;
            int width = presentationParams.BackBufferWidth;
            int height = presentationParams.BackBufferHeight;
            SurfaceFormat format = presentationParams.BackBufferFormat;

            // Create a texture for reading back the backbuffer contents.
            this.resolveTarget = new ResolveTexture2D(
                this.ScreenManager.GraphicsDevice,
                width,
                height,
                1,
                format);

            // Create 4th-size render targets for relaxation
            width /= 8;
            height /= 8;
            renderTarget1 = new RenderTarget2D(
                this.ScreenManager.GraphicsDevice,
                width,
                height,
                1,
                format,
                this.ScreenManager.GraphicsDevice.DepthStencilBuffer.MultiSampleType,
                this.ScreenManager.GraphicsDevice.PresentationParameters.MultiSampleQuality);
            renderTarget2 = new RenderTarget2D(
                this.ScreenManager.GraphicsDevice,
                width,
                height,
                1,
                format,
                this.ScreenManager.GraphicsDevice.DepthStencilBuffer.MultiSampleType,
                this.ScreenManager.GraphicsDevice.PresentationParameters.MultiSampleQuality);

            this.offsetsParameter = this.laplaceEffect.Parameters["TapOffsets"];

            // Precalculate offsets
            this.tapOffsets = new Vector2[4];
            float dx = 1.0f / width;
            float dy = 1.0f / height;
            this.tapOffsets[0] = new Vector2(0.0f,  dy);
            this.tapOffsets[1] = new Vector2(0.0f, -dy);
            this.tapOffsets[2] = new Vector2( dx, 0.0f);
            this.tapOffsets[3] = new Vector2(-dx, 0.0f);
        }

        /// <summary>
        /// Unload textures.
        /// </summary>
        public void UnloadContent()
        {
            resolveTarget.Dispose();
            renderTarget1.Dispose();
            renderTarget2.Dispose();
        }

        /// <summary>
        /// This is where it all happens. Grabs a scene that has already been rendered,
        /// and uses postprocess magic to add a glowing bloom effect over the top of it.
        /// </summary>
        /// <param name="gameTime">The current game time.</param>
        public void Draw(GameTime gameTime)
        {
            // Resolve the scene into a texture, so we can
            // use it as input data for laplace processing.
            this.ScreenManager.GraphicsDevice.ResolveBackBuffer(resolveTarget);

            // Pass 1: draw the scene into rendertarget 1, using a
            // shader that binarizes the image.
            this.extractEffect.Parameters["threshold"].SetValue(0.25f);
            DrawQuadWithEffect(resolveTarget, renderTarget1, this.extractEffect);

            // Repeat Pass 2 several times
            for (int i = 0; i < 12; i++)
            {
                // Pass 2a: draw from rendertarget 1 into rendertarget 2,
                // using a shader to apply laplacian relaxation.
                // Look up the sample weight and offset effect parameters.
                this.offsetsParameter.SetValue(tapOffsets);
                DrawQuadWithEffect(renderTarget1.GetTexture(), renderTarget2, this.laplaceEffect);

                // Pass 2b: draw from rendertarget 2 back into rendertarget 1,
                // using a shader to apply laplacian relaxation.
                this.offsetsParameter.SetValue(tapOffsets);
                DrawQuadWithEffect(renderTarget2.GetTexture(), renderTarget1, this.laplaceEffect);
            }

            // Pass 4: draw both rendertarget 1 and the original scene
            // image back into the main backbuffer, using a shader that
            // combines them to produce the final bloomed result
            this.ScreenManager.GraphicsDevice.Textures[1] = resolveTarget;
            this.combineEffect.Parameters["ContourFrequency"].SetValue(128.0f);
            Viewport viewport = this.ScreenManager.GraphicsDevice.Viewport;
            this.ScreenManager.SpriteBatch.Begin(
                SpriteBlendMode.None,
                SpriteSortMode.Immediate,
                SaveStateMode.None);
            this.combineEffect.Begin();
            this.combineEffect.CurrentTechnique.Passes[0].Begin();
            this.ScreenManager.SpriteBatch.Draw(
                renderTarget1.GetTexture(),
                new Rectangle(0, 0, viewport.Width, viewport.Height),
                Color.White);
            this.ScreenManager.SpriteBatch.End();
            this.combineEffect.CurrentTechnique.Passes[0].End();
            this.combineEffect.End();
        }

        /// <summary>
        /// Draw a texture full-screen into a rendertarget using a custom shader.
        /// </summary>
        /// <param name="texture">The texture to draw.</param>
        /// <param name="renderTarget">The render target.</param>
        /// <param name="effect">The effect to apply.</param>
        private void DrawQuadWithEffect(
            Texture2D texture,
            RenderTarget2D renderTarget,
            Effect effect)
        {
            this.ScreenManager.GraphicsDevice.SetRenderTarget(0, renderTarget);
            this.ScreenManager.SpriteBatch.Begin(
                SpriteBlendMode.None,
                SpriteSortMode.Immediate,
                SaveStateMode.None);
            effect.Begin();
            effect.CurrentTechnique.Passes[0].Begin();
            this.ScreenManager.SpriteBatch.Draw(
                texture,
                new Rectangle(0, 0, renderTarget.Width, renderTarget.Height),
                Color.White);
            this.ScreenManager.SpriteBatch.End();
            effect.CurrentTechnique.Passes[0].End();
            effect.End();
            this.ScreenManager.GraphicsDevice.SetRenderTarget(0, null);
        }
    }
}