

namespace NewGamePhysics.GraphicalElements
{
    using System;
    using System.Collections.Generic;
    using System.Text;

    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Graphics;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Utilities;

    /// <summary>
    /// Rotational action indicator to show frictional 
    /// forces on an axle or hinge.
    /// </summary>
    public class RotationalActionIndicator
    {
        /// <summary>
        /// The screen manager for drawing.
        /// </summary>
        private ScreenManager screenManager;

        /// <summary>
        /// Flag indicating of the action indicator is visible.
        /// </summary>
        private bool visible = false;

        /// <summary>
        /// Speed of rotation.
        /// </summary>
        private const double actionRotationSpeed = 4.0;

        /// <summary>
        /// Current rotational state of the action indicator.
        /// </summary>
        private float actionIndicatorRotation;

        /// <summary>
        /// Center of circular indicator.
        /// </summary>
        private Vector2 center;

        /// <summary>
        /// Scale of indicator. (Size of circle.)
        /// </summary>
        private float scale;

        /// <summary>
        /// The drawing color. Changes depending on intensity.
        /// </summary>
        private Color drawColor = Color.Red;

        /// <summary>
        /// Default constructor.
        /// </summary>
        public RotationalActionIndicator(ScreenManager screenManager)
        {
            this.screenManager = screenManager;
            this.actionIndicatorRotation = 0.0f;
            this.center = new Vector2();
            this.scale = 20.0f;
        }

        /// <summary>
        /// Set the 2D position of the indicator.
        /// </summary>
        /// <param name="position">The display position.</param>
        public void SetPosition(Vector2 position)
        {
            this.center = position;
        }

        /// <summary>
        /// Animate rotation display.
        /// </summary>
        /// <param name="intensity">The amount of the animation.</param>
        public void Animate(double intensity)
        {
            if (intensity == 0)
            {
                this.visible = false;
                return;
            }

            this.visible = true;
            this.actionIndicatorRotation += (float)(actionRotationSpeed * intensity);
            this.actionIndicatorRotation = MathHelper.WrapAngle(this.actionIndicatorRotation);

            if (intensity < 0.0)
            {
                this.drawColor = Color.Red;
            }
            else
            {
                this.drawColor = Color.Green;
            }
        }

        /// <summary>
        /// Draws a red circle.
        /// </summary>
        /// <param name="gameTime">The game time.</param>
        public void Draw(GameTime gameTime)
        {
            if (visible)
            {
                PrimitiveBatch primitiveBatch = screenManager.PrimitiveBatch;

                const int segments = 16;
                const double angle = MathHelper.TwoPi / segments;

                double curAngle = this.actionIndicatorRotation;
                primitiveBatch.Begin(PrimitiveType.LineList);
                for (int i = 0; i < segments; i++)
                {
                    Vector2 point = this.center + new Vector2(
                        this.scale * (float)Math.Round(Math.Sin(curAngle), 4),
                        this.scale * (float)Math.Round(Math.Cos(curAngle), 4));
                    primitiveBatch.AddVertex(point, this.drawColor);
                    curAngle += angle;
                }

                primitiveBatch.End();
            }
        }
    }
}
