﻿
namespace NewGamePhysics.GraphicalElements
{
    using System;
    using System.Text;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Graphics;

    /// <summary>
    /// Implements a left-to-right scrolling text scroller which 
    /// displays as band across the whole width of the screen.
    /// </summary>
    public class ScrollingText
    {
        /// <summary>
        /// Text scroller offset.
        /// X is changed, Y remains fixed.
        /// </summary>
        private Vector2 scrollerOffset;

        /// <summary>
        /// The speed with which the scroller moves left.
        /// </summary>
        private float scrollerSpeed;

        /// <summary>
        /// The text to scroll.
        /// </summary>
        private string text;

        /// <summary>
        /// The width of the screen.
        /// </summary>
        private int screenWidth;

        /// <summary>
        /// The font to use for drawing the text.
        /// </summary>
        private SpriteFont textFont;

        /// <summary>
        /// The size of the text.
        /// </summary>
        private Vector2 textSize;

        /// <summary>
        /// The color of the text.
        /// Defaults to white.
        /// </summary>
        private Color textColor;

        /// <summary>
        /// Relative size of the text. 
        /// Defaults to 1.0
        /// </summary>
        private float textScale;

        /// <summary>
        /// Anchor position for the text drawing.
        /// Defaults to 0,0
        /// </summary>
        private static Vector2 textOrigin = new Vector2(0, 0);

        /// <summary>
        /// 
        /// </summary>
        /// <param name="text">The text to scroll.</param>
        /// <param name="font">The font to use for the text.</param>
        /// <param name="screenWidth">The width of the screen.</param>
        /// <param name="yPos">The vertical position for the scroller.</param>
        public ScrollingText(string text, SpriteFont font, int screenWidth, int yPos)
        {

            this.text = text;
            this.textFont = font;
            this.textSize = this.textFont.MeasureString(text);

            this.textColor = Color.White;
            this.textScale = 1.0f;

            // Set x from screen, y from position
            this.screenWidth = screenWidth;
            this.scrollerOffset = new Vector2(this.screenWidth, yPos);
            this.scrollerSpeed = 100.0f;
        }

        /// <summary>
        /// Gets or sets the current text of the scroller.
        /// If text is set, the scroller offset resets.
        /// </summary>
        public string Text
        {
            get { return this.text; }
            set { 
                this.text = value;
                this.textSize = this.textFont.MeasureString(value);
                this.scrollerOffset = new Vector2(this.screenWidth, this.scrollerOffset.Y);
            }
        }

        /// <summary>
        /// Gets or sets the current scroller speed in pixel per second.
        /// Default is 100 pixel/sec.
        /// </summary>
        public float ScrollerSpeed
        {
            get { return this.scrollerSpeed; }
            set { this.scrollerSpeed = value; }
        }

        /// <summary>
        /// Gets or sets the current text color.
        /// Default is White.
        /// </summary>
        public Color TextColor
        {
            get { return this.textColor; }
            set { this.textColor = value; }
        }

        /// <summary>
        /// Gets or sets the current text scale.
        /// Default is 1.0.
        /// </summary>
        public float TextScale
        {
            get { return this.textScale; }
            set { this.textScale = value; }
        }

        /// <summary>
        /// Update the scroller state.
        /// </summary>
        public void Update(GameTime gameTime)
        {
            // Move scroller
            scrollerOffset.X -= scrollerSpeed * (float)gameTime.ElapsedGameTime.Milliseconds * 0.001f;

            // Wrap motion
            if (scrollerSpeed > 0.0f)
            {
                if ((this.scrollerOffset.X + this.textSize.X) < 0)
                {
                    scrollerOffset.X = this.screenWidth;
                }
            } 
            else
            {
                if (this.scrollerOffset.X > this.screenWidth)
                {
                    scrollerOffset.X = -this.textSize.X;
                }
            }
        }

        /// <summary>
        /// Draw the scroller.
        /// </summary>
        /// <param name="gameTime">Current game time.</param>
        /// <param name="spriteBatch">The sprite batch to use for drawing.</param>
        public void Draw(GameTime gameTime, SpriteBatch spriteBatch )
        {
            spriteBatch.Begin();
            spriteBatch.DrawString(
                this.textFont, 
                this.text, 
                this.scrollerOffset, 
                this.textColor, 
                0,
                textOrigin, 
                this.textScale, 
                SpriteEffects.None, 
                0);
            spriteBatch.End();
        }
    }
}
