using System;
using System.Collections.Generic;
using System.Text;

using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;

using NewGamePhysics.StateManager;
using NewGamePhysics.Utilities;
using NewGamePhysics.GraphicalElements;

namespace NewGamePhysics.GraphicalElements
{
    /// <summary>
    /// A value indicator box for a floating point value.
    /// Displays a formatted value, optionally a bar graph in a
    /// minimum/maximum range, and optionally a color change
    /// of the bar graph for low/high setting.
    /// </summary>
    public class ValueIndicator : GraphicalElementBase
    {
        /// <summary>
        /// The current value of the indicator.
        /// </summary>
        private double value;

        /// <summary>
        /// The value conversion format string including the unit.
        /// </summary>
        private string format;

        /// <summary>
        /// The value as a string. Updated only once per
        /// update interval.
        /// </summary>
        private string valueAsString = String.Empty;

        /// <summary>
        /// The last game time when the value was updated.
        /// </summary>
        private DateTime valueUpdate;

        /// <summary>
        /// The minimum value of the bar indicator.
        /// </summary>
        private double minimumValue;

        /// <summary>
        /// The maximum value of the bar indicator.
        /// </summary>
        private double maximumValue;

        /// <summary>
        /// Flag indicating if there is a zero in the 
        /// min/max range.
        /// </summary>
        private bool haveZero;

        /// <summary>
        /// The low value of the indicator when a 
        /// color change occurs. Must be greater than
        /// the minimum.
        /// </summary>
        private double lowValue;

        /// <summary>
        /// The hight value of the indicator when a
        /// color change occirs. Must be less than the 
        /// maximum.
        /// </summary>
        private double highValue;

        /// <summary>
        /// The position where the indicator is drawn on the screen.
        /// </summary>
        private Vector2 position;

        /// <summary>
        /// The width of the indicator in pixels.
        /// </summary>
        public const float Width = 200.0f;

        /// <summary>
        /// The height of the indicator in pixels.
        /// </summary>
        public const float Height = 40.0f;

        /// <summary>
        /// The name of the indicator.
        /// </summary>
        private string label;

        /// <summary>
        /// The color of the indicator box and the font.
        /// </summary>
        private Color drawColor;

        /// <summary>
        /// Flag indicating if low/high coloring is done.
        /// </summary>
        private bool lowHighColoring;

        /// <summary>
        /// The current color of the indicator bar graph.
        /// Updated when value is set.
        /// </summary>
        private Color barColor;

        /// <summary>
        /// The possible color of the indicator bar graph.
        /// [0] = red (=min or =max)
        /// [1] = yellow (=[min-low] or =[high-max])
        /// [2] = green (=[low-high])
        /// </summary>
        private Color[] barColors;

        /// <summary>
        /// Minimum time in milliseconds between value updates.
        /// Default: 250 (quarter second)
        /// </summary>
        private double updateInterval = 250.0;

        /// <summary>
        /// Black texture for drawing the backdrop of the indicator.
        /// </summary>
        private Texture2D backgroundTexture;

        /// <summary>
        /// White texture for drawing the bar graph of the indicator.
        /// </summary>
        private Texture2D whiteTexture;

        /// <summary>
        /// The font of the label.
        /// </summary>
        private SpriteFont spriteFont;

        /// <summary>
        /// Create new indicator object for use with autosizing.
        /// </summary>
        /// <param name="manager">The screen manager to use for drawing the indicator.</param>
        /// <param name="label">The label of the indicator.</param>
        /// <param name="format">The numeric format string to use to convert the number for display.</param>
        public ValueIndicator(
            ScreenManager manager,
            string label,
            string format)
            : this(manager, label, format, 0.0, 0.0)
        {
        }

        /// <summary>
        /// Create new indicator object with fixed value range.
        /// </summary>
        /// <param name="manager">The screen manager to use for display.</param>
        /// <param name="label">The label of the indicator.</param>
        /// <param name="format">The display to use for the value.</param>
        /// <param name="minimum">The minimum value of the indicator.</param>
        /// <param name="maximum">The maximum value of the indicator.</param>
        public ValueIndicator(
            ScreenManager manager, 
            string label, 
            string format,
            double minimum,
            double maximum) : base(manager)
        {
            // Store settings
            this.label = label;
            this.format = format;

            // Reset ranges
            this.minimumValue = minimum;
            this.maximumValue = maximum;
            double delta = 0.1 * (maximum - minimum);
            this.lowValue = minimum + delta;
            this.highValue = maximum - delta;

            // Update flag
            this.haveZero = ((this.minimumValue < 0.0) && (this.maximumValue > 0.0));

            // Default position and size
            this.position = new Vector2();

            // Color for numbers
            this.drawColor = new Color(180, 255, 180);

            // Bar indicator colors
            this.barColors = new Color[3];
            // red
            this.barColors[0] = new Color(255, 0, 0);
            // yellow
            this.barColors[1] = new Color(255, 255, 0);
            // green
            this.barColors[2] = new Color(0, 255, 0);

            // Guage backdrop texture
            this.backgroundTexture = ScreenManager.Game.Content.Load<Texture2D>(@"Sprites\value_gauge");

            // White indicator texture
            this.whiteTexture = TextureHelpers.Create(ScreenManager.GraphicsDevice, Color.White);

            // Now set a default value, set bar color.
            this.SetValue(0.0);

            // Label font
            this.spriteFont = ScreenManager.Fonts["small"];
        }

        #region properties

        /// <summary>
        /// Gets or sets the minimum value for the bar indicator.
        /// </summary>
        public double MinimumValue
        {
            get
            {
                return this.minimumValue;
            }

            set
            {
                this.minimumValue = value;

                // Maybe also adjust low value
                if (this.lowValue < value)
                {
                    this.lowValue = value;
                }

                // Update flag
                this.haveZero = ((this.minimumValue < 0.0) && (this.maximumValue > 0.0));
            }
        }

        /// <summary>
        /// Gets or sets the maximum value for the bar indicator.
        /// </summary>
        public double MaximumValue
        {
            get
            {
                return this.maximumValue;
            }

            set
            {
                this.maximumValue = value;

                // Maybe also adjust high value
                if (this.highValue > value)
                {
                    this.highValue = value;
                }

                // Update flag
                this.haveZero = ((this.minimumValue < 0.0) && (this.maximumValue > 0.0));
            }
        }

        /// <summary>
        /// Gets or sets the low value for the bar indicator
        /// when the color changes.
        /// </summary>
        public double LowValue
        {
            get { return this.lowValue; }

            set
            {
                if (value < this.minimumValue)
                {
                    throw new ArgumentOutOfRangeException("value");
                }

                this.lowValue = value;
            }
        }

        /// <summary>
        /// Gets or sets the high value for the bar indicator
        /// when the color changes
        /// </summary>
        public double HighValue
        {
            get { return this.highValue; }

            set
            {
                if (value > this.maximumValue)
                {
                    throw new ArgumentOutOfRangeException("value");
                }

                this.highValue = value;
            }
        }

        /// <summary>
        /// Gets or sets flag indicating of low/high coloring is done.
        /// </summary>
        public bool LowHighColoring
        {
            get { return this.lowHighColoring; }
            set { this.lowHighColoring = value; }
        }

        /// <summary>
        /// Gets or sets the minimum time in milliseconds between value updates.
        /// Default: 250 (quarter second).
        /// </summary>
        public double UpdateInterval
        {
            get { return this.updateInterval; }
            set { this.updateInterval = value; }
        }

        #endregion

        /// <summary>
        /// Sets a new value and autoupdates min/max + 
        /// low/high values.
        /// </summary>
        /// <param name="value">The value to set.</param>
        public void SetValue(double value)
        {
            // Recalculate min and max
            this.MinimumValue = Math.Min(this.minimumValue, Autorange.GetSnapValue(value, false));
            this.MaximumValue = Math.Max(this.maximumValue, Autorange.GetSnapValue(value, true));

            // Use 10% for low/high ranges
            double delta = 0.1 * (this.maximumValue - this.minimumValue);
            this.LowValue = this.minimumValue + delta;
            this.HighValue = this.maximumValue - delta;

            // Set value
            this.value = value;
            UpdateBarColor();
        }

        /// <summary>
        /// Sets a new value within the currently defined 
        /// min/max range. The value is clipped to the 
        /// current min or max if out of bounds.
        /// </summary>
        /// <param name="value">The value to set.</param>
        public void SetValueInRange(double value)
        {
            // Set value, but keep it in range
            if (value < minimumValue)
            {
                this.value = this.minimumValue;
            }
            else if (value > maximumValue)
            {
                this.value = this.maximumValue;
            }
            else
            {
                this.value = value;
            }

            UpdateBarColor();
        }

        /// <summary>
        /// Sets the minimum.
        /// </summary>
        /// <param name="value">New minimum.</param>
        public void SetMinimum(double value)
        {
            this.minimumValue = value;
        }

        /// <summary>
        /// Sets the maximum.
        /// </summary>
        /// <param name="value">New maximum.</param>
        public void SetMaximum(double value)
        {
            this.maximumValue = value;
        }

        /// <summary>
        /// Reposition indicator object.
        /// </summary>
        /// <param name="position"></param>
        public void SetPosition(Vector2 position)
        {
            this.position.X = position.X;
            this.position.Y = position.Y;
        }

        /// <summary>
        /// Draws the indicator.
        /// </summary>
        /// <param name="primitiveBatch">The primitive batch to use for drawing the lines.</param>
        /// <param name="dotColor">The color for the dots.</param>
        public void Draw(GameTime gameTime)
        {
            float outerHeight = Height;
            float innerHeightStep = 12.0f;
            float innerHeight = 3.0f * innerHeightStep;

            float outerWidth = Width;
            float innerWidth = Width - 4.0f;
            float rowOffsetX = 2.0f;
            float row1OffsetY = 2.0f;
            float row2OffsetY = 2.0f + innerHeightStep;
            float row3OffsetY = 2.0f + 2.0f*innerHeightStep;
            float rowHeight = 10.0f;
            float valueOffset = innerWidth * (float)((this.value - this.minimumValue) / (this.maximumValue - this.minimumValue));
            float zeroOffset = innerWidth * (float)((0.0 - this.minimumValue) / (this.maximumValue - this.minimumValue));

            Rectangle destination;

            // Prepare value string
            DateTime currentDate = DateTime.Now;
            TimeSpan elapsedSpan = new TimeSpan(currentDate.Ticks - valueUpdate.Ticks);
            if (elapsedSpan.TotalMilliseconds > this.updateInterval)
            {
                valueAsString = String.Format(
                    this.format,
                    this.value);
                valueUpdate = currentDate;
            }

            // Backdrop of gauge
            destination = new Rectangle((int)this.position.X, (int)this.position.Y, (int)outerWidth, (int)outerHeight);
            SpriteBatch.Begin();
            SpriteBatch.Draw(this.backgroundTexture, destination, Color.White);
            SpriteBatch.End();            

            // Row 1: value + unit
            Vector2 textSize = spriteFont.MeasureString(valueAsString);
            SpriteBatch.Begin();
            SpriteBatch.DrawString(
                spriteFont, 
                valueAsString,
                new Vector2(this.position.X + rowOffsetX, this.position.Y + row1OffsetY),
                this.drawColor, 0, new Vector2(), 1.0f,
                SpriteEffects.None, 0);
            SpriteBatch.End();

            // Row 2: Bar indicator
            if ((this.haveZero) && (valueOffset < zeroOffset))
            {
                destination = new Rectangle((int)(this.position.X + rowOffsetX + valueOffset), (int)(this.position.Y + row2OffsetY) + 1, (int)(zeroOffset - valueOffset), (int)rowHeight - 1);
            }
            else
            {
                destination = new Rectangle((int)(this.position.X + rowOffsetX + zeroOffset),  (int)(this.position.Y + row2OffsetY) + 1, (int)(valueOffset - zeroOffset), (int)rowHeight - 1);
            }
            SpriteBatch.Begin();
            SpriteBatch.Draw(this.whiteTexture, destination, this.barColor);
            SpriteBatch.End();

            // Lines
            PrimitiveBatch.Begin(PrimitiveType.LineList);
            Vector2 offset = new Vector2(rowOffsetX, row2OffsetY);

            // Zero
            if (this.haveZero)
            {
                Color lineColor = new Color(this.drawColor, 128);
                PrimitiveBatch.AddVertex(this.position + offset + new Vector2(zeroOffset, 0.0f), lineColor);
                PrimitiveBatch.AddVertex(this.position + offset + new Vector2(zeroOffset, rowHeight), lineColor);
            }

            // Value
            PrimitiveBatch.AddVertex(this.position + offset + new Vector2(valueOffset, 0.0f), this.drawColor);
            PrimitiveBatch.AddVertex(this.position + offset + new Vector2(valueOffset, rowHeight), this.drawColor);

            PrimitiveBatch.End();

            // Label
            if (!String.IsNullOrEmpty(this.label))
            {
                textSize = spriteFont.MeasureString(this.label);
                float textOffset = (innerWidth - textSize.X) * 0.5f;
                SpriteBatch.Begin();
                SpriteBatch.DrawString(
                    spriteFont,
                    this.label,
                    new Vector2(this.position.X + rowOffsetX + textOffset, this.position.Y + row3OffsetY),
                    Color.Black, 0, new Vector2(), 1.0f,
                    SpriteEffects.None, 0);
                SpriteBatch.End();
            }
        }

        /// <summary>
        /// Determines the bar color from the value.
        /// </summary>
        private void UpdateBarColor()
        {
            if (this.lowHighColoring && 
                ((this.value == this.minimumValue) || (this.value == this.maximumValue)))
            {
                // red
                this.barColor = barColors[0];
            }
            else if (this.lowHighColoring && 
                ((this.value < this.lowValue) || (this.value > this.highValue)))
            {
                // yellow
                this.barColor = barColors[1];
            }
            else
            {
                // green
                this.barColor = barColors[2];
            }
        }
    }
}
