﻿//-----------------------------------------------------------------------------
// BitUnbiaser.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Mathematics
{
    using System;
    using System.Text;

    /// <summary>
    /// Enum describing different bit unbiasing algorithms.
    /// </summary>
    public enum BitUnbiasAlgorithm
    {
        /// <summary>
        /// Von Neuman algorithm to unbias a bitstream.
        /// (max bit yield 50%)
        /// </summary>
        VonNeuman,

        /// <summary>
        /// (Unadvanced) Multilevel Strategy
        /// (max bit yield 75%)
        /// </summary>
        MLS,

        /// <summary>
        /// Advanced Multilevel Strategy
        /// (max bit yield 100%)
        /// </summary>
        AMLS,
    }

    /// <summary>
    /// Class to facilitate unbiasing bitstreams represented by
    /// strings containing 0 and 1 characters.
    /// </summary>
    public class BitUnbiaser
    {
        /// <summary>
        /// Currently selected algorithm for the bit unbiaser.
        /// </summary>
        private BitUnbiasAlgorithm algorithm;

        /// <summary>
        /// Creates an unbiasing object.
        /// </summary>
        /// <param name="algorithm">The bit unbiasing algorithm.</param>
        public BitUnbiaser(BitUnbiasAlgorithm algorithm)
        {
            this.algorithm = algorithm;
        }

        /// <summary>
        /// Unbias an bit stream using the currently set algorithm.
        /// </summary>
        /// <param name="input">
        /// An input string consisting of 0 or 1 characters representing bits
        /// of minimum 2 bits in length.
        /// </param>
        /// <returns>
        /// The unbiased bit string in 0/1 character representation
        /// </returns>
        public string Process(string input)
        {
            if (string.IsNullOrEmpty(input))
            {
                throw new ArgumentNullException("input", "input string cannot be null or empty");
            }

            if (input.Length < 2)
            {
                throw new ArgumentException("input must consist of at least 2 characters", "input"); 
            }

            StringBuilder output = new StringBuilder(input.Length);
            Amls(input, ref output);
            return output.ToString();
        }

        /// <summary>
        /// Start recursive AMLS algorithm implementation.
        /// </summary>
        /// <param name="input">The biased input bitstream.</param>
        /// <param name="output">The unbiased output bitstream.</param>
        private void Amls(string input, ref StringBuilder output)
        {
            char[] inputC = input.ToCharArray();
            int start = 0;
            int end = input.Length - 1;
            AmlsStep(ref inputC, ref output, start, end);
        }

        /// <summary>
        /// Process input stream to yield output bits. Generates two additional
        /// biased streams.
        /// </summary>
        /// <param name="input">The input bitstream as character array.</param>
        /// <param name="output">The output string builder.</param>
        /// <param name="start">The current start index.</param>
        /// <param name="end">The current end index</param>
        private void AmlsStep(ref char[] inputC, ref StringBuilder output, int start, int end)
        {
            if (start >= end)
            {
                return;
            }

            int indexD = start;
            int indexL = start;
            int indexH = end;

            do
            {
                if (inputC[indexL] == inputC[indexH])
                {
                    inputC[indexD] = inputC[indexL];
                    indexD++;
                    inputC[indexH] = '0';
                }
                else
                {
                    output.Append(inputC[indexL]);
                    inputC[indexH] = '1';
                }

                indexH--;
                indexL++;
            } while (indexH > indexL);

            // Maybe terminate early 
            if (this.algorithm == BitUnbiasAlgorithm.VonNeuman)
            {
                return;
            }

            // Recurse for MLS
            indexD--;
            AmlsStep(ref inputC, ref output, start, indexD);

            // Maybe terminate early
            if (this.algorithm == BitUnbiasAlgorithm.MLS)
            {
                return;
            }

            // Recurse for AMLS
            indexH++;
            AmlsStep(ref inputC, ref output, indexH, end);
        }
    }
}
