﻿//-----------------------------------------------------------------------------
// EdgeDetector.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Mathematics
{
    using System;

    /// <summary>
    /// Edge detection types.
    /// </summary>
    public enum EdgeDetectionType
    {
        /// <summary>
        /// Detect rising edges
        /// </summary>
        Rising,

        /// <summary>
        /// Detect falling edges
        /// </summary>
        Falling,

        /// <summary>
        /// Detect rising and falling edges
        /// </summary>
        Both,
    }

    /// <summary>
    /// Represents an edge detector.
    /// </summary>
    public class EdgeDetector
    {
        /// <summary>
        /// The type of the detection.
        /// </summary>
        private EdgeDetectionType detectionType;

        /// <summary>
        /// The binarization threshold of the detector.
        /// </summary>
        private double threshold;

        /// <summary>
        /// Hide default constructor.
        /// </summary>
        private EdgeDetector()
        {
        }

        /// <summary>
        /// Create a new instance of an edge detector.
        /// </summary>
        /// <param name="detectionType">
        /// The detection type (edge(s) to detect).</param>
        /// <param name="threshold">
        /// The binarization threshold in the range 0.0 to 1.0 (against normalized input)
        /// </param>
        public EdgeDetector(EdgeDetectionType detectionType, double threshold)
        {
            this.detectionType = detectionType;
            this.threshold = threshold;
        }

        /// <summary>
        /// Analyzes samples and returns new array with samples set to 1.0 where
        /// edges were detected. The absolute value of the input is normalized
        /// and binarized at the threshold.
        /// </summary>
        /// <param name="samples">The samples to analyze.</param>
        /// <returns>
        /// The detection result containing 0.0 (no edge) and 1.0 (edge)
        /// values.
        /// </returns>
        public double[] Calculate(double[] samples)
        {
            int nMax = samples.Length;
            double[] result = new double[nMax];

            // Make absolute and determine maximum
            double maximum = Math.Abs(samples[0]);
            for (int i = 0; i < nMax; i++)
            {
                double value = Math.Abs(samples[i]);
                result[i] = value;
                if (value > maximum)
                {
                    maximum = value;
                }
            }

            // Binarize
            double margin = maximum * this.threshold;
            for (int i = 0; i < nMax; i++)
            {
                result[i] = (result[i] > margin) ? 1.0 : 0.0;
            }

            // Edge detect
            double lastValue = 0.0;
            for (int i = 0; i < nMax; i++)
            {
                bool risingEdge = ((lastValue == 0.0) && (result[i] == 1.0));
                bool fallingEdge = ((lastValue == 1.0) && (result[i] == 0.0));
                lastValue = result[i];
                switch (this.detectionType)
                {
                    case EdgeDetectionType.Falling:
                        result[i] = (fallingEdge) ? 1.0 : 0.0;
                        break;
                    case EdgeDetectionType.Rising:
                        result[i] = (risingEdge) ? 1.0 : 0.0;
                        break;
                    case EdgeDetectionType.Both:
                        result[i] = (risingEdge || fallingEdge) ? 1.0 : 0.0;
                        break;
                }
            }

            return result;
        }
    }
}
