﻿//-----------------------------------------------------------------------------
// EquirectangularProjection.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Mathematics
{
    using System;

    /// <summary>
    /// Calculate the equirectangular projection 
    /// (also called the equidistant cylindrical 
    /// projection, geographic projection, 
    /// plate carrée or carte parallelogrammatique 
    /// projection or CPP between
    /// lat,lon (degrees) to/from x,y ([0,1]).
    /// Reference: 
    /// http://en.wikipedia.org/wiki/Plate_carr%C3%A9e_projection
    /// </summary>
    public class EquirectangularProjection
    {
        /// <summary>
        /// Convert longitude [0,360] to [0,1] using 
        /// the mercator projection.
        /// </summary>
        /// <param name="lon">The longitude in degrees.</param>
        /// <returns>The map position.</returns>
        public static double lonToX(double lon)
        {
            if ((lon < 0) || (lon > 360.0))
            {
                throw new ArgumentOutOfRangeException("lon");
            }

            return lon / 360.0;
        }

        /// <summary>
        /// Convert latitude [-90.0,90] to a map position [0,1] using the
        /// equirectangilar projection.
        /// </summary>
        /// <param name="lat">The latitude in degrees.</param>
        /// <returns>The map position.</returns>
        public static double latToY(double lat)
        {
            if ((lat < -90.0) || (lat > 90.0))
            {
                throw new ArgumentOutOfRangeException("lat");
            }

            return (lat / 90.0 + 1.0) / 2.0;
        }

        /// <summary>
        /// Convert a map position [0,1] to longitude [0,360] using
        /// the equirectangular projection.
        /// </summary>
        /// <param name="x">The horizontal map position.</param>
        /// <returns>The longitude value.</returns>
        public static double xToLon(double x)
        {
            if ((x < 0.0) || (x > 1.0))
            {
                throw new ArgumentOutOfRangeException("lon");
            }

            return x * 360.0;
        }

        /// <summary>
        /// Convert [0,1] to latitude [-90,90] using the 
        /// equirectangular projection.
        /// </summary>
        /// <param name="y">The vertical map position.</param>
        /// <returns>The latitude in degrees.</returns>
        public static double yToLat(double y)
        {
            if ((y < 0.0) || (y > 1.0))
            {
                throw new ArgumentOutOfRangeException("lon");
            }

            return y * 180.0 - 90.0;
        }
    }
}
