﻿//-----------------------------------------------------------------------------
// Factorial.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Mathematics
{
    using System;

    /// <summary>
    /// Calculate factorials.
    /// </summary>
    public static class Factorial
    {
        /// <summary>
        /// Calculate the factorial of an integer x (x!).
        /// </summary>
        /// <param name="x">The number x to calculate the factorial on.</param>
        /// <returns>The factorial of x (x!)</returns>
        public static ulong Calc(uint x)
        {
            if (x > 21)
            {
                throw new ArgumentOutOfRangeException(
                    "x",
                    "value must be in the range [0,21]");
            }

            // Special case 0!=1
            if (x == 0)
            {
                return 1;
            }

            ulong f = x;
            for (ulong i = 2; i < x; i++) 
            {
                ulong ui = (ulong)i;
                f *= ui;
            }

            return f;
        }

        /// <summary>
        /// Calculate the double factorial of an integer x (x!!).
        /// </summary>
        /// <param name="x">The number x to calculate the double factorial on.</param>
        /// <returns>The double factorial of x (x!!)</returns>
        /// <remarks>
        /// Reference: http://mathworld.wolfram.com/DoubleFactorial.html
        /// </remarks>
        public static ulong CalcDouble(int x)
        {
            if ((x < -1) || (x > 32))
            {
                throw new ArgumentOutOfRangeException(
                    "x", 
                    "value must be in the range [-1,32]");
            }

            // Special cases -1!! = 0!! = 1
            if ((x == -1) || (x == 0))
            {
                return 1;
            }

            ulong f = 1;
            for (ulong i = (ulong)x; i >= 2; i -= 2)
            {
                f *= i;
            }

            return f;
        }
    }
}
