﻿//-----------------------------------------------------------------------------
// Gaussian.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Mathematics
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    
    /// <summary>
    /// Special functions: Gaussian function.
    /// (Probability density function of the normal distribution.)
    /// </summary>
    public static class Gaussian
    {
        /// <summary>
        /// The inverse square root of two PI (1/sqrt(2*PI)).
        /// </summary>
        private static double InverseRootTwoPi = 1.0 / Math.Sqrt(2.0 * Math.PI);

        /// <summary>
        /// Calculates the 1-dimensional gaussian around zero mean.
        /// </summary>
        /// <param name="deviation">The deviation ('width').</param>
        /// <param name="x">The value to calculate the gaussian for.</param>
        /// <returns>The gaussian function value.</returns>
        public static double DistributionValue(double deviation, double x)
        {
            return DistributionValue(0.0, deviation, x);
        }

        /// <summary>
        /// Calculates the 1-dimensional gaussian around a mean value.
        /// </summary>
        /// <param name="mean">The mean of the curve.</param>
        /// <param name="deviation">The deviation ('width').</param>
        /// <param name="x">The value to calculate the gaussian for.</param>
        /// <returns>The gaussian function value.</returns>
        public static double DistributionValue(double mean, double deviation, double x)
        {
            if (deviation <= 0.0)
            {
                throw new ArgumentException("deviation cannot be less or equals than zero", "deviation");
            }

            double scale = InverseRootTwoPi / deviation;
            double negativeInverseTwoSigmaSquared = -1.0 / (2.0* deviation * deviation);
            double shiftedInput = x - mean;
            double shiftedInputSquared = shiftedInput * shiftedInput;
            return scale * Math.Exp(negativeInverseTwoSigmaSquared * shiftedInputSquared);
        }
    }
}
