﻿//-----------------------------------------------------------------------------
// MercatorProjection.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Mathematics
{
    using System;

    /// <summary>
    /// Calculate the elliptical mercator projection
    /// lat,lon (degrees) to/from x,y ([0,1]).
    /// Reference:
    /// http://wiki.openstreetmap.org/index.php/Mercator
    /// </summary>
    class MercatorProjection
    {
        /// <summary>
        /// The major radius of the elliptical earth in m.
        /// </summary>
        private static readonly double MajorRadius = 6378137.0;

        /// <summary>
        /// The minor radius of the elliptical earth in m.
        /// </summary>
        private static readonly double MinorRadius = 6356752.3142;

        /// <summary>
        /// The radius-ratio of the elliptical earth.
        /// </summary>
        private static readonly double Ratio = MinorRadius / MajorRadius;

        /// <summary>
        /// The eccentricity of the elliptical earth.
        /// </summary>
        private static readonly double Eccentricity = Math.Sqrt(1.0 - (Ratio * Ratio));

        /// <summary>
        /// Degree to radians conversion factor.
        /// </summary>
        private static readonly double Deg2Rad = Math.PI / 180.0;

        /// <summary>
        /// Radians to degrees factor.
        /// </summary>
        private static readonly double Rad2Deg = 180.0 / Math.PI;

        /// <summary>
        /// Half of Pi.
        /// </summary>
        private static readonly double PiHalf = Math.PI / 2.0;

        /// <summary>
        /// Convert longitude [0,360] to [0,1] using 
        /// the mercator projection.
        /// </summary>
        /// <param name="lon">The longitude in degrees.</param>
        /// <returns>The map position.</returns>
        public static double lonToX(double lon)
        {
            if ((lon < 0) || (lon > 360.0))
            {
                throw new ArgumentOutOfRangeException("lon");
            }

            return lon / 360.0;
        }

        /// <summary>
        /// Convert latitude [-90.0,90] to a map position [0,1] using the
        /// mercator projection. For angles close to the poles
        /// (1 degree) an approximation is used.
        /// </summary>
        /// <param name="lat">The latitude in degrees.</param>
        /// <returns>The map position.</returns>
        public static double latToY(double lat)
        {
            if ((lat < -90.0) || (lat > 90.0))
            {
                throw new ArgumentOutOfRangeException("lat");
            }

            // Limit angles close to poles
            lat = Math.Min(89.0, Math.Max(lat, -89.0));

            // Mercator calculation
            double phi = DegToRad(lat);
            double sinphi = Math.Sin(phi);
            double con = Eccentricity * sinphi;
            con = Math.Pow(((1.0 - con) / (1.0 + con)), 0.5 * Eccentricity);
            double ts = Math.Tan(0.5 * ((Math.PI * 0.5) - phi)) / con;
            double y = -Math.Log(ts) * 0.5;
            y += 1.0;
            y *= 0.5;
            y = Math.Min(1.0, Math.Max(y, 0.0));
            return y;
        }

        /// <summary>
        /// Convert a map position [0,1] to longitude [0,360] using
        /// the mercator projection.
        /// </summary>
        /// <param name="x">The horizontal map position.</param>
        /// <returns>The longitude in degrees.</returns>
        public static double xToLon(double x)
        {
            if ((x < 0.0) || (x > 1.0))
            {
                throw new ArgumentOutOfRangeException("lon");
            }

            return x * 360.0;
        }

        /// <summary>
        /// Convert [0,1] to latitude [-90,90] using the 
        /// mercator projection.
        /// </summary>
        /// <param name="y">The vertical map position.</param>
        /// <returns>The latitude in degrees.</returns>
        public static double yToLat(double y)
        {
            if ((y < 0.0) || (y > 1.0))
            {
                throw new ArgumentOutOfRangeException("lon");
            }

            double ts = Math.Exp(-y);
            double phi = PiHalf - 2 * Math.Atan(ts);
            double dphi = 1.0;
            int i = 0;
            while ((Math.Abs(dphi) > 0.000000001) && (i < 15))
            {
                double c = Eccentricity * Math.Sin(phi);
                dphi = PiHalf - 2 * Math.Atan(ts * Math.Pow((1.0 - c) / (1.0 + c), 0.5 * Eccentricity)) - phi;
                phi += dphi;
                i++;
            }
            return RadToDeg(phi);
        }

        /// <summary>
        /// Conversion between radians and degrees.
        /// </summary>
        /// <param name="rad">The radians value</param>
        /// <returns>The degrees value.</returns>
        private static double RadToDeg(double rad)
        {
            return rad * Rad2Deg;
        }

        /// <summary>
        /// Conversion between degrees and radians.
        /// </summary>
        /// <param name="deg">The degrees value.</param>
        /// <returns>The radians valiue.</returns>
        private static double DegToRad(double deg)
        {
            return deg * Deg2Rad;
        }
    }
}
