//-----------------------------------------------------------------------------
// NystromIntegrator.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;

namespace NewGamePhysics.Mathematics
{
    /// <summary>
    /// Nystrm modification of fourth-order Runge-Kutta method
    /// of a solution of a set of ordinary second order differential
    /// equations for dynamics simulations. 
    /// Works with the second derivative and must consider the
    /// first derivative as input.
    /// Based on code from Vit Buchta, June 2007.
    /// </summary>
    public class NystromIntegrator
    {
        /// <summary>
        /// Calculates second derivative.
        /// </summary>
        ISecondDerivative secondDerivative;

        /// <summary>
        /// The current value of the time during iterations.
        /// </summary>
        private double currentTime;

        /// <summary>
        /// The current value of the position during iterations.
        /// </summary>
        private VectorN currentPosition;

        /// <summary>
        /// The current value of the velocity during iterations.
        /// </summary>
        private VectorN currentVelocity;

        /// <summary>
        /// The current value of the acceleration during iterations.
        /// </summary>
        private VectorN currentAcceleration;

        /// <summary>
        /// The time interval of an integration step.
        /// </summary>
        double timeStepsize;

        /// <summary>
        /// A constructor for the Nystrom integrator object. 
        /// </summary>
        /// <param name="secondDerivative"> An object implementing the ISecondDerivative interface</param>
        /// <param name="timeStepsize"> The time step</param>
        /// <param name="currentPosition"> The vector of initial positions</param>
        /// <param name="currentVelocity"> The vector of initial velocities</param>
        /// <param name="initialAcceleration"> The vector of initial accelerations</param>
        public NystromIntegrator(
            ISecondDerivative secondDerivative, 
            double timeStepsize,
            VectorN initialPosition, 
            VectorN initialVelocity)
        {
            this.secondDerivative = secondDerivative;
            this.timeStepsize = timeStepsize;
            this.Reset(initialPosition, initialVelocity);
        }

        /// <summary>
        /// Resets the Nystrom integrator object. Sets the time to 0 and
        /// recalculates the current acceleration. 
        /// </summary>
        /// <param name="currentPosition"> The vector of initial positions</param>
        /// <param name="currentVelocity"> The vector of initial velocities</param>
        /// <returns>The current acceleration after the reset.</returns>
        public VectorN Reset(
            VectorN initialPosition,
            VectorN initialVelocity)
        {
            // Set initial conditions
            this.currentTime = 0.0;
            this.currentPosition = initialPosition;
            this.currentVelocity = initialVelocity;

            // Calculate current acceleration
            this.currentAcceleration = secondDerivative.GetValue(
                this.currentTime, 
                this.currentPosition, 
                this.currentVelocity);

            return this.currentAcceleration;
        }

       /// <summary>
       /// One step of the integration procedure. Recalculates all positions, velocities
       /// and current acceleration. Updates the time by a time step.
       /// </summary>
       /// <param name="newTime"> The time at the end of the time interval</param>
       /// <param name="newPosition"> The vector of next positions</param>
       /// <param name="newVelocity"> The vector of next velocities</param>
       /// <param name="newAcceleration"> The vector of next accelerations</param>
        public void Step(
            out double newTime, 
            out VectorN newPosition, 
            out VectorN newVelocity, 
            out VectorN newAcceleration)
        {
            double timeStepsizeSquared = timeStepsize * timeStepsize;
            double timeStepsizeHalf = timeStepsize / 2.0;
            VectorN k1 = currentAcceleration;
            VectorN k2 = secondDerivative.GetValue(currentTime + timeStepsizeHalf, currentPosition + timeStepsizeHalf * currentVelocity + timeStepsizeSquared / 8 * k1, currentVelocity + timeStepsizeHalf * k1);
            VectorN k3 = secondDerivative.GetValue(currentTime + timeStepsizeHalf, currentPosition + timeStepsizeHalf * currentVelocity + timeStepsizeSquared/8 * k2, currentVelocity + timeStepsizeHalf * k2);
            VectorN k4 = secondDerivative.GetValue(currentTime + timeStepsize,   currentPosition + timeStepsize   * currentVelocity + timeStepsizeSquared/2 * k3, currentVelocity + timeStepsize   * k3);

            newTime = currentTime + timeStepsize;
            newPosition = currentPosition + timeStepsize * currentVelocity + timeStepsizeSquared/6 * (k1 + k2 + k3);
            newVelocity = currentVelocity + timeStepsize/6 * (k1 + 2 * k2 + 2 * k3 + k4);
            newAcceleration = secondDerivative.GetValue(newTime, newPosition, newVelocity);

            currentTime = newTime;
            currentPosition = newPosition;
            currentVelocity = newVelocity;
            currentAcceleration = newAcceleration;
        }
    }
}
