﻿//-----------------------------------------------------------------------------
// ValueUnbiaser.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Mathematics
{
    using System;
    using System.Collections.Generic;
    using System.Text;

    /// <summary>
    /// Enum describing different value unbiasing algorithms.
    /// </summary>
    public enum ValueUnbiasAlgorithm
    {
        /// <summary>
        /// Hotbits pairwise interval comparison technique.
        /// (0.5bits per sample)
        /// </summary>
        Pairwise,

        /// <summary>
        /// Compare against the block median.
        /// (1bit per sample)
        /// </summary>
        Median,

        /// <summary>
        /// Cipehrgoth's recursive partitioning scheme.
        /// (>1bit per sample)
        /// </summary>
        Partition,
    }

    /// <summary>
    /// Creates unbiased bit-streams from uncorrelated values.
    /// </summary>
    public class ValueUnbiaser
    {
        /// <summary>
        /// Currently selected algorithm for the int unbiaser.
        /// </summary>
        private ValueUnbiasAlgorithm algorithm;

        /// <summary>
        /// The bit-level unbiaser.
        /// </summary>
        private BitUnbiaser bitUnbiaser;

        /// <summary>
        /// Creates an instance of a value unbiaser which converts uncorrelated
        /// values into bit-streams.
        /// </summary>
        /// <param name="algorithm">The value unbiasing algorithm.</param>
        public ValueUnbiaser(ValueUnbiasAlgorithm algorithm)
        {
            this.algorithm = algorithm;

            // Create AMLS unbiaser for partitioning technique
            this.bitUnbiaser = new BitUnbiaser(BitUnbiasAlgorithm.AMLS);
        }

        /// <summary>
        /// Extract an unbiased bitstream as string of 0 and 1 characters from 
        /// an array of uncorrelated values.
        /// </summary>
        /// <param name="uncorrelatedData">Uncorrelated values (i.e. timing measurements
        /// of radioactive decay events).</param>
        /// <returns>Extracted bitstream as string consisting of 0s and 1s</returns>
        public string Extract(double[] uncorrelatedData)
        {
            if (uncorrelatedData.Length < 2)
            {
                return string.Empty;
            }

            StringBuilder sb = new StringBuilder();

            // Calculate intervals between data values
            double[] intervals = new double[uncorrelatedData.Length - 1];
            for (int i = 0; i < intervals.Length; i++)
            {
                intervals[i] = uncorrelatedData[i + 1] - uncorrelatedData[i];
            }

            // Unbias using selected algorithm
            switch (this.algorithm)
            {
                case ValueUnbiasAlgorithm.Pairwise:
                    this.PairwiseExtract(intervals, ref sb);
                    break;
                case ValueUnbiasAlgorithm.Median:
                    this.MedianExtract(intervals, ref sb);
                    break;
                case ValueUnbiasAlgorithm.Partition:
                    this.RecursiveExtract(intervals, ref sb);
                    break;
            }

            // Return collected bits
            return sb.ToString();
        }

        /// <summary>
        /// Generic swap
        /// </summary>
        /// <typeparam name="T">Type to swap.</typeparam>
        /// <param name="item1">Item 1 to swap.</param>
        /// <param name="item2">Item 2 to swap.</param>
        private static void Swap<T>(ref T item1, ref T item2)
        {
            T temp = item1;
            item1 = item2;
            item2 = temp;
        }

        /// <summary>
        /// Compare two intervals at a time, switching comparison order.
        /// </summary>
        /// <param name="intervals">Array of interval values.</param>
        /// <param name="collector">Bit collector.</param>
        private void PairwiseExtract(double[] intervals, ref StringBuilder collector)
        {
            if (intervals.Length < 3)
            {
                return;
            }

            int currentPos = 0;
            int lastPos = intervals.Length - 2;
            int swap = 0;
            do
            {

                double dt1 = intervals[currentPos++];
                double dt2 = intervals[currentPos++];

                // Local unbias by swapping order
                if (swap == 0)
                {
                    Swap<double>(ref dt1, ref dt2);
                    swap++;
                }
                else
                {
                    swap = 0;
                }

                // Add bits only when intervals differ
                if (dt1 < dt2)
                {
                    collector.Append('0');
                }
                else if (dt1 > dt2)
                {
                    collector.Append('1');
                }
                
                currentPos += 2;
            }
            while (currentPos <= lastPos);
        }

        /// <summary>
        /// Determine the median in the intervals array and 
        /// generate bits against the median.
        /// </summary>
        /// <param name="intervals">Array of interval values.</param>
        /// <param name="collector">Bit collector.</param>
        private void MedianExtract(double[] intervals, ref StringBuilder collector)
        {
            if (intervals.Length < 3)
            {
                return;
            }

            // Determine median interval
            double[] sortedIntervals = new double[intervals.Length];
            intervals.CopyTo(sortedIntervals, 0);
            Array.Sort(sortedIntervals);

            double medianLow;
            double medianHigh;
            if ((intervals.Length & 1) == 1)
            {
                // Odd number of elements, take middle value
                medianLow = sortedIntervals[(intervals.Length - 1) / 2];
                medianHigh = medianLow;
            }
            else
            {
                // Even number of elements take both values in the center
                medianLow = sortedIntervals[intervals.Length / 2 - 1];
                medianHigh = sortedIntervals[intervals.Length / 2];
            }

            // Generate bits using median values, excluding 
            // the median itself
            for (int i = 0; i < intervals.Length; i++)
            {
                if (intervals[i] < medianLow)
                {
                    collector.Append('0');
                }
                else if (intervals[i] > medianHigh)
                {
                    collector.Append('1');
                }
            }
        }

        /// <summary>
        /// Advanced recursive bit extraction, but repeatedly partioning the 
        /// interval array and applying AMLS bit unbiasing on the fragments.
        /// </summary>
        /// <param name="intervals">Array of interval values.</param>
        /// <param name="collector">Bit collector.</param>
        private void RecursiveExtract(double[] intervals, ref StringBuilder collector)
        {
            if (intervals.Length < 3)
            {
                return;
            }

            // Pick first value as pivot
            double pivot = intervals[0];

            // Compile list of all non-pivot integers
            List<double> nonPivotData = new List<double>();
            for (int i = 1; i < intervals.Length; i++)
            {
                if (intervals[i] != pivot)
                {
                    nonPivotData.Add(intervals[i]);
                }
            }

            // Early finish
            if (nonPivotData.Count == 0)
            {
                return;
            }

            StringBuilder amlsData;

            // Get bits (part 1)
            amlsData = new StringBuilder();
            for (int i = 1; i < intervals.Length; i++)
            {
                amlsData.Append((intervals[i] == pivot) ? '1' : '0');
            }

            if (amlsData.Length > 1)
            {
                collector.Append(this.bitUnbiaser.Process(amlsData.ToString()));
            }

            // Get bits (part 2)
            amlsData = new StringBuilder();
            for (int i = 0; i < nonPivotData.Count; i++)
            {
                amlsData.Append((nonPivotData[i] < pivot) ? '1' : '0');
            }

            if (amlsData.Length > 1)
            {
                collector.Append(this.bitUnbiaser.Process(amlsData.ToString()));
            }

            // Split data for recursion
            List<double> left = new List<double>();
            List<double> right = new List<double>();
            for (int i = 1; i < nonPivotData.Count; i++)
            {
                if (nonPivotData[i] < pivot)
                {
                    left.Add(nonPivotData[i]);
                }
                else
                {
                    right.Add(nonPivotData[i]);
                }
            }

            // Recurse
            RecursiveExtract(left.ToArray(), ref collector);
            RecursiveExtract(right.ToArray(), ref collector);
        }
    }
}
