﻿//-----------------------------------------------------------------------------
// InfoLinkTransmitter.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Networking
{
    using System;
    using System.Collections.Generic;
    using System.Net;
    using System.Net.Sockets;
    using System.Text;
    using System.Threading;

    /// <summary>
    /// Class to broadcast InfoLink objects via UDP to an InfoLinkReceiver.
    /// </summary>
    public class InfoLinkTransmitter : InfoLinkNetBase
    {
        /// <summary>
        /// Milliseconds between queue checks.
        /// </summary>
        private const int PollingInterval = 1000;

        /// <summary>
        /// Milliseconds between packet transmissions.
        /// </summary>
        private const int TransmissionInterval = 100;

        /// <summary>
        /// Sync root for thread syncronization.
        /// </summary>
        private object syncRoot = new object();

        /// <summary>
        /// Queue of info link items to send
        /// </summary>
        private List<InfoLink> transmitQueue = new List<InfoLink>();

        /// <summary>
        /// The thread running state.
        /// </summary>
        private volatile bool running = false;

        /// <summary>
        /// The thread running the transmitter.
        /// </summary>
        private Thread transmitterThread;

        /// <summary>
        /// Count of sent packets.
        /// </summary>
        private long transmitted;

        /// <summary>
        /// Count of transmission errors.
        /// </summary>
        private long errors;

        /// <summary>
        /// Creates am InfoLinkTransmitter to broadcast InfoLink objects 
        /// via UDP on the default port.
        /// </summary>
        public InfoLinkTransmitter()
        {
            this.port = InfoLinkNetDefaultPort;
        }

        /// <summary>
        /// Creates am InfoLinkTransmitter to broadcast InfoLink objects 
        /// via UDP on the specified port.
        /// </summary>
        /// <param name="port">The port to use.</param>
        public InfoLinkTransmitter(int port)
        {
            this.port = port;
        }

        /// <summary>
        /// Indicates wether the transmitter thread is running.
        /// </summary>
        public bool Running
        {
            get { return this.running; }
            set { this.running = value; }
        }

        /// <summary>
        /// Gets the number of errors which occured during transmission.
        /// </summary>
        public long Errors
        {
            get {
                return Interlocked.Read(ref this.errors);
            }
        }

        /// <summary>
        /// Gets the number of transmitted packets.
        /// </summary>
        public long Transmitted
        {
            get
            {
                return Interlocked.Read(ref this.transmitted);
            }
        }

        /// <summary>
        /// Destructor for InfoLinkTransmitter.
        /// </summary>
        ~InfoLinkTransmitter()
        {
            if (this.Connected)
            {
                this.Close();
            }
        }

        /// <summary>
        /// Connect and start the transmitter thread.
        /// </summary>
        public void StartTransmitter()
        {
            if (this.Connected)
            {
                this.Close();
            }

            this.Open();
            this.transmitterThread = new Thread(new ThreadStart(RunTransmitterThread));
            this.transmitterThread.Name = "InfoLink Transmitter";
            this.transmitterThread.IsBackground = true;
            this.transmitterThread.Start();
        }

        /// <summary>
        /// Stop the transmitter thread and disconnect.
        /// </summary>
        public void StopTransmitter()
        {
            if (this.Connected)
            {
                this.Close();
            }
        }

        /// <summary>
        /// Queue InfoLink object for broadcast.
        /// </summary>
        /// <param name="infoLink">The InfoLink object to transmit.</param>
        public void Transmit(InfoLink infoLink)
        {
            // Simply add the object to the queue for sending
            lock (syncRoot)
            {
                transmitQueue.Add(infoLink);
            }
        }

        /// <summary>
        /// Queue InfoLink objects for broadcast.
        /// </summary>
        /// <param name="infoLinks">The InfoLink objecst to transmit.</param>
        public void Transmit(InfoLink[] infoLinks)
        {
            // Simply add the object to the queue for sending
            lock (syncRoot)
            {
                transmitQueue.AddRange(infoLinks);
            }
        }

        /// <summary>
        /// Broadcast infoLink datagrams in a loop.
        /// </summary>
        private void RunTransmitterThread()
        {
            this.running = this.Connected;

            while (this.running)
            {
                // Check if there is a new infoLink item in the queue
                InfoLink infoLink = null;
                lock (syncRoot)
                {
                    if (transmitQueue.Count > 0)
                    {
                        // Dequeue
                        infoLink = transmitQueue[0];
                        transmitQueue.RemoveAt(0);
                    }
                }

                // Maybe transmit
                if (infoLink != null)
                {
                    string message = InfoLinkSerializer.Serialize(infoLink);
                    byte[] sendbuf = Encoding.UTF8.GetBytes(message);

                    try
                    {
                        int bytesSent;

                        bytesSent = this.udpClient.Send(sendbuf, sendbuf.Length);
                        if (bytesSent == sendbuf.Length)
                        {
                            Interlocked.Increment(ref this.transmitted);
                        }
                        else
                        {
                            Interlocked.Increment(ref this.errors);
                        }
                    }
                    catch
                    {
                        Interlocked.Increment(ref this.errors);
                    }

                    // Pace packet transmission
                    Thread.Sleep(TransmissionInterval);
                }
                else
                {
                    // Poll slowly
                    Thread.Sleep(PollingInterval);
                }
            }
        }

        /// <summary>
        /// Open the UDP client connection.
        /// </summary>
        private void Open()
        {
            this.udpClient = new UdpClient();
            this.endPoint = new IPEndPoint(IPAddress.Broadcast, this.port);
            this.udpClient.EnableBroadcast = true;
            this.udpClient.Connect(this.endPoint);
            this.Connected = true;
            this.errors = 0;
        }

        /// <summary>
        /// Close the UDP client connection.
        /// </summary>
        private void Close()
        {
            try
            {
                this.running = false;
                this.udpClient.Close();
                this.transmitterThread = null;
            }
            finally
            {
                this.Connected = false;
            }
        }
    }
}
