﻿//-----------------------------------------------------------------------------
// DoublePendulumSimulationBase.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.PhysicalElements
{
    using System;
    
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Graphics;

    using NewGamePhysics.Mathematics;
    using NewGamePhysics.Physics;

    /// <summary>
    /// Enum describing double pendulum hinges.
    /// Value corresponds to the arry index of the angle, speed or acceleration
    /// state value arrays.
    /// </summary>
    public enum DoublePendulumHinges
    {
        /// <summary>
        /// Selects the top (at rest) hinge.
        /// </summary>
        Hinge1 = 0,

        /// <summary>
        /// Selects the bottom (at rest) hinge.
        /// </summary>
        Hinge2 = 1,
    }

    /// <summary>
    /// Enum describing different state values available for a double pendulum.
    /// Value corresponds to the array index in the combined state value array.
    /// </summary>
    public enum DoublePendulumStateValues
    {
        /// <summary>
        /// Selects displacement angle of the top (at rest) mass.
        /// </summary>
        Angle1 = 0,

        /// <summary>
        /// Selects displacement angle of the bottom (at rest) mass.
        /// </summary>
        Angle2 = 1,

        /// <summary>
        /// Selects the angular velocity of the top (at rest) mass.
        /// </summary>
        Velocity1 = 2,

        /// <summary>
        /// Selects the angular velocity of the bottom (at rest) mass.
        /// </summary>
        Velocity2 = 3,

        /// <summary>
        /// Selects the angular velocity of the top (at rest) mass.
        /// </summary>
        Acceleration1 = 4,

        /// <summary>
        /// Selects the angular velocity of the bottom (at rest) mass.
        /// </summary>
        Acceleration2 = 5,
    }

    /// <summary>
    /// Base class for a double pendulum simulation.
    /// </summary>
    public abstract class DoublePendulumSimulationBase
    {
        /// <summary>
        /// The acceleration function of the double pendulum.
        /// </summary>
        private ISecondDerivative pendulumAcceleration;

        /// <summary>
        /// The integrator for the double pendulum.
        /// </summary>
        internal NystromIntegrator pendulumIntegrator;

        /// <summary>
        /// Animation parameter (time).
        /// </summary>
        private double t;

        /// <summary>
        /// Integrator stepsize.
        /// </summary>
        private double h;

        /// <summary>
        /// The friction model for the two hinges.
        /// </summary>
        private RotationalFrictionModel frictionModel;

        /// <summary>
        /// Origin of the pendulum anchor.
        /// </summary>
        internal Vector2 origin;

        /// <summary>
        ///  Physical state of the pendulum: position angle.
        /// </summary>
        internal VectorN theta;

        /// <summary>
        ///  Physical state of the pendulum: rotational speed.
        /// </summary>
        internal VectorN omega;

        /// <summary>
        ///  Physical state of the pendulum: rotational acceleration.
        /// </summary>
        internal VectorN acceleration;

        /// <summary>
        /// Precalculated square root of 2.
        /// </summary>
        internal static double sqrt2 = Math.Sqrt(2.0);

        #region public_properties

        /// <summary>
        /// Gets or sets the integrator stepsize (time step) in seconds.
        /// </summary>
        public double H
        {
            get { return h; }
            set { h = value; }
        }

        /// <summary>
        /// Gets or sets the animation parameter (time) in seconds.
        /// </summary>
        public double T
        {
            get { return t; }
            set { t = value; }
        }

        #endregion

        #region internal_properties

        /// <summary>
        /// Gets or sets the friction model for the hinges.
        /// </summary>
        internal RotationalFrictionModel FrictionModel
        {
            get { return this.frictionModel; }
            set { this.frictionModel = value; }
        }

        /// <summary>
        /// Gets or sets the acceleration function of the double pendulum.
        /// </summary>
        internal ISecondDerivative PendulumAcceleration
        {
            get { return this.pendulumAcceleration; }
            set { this.pendulumAcceleration = value; }
        }

        #endregion

        #region abstract_methods

        /// <summary>
        /// Resets the initial conditions (angles and rotational speeds) for 
        /// the pendulum simulation to a be at rest.
        /// </summary>
        public abstract void SetInitialConditionsAtRest();

        /// <summary>
        /// Calculate pedulum positions.
        /// </summary>
        /// <returns>The pendulum positions.</returns>
        public abstract Vector2[] GetPosition();

        /// <summary>
        /// Calculate scaled and positioned pedulum positions.
        /// </summary>
        /// <returns>The pendulum positions.</returns>
        /// <param name="origin">Anchor position of double square pendulum.</param>
        /// <param name="scale">Scaling factor for pendulum in pixels/m.</param>
        public abstract Vector2[] GetPosition(Vector2 origin, double scale);

        /// <summary>
        /// Calculate the sizes of each component.
        /// </summary>
        /// <returns>Array containing size values.</returns>
        public abstract double[] GetSize();

        /// <summary>
        /// Calculate the sizes of each component with the scale applied.
        /// </summary>
        /// <param name="scale">The scale factor for the size calculation.</param>
        /// <returns>Array containing size values.</returns>
        public abstract double[] GetSize(double scale);

        /// <summary>
        /// Calculates the current total energy of the system.
        /// </summary>
        /// <returns>The energy value.</returns>
        public abstract double GetEnergy();

        #endregion

        #region common_public_methods

        /// <summary>
        /// Sets the initial conditions (angles and rotational speeds) for 
        /// the pendulum simulation. Angles are measured relative to the direction of
        /// the gravitational acceleration in a counter-clockwise way.
        /// </summary>
        /// <param name="theta1">
        /// Angle of the top square measures along the line from the hinge to the 
        /// corner across the hinge, in radians.
        /// 0.0 is vertically downwards, counter-clockwise is positive. 
        /// </param>
        /// <param name="omega1">
        /// Rotational speed of the top pendulum square.
        /// In radians per second, counter-clockwise is positive.</param>
        /// <param name="theta2">
        /// Angle of the bottom square measures along the line from the hinge to the 
        /// corner across the hinge, in radians.
        /// 0.0 is vertically downwards, counter-clockwise is positive.</param>
        /// <param name="omega2">Rotational speed of the bottom pendulum square.
        /// In radians per second, counter-clockwise is positive.</param>
        public void SetInitialConditions(double theta1, double omega1, double theta2, double omega2)
        {
            // Store values
            this.theta[0] = theta1;
            this.theta[1] = theta2;
            this.omega[0] = omega1;
            this.omega[1] = omega2;

            // (Re)create integrator
            this.pendulumIntegrator = new NystromIntegrator(
                (ISecondDerivative)this.pendulumAcceleration,
                this.H,
                this.theta,
                this.omega);
        }

        /// <summary>
        /// Drive the pendulum with a tangential rotational force applied to a hinge.
        /// </summary>
        /// <param name="which">Which hinge to drive.</param>
        /// <param name="force">The tangential force to apply to the hinge.</param>
        /// <returns>The rotational speed change that was applied.</returns>
        public double Drive(DoublePendulumHinges which, double force)
        {
            // Apply a force to the hinges modifying speed
            double deltaOmega = force * this.H;
            this.omega[(int)which] += deltaOmega;
            return deltaOmega;
        }

        /// <summary>
        /// Animates the pendulum through one time step.
        /// </summary>
        public void Animate()
        {
            // Animate pendulum
            // double t; VectorN theta, omega, acceleration;
            this.pendulumIntegrator.Step(
                out this.t,
                out this.theta,
                out this.omega,
                out this.acceleration);

            /// Apply friction to the hinges modifying speed
            for (int i = 0; i < 2; i++)
            {
                this.omega[i] = this.FrictionModel.ApplyFriction(this.omega[i]);
            }
        }

        /// <summary>
        /// Get pedulum angles state (from last calculation).
        /// </summary>
        /// <returns>Two pendulum angles in radians.</returns>
        public double[] GetAngle()
        {
            double[] angles = new double[2];
            angles[0] = this.theta[0];
            angles[1] = this.theta[1];
            return angles;
        }

        /// <summary>
        /// Get pedulum angles state (from last calculation).
        /// </summary>
        /// <param name="which">The hinge to return the value for.</param>
        /// <returns>A pendulum angle in radians.</returns>
        public double GetAngle(DoublePendulumHinges which)
        {
            return this.theta[(int)which];
        }

        /// <summary>
        /// Get pedulum speeds from last calculation.
        /// </summary>
        /// <returns>Two pendulum speeds in radians/sec.</returns>
        public double[] GetVelocity()
        {
            double[] omega = new double[2];
            omega[0] = this.omega[0];
            omega[1] = this.omega[1];
            return omega;
        }

        /// <summary>
        /// Get A pedulum speed from last calculation.
        /// </summary>
        /// <param name="which">The hinge to return the value for.</param>
        /// <returns>A pendulum speed in radians/sec.</returns>
        public double GetVelocity(DoublePendulumHinges which)
        {
            return this.omega[(int)which];

        }

        /// <summary>
        /// Get pedulum acceleration state (from last calculation).
        /// </summary>
        /// <returns>Two pendulum accelerations in radians/sec^2.</returns>
        public double[] GetAcceleration()
        {
            double[] omega = new double[2];
            omega[0] = this.acceleration[0];
            omega[1] = this.acceleration[1];
            return omega;
        }

        /// <summary>
        /// Get pedulum acceleration state (from last calculation).
        /// </summary>
        /// <param name="which">The hinge to return the value for.</param>
        /// <returns>A pendulum acceleration in radians/sec^2.</returns>
        public double GetAcceleration(DoublePendulumHinges which)
        {
            return this.acceleration[(int)which];
        }

        /// <summary>
        /// Get complete pedulum state (from last calculation).
        /// </summary>
        /// <returns>Two pendulum angle, speed and accelerations values each
        /// in their respective units.</returns>
        public double[] GetState()
        {
            double[] values = new double[6];
            values[(int)DoublePendulumStateValues.Angle1] =
                this.theta[(int)DoublePendulumHinges.Hinge1];
            values[(int)DoublePendulumStateValues.Angle2] =
                this.theta[(int)DoublePendulumHinges.Hinge2];
            values[(int)DoublePendulumStateValues.Velocity1]
                = this.omega[(int)DoublePendulumHinges.Hinge1];
            values[(int)DoublePendulumStateValues.Velocity2]
                = this.omega[(int)DoublePendulumHinges.Hinge2];
            values[(int)DoublePendulumStateValues.Acceleration1]
                = this.theta[(int)DoublePendulumHinges.Hinge1];
            values[(int)DoublePendulumStateValues.Acceleration2]
                = this.theta[(int)DoublePendulumHinges.Hinge2];
            return values;
        }

        /// <summary>
        /// Get complete pedulum state (from last calculation).
        /// </summary>
        /// <param name="which">The type of state value to return.</param>
        /// <returns>A pendulum angle, speed or acceleration value
        /// in its respective unit.</returns>
        public double GetState(DoublePendulumStateValues which)
        {
            return this.GetState()[(int)which];
        }

        #endregion
    }
}
