﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using NewGamePhysics.Networking;

namespace NewGamePhysics.PhysicalElements
{
    /// <summary>
    /// Enumeration for the possible entropy sources.
    /// </summary>
    public enum EntropySourceType
    {
        /// <summary>
        /// Use the software pseudo random number generator:
        /// simulated entropy
        /// </summary>
        Pseudo,

        /// <summary>
        /// Use the physical random bit source
        /// of the play truly random webservice
        /// </summary>
        PlayTrulyRandom
    }

    /// <summary>
    /// A random number generator using a physical entropy source (if available).
    /// Falls transparently back on PRNG on errors.
    /// </summary>
    public class PhysicalRandomNumberGenerator
    {
        /// <summary>
        /// The current type of random number source.
        /// </summary>
        private EntropySourceType entropySource = 
            EntropySourceType.Pseudo;

        /// <summary>
        /// The name of the agent using the generator.
        /// </summary>
        private string agentName = 
            string.Empty;

        /// <summary>
        /// Count of entropy bits that were retrieved.
        /// </summary>
        private int retrievedEntropyBits;

        /// <summary>
        /// Pseudo random number generator.
        /// </summary>
        private Random prng = 
            new Random();

        /// <summary>
        /// PlayTrulyRandom entropy source.
        /// </summary>
        private PlayTrulyRandom playTrulyRandom;

        /// <summary>
        /// Gets the current PTR instance.
        /// </summary>
        public PlayTrulyRandom PlayTrulyRandom
        {
            get { return this.playTrulyRandom; }
        }

        /// <summary>
        /// Gets the current type of random number entropy.
        /// </summary>
        public EntropySourceType EntropySource
        {
            get { return this.entropySource; }
            set { 
                this.entropySource = value; 

                // Re-initialize
                this.Initialize();
            }
        }

        /// <summary>
        /// Gets the current type of random number entropy.
        /// </summary>
        public string AgentName
        {
            get { return this.agentName; }
        }

        /// <summary>
        /// Gets or Sets the count of entropy bits that were retrieved.
        /// </summary>
        public int RetrievedEntropyBits
        {
            get { return this.retrievedEntropyBits; }
            set { this.retrievedEntropyBits = value; }
        }

        /// <summary>
        /// Create a physical random number generator.
        /// </summary>
        /// <param name="source">The source of the entropy.</param>
        /// <param name="agentName">The name of the agent using the generator (i.e. the
        /// name of the game or webservice access code).</param>
        public PhysicalRandomNumberGenerator(
            EntropySourceType source, 
            string agentName)
        {
            // Keep state
            this.entropySource = source;
            this.agentName = agentName;

            // Initialization
            this.Initialize();
        }

        /// <summary>
        /// Generate random number within a given range [min,max]
        /// against the current entropy source.
        /// </summary>
        /// <param name="minValue">
        /// The inclusive minimum value to be generated.
        /// </param>
        /// <param name="maxValue">
        /// The inclusive maximum value to be generated. 
        /// maxValue must be greater or equals than minValue.
        /// </param>
        /// <returns>A random number within the range.</returns>
        public int Next(int minValue, int maxValue)
        {
            if (maxValue < minValue)
            {
                throw new ArgumentException(
                    "maxValue must be greater or equals than minValue",
                    "maxValue");
            }

            int result = 0;

            // Generate based on entropy source type
            switch (this.entropySource)
            {
                // OS based pseudo generator
                case EntropySourceType.Pseudo:
                    // Adjust for exclusive range of .Next
                    if (maxValue < int.MaxValue)
                    {
                        maxValue++;
                    }

                    result = this.prng.Next(minValue, maxValue);
                    this.retrievedEntropyBits += 32;
                    break;

                // PTR webservice
                case EntropySourceType.PlayTrulyRandom:
                    try
                    {
                        int retrievedBits;
                        result = playTrulyRandom.Next(minValue, maxValue, out retrievedBits);
                        this.retrievedEntropyBits += retrievedBits;
                    }
                    catch (Exception)
                    {
                        // On error, fall back on pseudo random number generator
                        this.entropySource = EntropySourceType.Pseudo;
                        result = this.Next(minValue, maxValue);
                    }

                    break;
            }

            return result;
        }

        /// <summary>
        /// Type based initialization of PRNG.
        /// </summary>
        private void Initialize()
        {
            // Always initialize pseudo random number generator
            this.prng = new Random();

            // Initialize based on type
            switch (this.entropySource)
            {
                case EntropySourceType.PlayTrulyRandom:
                    this.playTrulyRandom = new PlayTrulyRandom(agentName);
                    try
                    {
                        // Register with webservice
                        this.playTrulyRandom.RegisterAgent();
                    }
                    catch
                    {
                        // On error, fall back on pseudo random number generator
                        this.entropySource = EntropySourceType.Pseudo;
                    }

                    break;
            }
        }
    }
}
