﻿//-----------------------------------------------------------------------------
// AdvancedEntropyCollector.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Physics
{
    using System;
    using System.IO;
    using System.Collections.Generic;
    using System.Text;

    using NewGamePhysics.Utilities;
    using NewGamePhysics.Mathematics;

    /// <summary>
    /// Extracts entropy from an array of floating point values of physical
    /// processes (measurements or simulations fed by physical processes).
    /// Uses a recursive partition scheme and the AMLS unbiaser to extract
    /// random bits from the values.
    /// The submissions of the values will yield usable entropy if:
    /// * if the value changes sufficiently much between submissions 
    /// * the submissions are independent from the value generation
    /// * either the value source or the submission trigger is based on
    ///   a real physical random process (i.e. uncorrelated user trigger)
    /// </summary>
    public class AdvancedEntropyCollector
    {
        /// <summary>
        /// Collection of value to generate entropy from.
        /// </summary>
        private List<double> valuePool;

        /// <summary>
        /// The unbiaser to perform the value-to-bit conversion.
        /// </summary>
        private ValueUnbiaser unbiaser;

        /// <summary>
        /// Default constructor.
        /// </summary>
        public AdvancedEntropyCollector()
        {
            // Initialize value collector
            this.valuePool = new List<double>();
            
            // Create unbiaser using partitioning algorithm
            this.unbiaser = new ValueUnbiaser(ValueUnbiasAlgorithm.Partition);

        }

        /// <summary>
        /// Returns the number of collected values for entropy generation.
        /// </summary>
        public int ValuePoolSize
        {
            get { return this.valuePool.Count; }
        }

        /// <summary>
        /// Reset collector to initial state.
        /// </summary>
        public void Reset()
        {
            // Clear value pool
            valuePool.Clear();
        }

        /// <summary>
        /// Adds a float value to the collector.
        /// </summary>
        /// <param name="value">Value to add.</param>
        public void AddValue(double value)
        {
            valuePool.Add(value);
        }

        /// <summary>
        /// Return the current value pool as unbiased entropy bits 
        /// in a string of 0 and 1 characters. Does not reset or modify
        /// the value pool.
        /// </summary>
        /// <returns>The current value pool as entropy bit string.</returns>
        public override string ToString()
        {
            if (this.valuePool.Count > 2)
            {
                return this.unbiaser.Extract(this.valuePool.ToArray());
            }
            else
            {
                return string.Empty;
            }

        }

    }
}
