﻿//-----------------------------------------------------------------------------
// ButterworthFilter.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Physics
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;

    /// <summary>
    /// Butterworth filter types.
    /// </summary>
    public enum ButterworthFilterType
    {
        /// <summary>
        /// A low-pass filter
        /// </summary>
        LowPass,

        /// <summary>
        /// A high-pass filter
        /// </summary>
        HighPass,
    }

    /// <summary>
    /// Implementation of a numerical biquad (tweaked) Butterworth filter.
    /// </summary>
    /// <remarks>
    /// Source: http://www.musicdsp.org/archive.php?classid=3#38 and
    /// Source: http://www.musicdsp.org/archive.php?classid=3#128 and
    /// Source: http://en.wikipedia.org/wiki/Digital_biquad_filter
    /// </remarks>
    public class ButterworthFilter
    {
        /// <summary>
        /// Square root of 2
        /// </summary>
        static public double DefaultResonance = Math.Sqrt(2.0);

        /// <summary>
        /// Biquad filter parameter a1.
        /// </summary>
        private double a1;

        /// <summary>
        /// Biquad filter parameter a2.
        /// </summary>
        private double a2;

        /// <summary>
        /// Biquad filter filter parameter a3.
        /// </summary>
        private double a3;

        /// <summary>
        /// Biquad filter parameter b1.
        /// </summary>
        private double b1;

        /// <summary>
        /// Biquad filter parameter b2.
        /// </summary>
        private double b2;

        /// <summary>
        /// Create an instance of a biquad Butterworth filter.
        /// </summary>
        /// <param name="filterType">
        /// The type of the filter.
        /// </param>
        /// <param name="sampleRate">
        /// The sample rate (Hz) to assume.
        /// </param>
        /// <param name="cutoffFrequency">
        /// The cutoff frequency (Hz).
        /// Theoretical value 0 Hz to SampleRate/2, but practical is only SampleRate/4.
        /// </param>
        /// <remarks>
        /// Uses default resonance of sqrt(2).
        /// </remarks>
        public ButterworthFilter(
            ButterworthFilterType filterType,
            double sampleRate,
            double cutoffFrequency) : this(filterType, sampleRate, cutoffFrequency, DefaultResonance) 
        {
        }

        /// <summary>
        /// Create an instance of a biquad Butterworth filter.
        /// </summary>
        /// <param name="filterType">
        /// The type of the filter.
        /// </param>
        /// <param name="sampleRate">
        /// The sample rate (Hz) to assume.
        /// </param>
        /// <param name="cutoffFrequency">
        /// The cutoff frequency (Hz).
        /// Theoretical value 0 Hz to SampleRate/2, but practical is only SampleRate/4.
        /// </param>
        /// <param name="resonance">
        /// A resonance amount. Allowable range is from sqrt(2) to 0.1 only.
        /// At sqrt(2) resonance is 0dB, smaller values increase resonance.
        /// </param>
        public ButterworthFilter(
            ButterworthFilterType filterType,
            double sampleRate,
            double cutoffFrequency,
            double resonance)
        {
            // Check input parameters
            if (sampleRate <= 0.0)
            {
                throw new ArgumentException("The sample rate cannot be zero or negative.");
            }

            if ((cutoffFrequency <= 0.0) || 
                (cutoffFrequency > (sampleRate / 2.0)))
            {
                throw new ArgumentException("The cutoff frequency should be greater than " +
                    "zero, but less than half the sample rate");
            }

            if ((resonance < 0.1) || (resonance > DefaultResonance))
            {
                throw new ArgumentException("The resonance parameter should be in " +
                    " the range 0.1 to sqrt(2)");
            }

            // Calculate filter parameters
            switch (filterType)
            {
                case ButterworthFilterType.LowPass:
                    this.InitializeLowPass(sampleRate, cutoffFrequency, resonance);
                    break;
                case ButterworthFilterType.HighPass:
                    this.InitializeHighPass(sampleRate, cutoffFrequency, resonance);
                    break;
            }
        }

        /// <summary>
        /// Calculate parameters for low pass filter.
        /// </summary>
        /// <param name="sampleRate">The sample rate.</param>
        /// <param name="cutoffFrequency">The cutoff frequency.</param>
        /// <param name="resonance">The resonance.</param>
        private void InitializeLowPass(
            double sampleRate,
            double cutoffFrequency,
            double resonance)
        {
            double c = 1.0 / Math.Tan(Math.PI * cutoffFrequency / sampleRate);
            this.a1 = 1.0 / (1.0 + resonance * c + c * c);
            this.a2 = 2 * this.a1;
            this.a3 = this.a1;
            this.b1 = 2.0 * (1.0 - c * c) * this.a1;
            this.b2 = (1.0 - resonance * c + c * c) * this.a1;
        }

        /// <summary>
        /// Calculate parameters for high pass filter.
        /// </summary>
        /// <param name="sampleRate">The sample rate.</param>
        /// <param name="cutoffFrequency">The cutoff frequency.</param>
        /// <param name="resonance">The resonance.</param>
        private void InitializeHighPass(
            double sampleRate,
            double cutoffFrequency,
            double resonance)
        {
            double c = Math.Tan(Math.PI * cutoffFrequency / sampleRate);
            this.a1 = 1.0 / (1.0 + resonance * c + c * c);
            this.a2 = -2 * this.a1;
            this.a3 = this.a1;
            this.b1 = 2.0 * (c * c - 1.0) * this.a1;
            this.b2 = (1.0 - resonance * c + c * c) * this.a1;
        }

        /// <summary>
        /// Calculate the filtered samples from the input samples.
        /// </summary>
        /// <param name="samples">
        /// Sample array. Minimum size of array is 3 samples.
        /// </param>
        /// <returns>Filtered sample array. 
        /// First two samples are 0.0.</returns>
        public double[] Calculate(
            double[] samples)
        {
            if (samples == null)
            {
                throw new ArgumentNullException("samples");
            }

            int nMax = samples.Length;
            if (nMax < 3)
            {
                throw new ArgumentException(
                    "samples", 
                    "The samples array must contain at least 3 values.");
            }

            double[] filteredSamples = new double[nMax];
            filteredSamples[0] = 0.0;
            filteredSamples[1] = 0.0;
            for (int n = 2; n < nMax; n++)
            {
                filteredSamples[n] =
                    this.a1 * samples[n] +
                    this.a2 * samples[n - 1] +
                    this.a3 * samples[n - 2] -
                    this.b1 * filteredSamples[n - 1] -
                    this.b2 * filteredSamples[n - 2];
            }

            return filteredSamples;
        }
    }
}
