﻿// --------------------------------------------------------
// CelestialBody.cs
// (A. Schiffler, 2009)
// --------------------------------------------------------

namespace NewGamePhysics.Physics
{
    using System;
    using System.Text;

    /// <summary>
    /// Enum describing various celestial objects.
    /// </summary>
    public enum CelestialObject
    {
        Sun,
        Mercury,
        Venus,
        Earth,
        Moon,
        Mars,
        Jupiter,
        Io,
        Europa,
        Ganymede,
        Callisto,
        Saturn,
        Uranus,
        Neptune,
        Pluto
    }

    /// <summary>
    /// Class representing a celestial body.
    /// </summary>
    public class CelestialBody
    {
        /// <summary>
        /// The celestial object enum.
        /// </summary>
        private CelestialObject celestialObject;

        /// <summary>
        /// Create a new instance of a celestial body.
        /// </summary>
        /// <param name="type"></param>
        public CelestialBody(CelestialObject type)
        {
            this.celestialObject = type;
        }

        /// <summary>
        /// Gets or sets the celestial object enum for this body.
        /// </summary>
        public CelestialObject CelestialObject
        {
            get { return this.celestialObject; }
            set { this.celestialObject = value; }
        }

        public string GetBodyInfo()
        {
            StringBuilder sb = new StringBuilder();

            switch (this.celestialObject)
            {
                case CelestialObject.Sun:
                    sb.AppendLine("The Sun is the star at the center of the Solar System.");
                    sb.AppendLine("Distance from Earth: 1.496E11 m (8.31 min at light speed)");
                    sb.AppendLine("Mean Diameter: 1.392E9 m (109 Earths)");
                    sb.AppendLine("Mass: 1.9891E30 kg (332946 Earths)");
                    sb.AppendLine("Temperature: 5778 K (on surface)");
                    sb.AppendLine("Source: http://www.wikipedia.com");
                    break;
                case CelestialObject.Mercury:
                    sb.AppendLine("The innermost and smallest planet of the Solar System.");
                    sb.AppendLine("Mean Radius: 2.4397E6 m (0.383 Earths)");
                    sb.AppendLine("Mass: 3.3022E23 kg (0.055 Earths)");
                    sb.AppendLine("Temperature: 80-700 K");
                    sb.AppendLine("Composition: 70% metallic, 30% silicate material");
                    sb.AppendLine("Source: http://www.wikipedia.com");
                    break;
                case CelestialObject.Venus:
                    sb.AppendLine("The second-closest planet to the Sun in the Solar System.");
                    sb.AppendLine("Mean Radius: 6.0518E6 m (0.95 Earths)");
                    sb.AppendLine("Mass: 4.8685E24 kg (0.815 Earths)");
                    sb.AppendLine("Temperature: 735 K");
                    sb.AppendLine("Atmospheric Pressure: 92x Earth's Pressure");
                    sb.AppendLine("Source: http://www.wikipedia.com");
                    break;
                case CelestialObject.Earth:
                    sb.AppendLine("The third planet from the Sun in the Solar System.");
                    sb.AppendLine("Mean Radius: 6.371E6 m");
                    sb.AppendLine("Mass: 5.9736E24 kg");
                    sb.AppendLine("Temperature: 184-331 K");
                    sb.AppendLine("Age: 4.54E9 years");
                    sb.AppendLine("Source: http://www.wikipedia.com");
                    break;
                case CelestialObject.Moon:
                    sb.AppendLine("Earth's only natural satellite and the fifth larges in the Solar System.");
                    sb.AppendLine("Mean Radius: 1.7371E6 m (0.273 Earths)");
                    sb.AppendLine("Mass: 7.3477E22 kg (0.0123 Earths)");
                    sb.AppendLine("Temperature: 70-390 K");
                    sb.AppendLine("Distance Earth-Moon: 3.84403E8 m (30 Earths)");
                    sb.AppendLine("Source: http://www.wikipedia.com");
                    break;
                case CelestialObject.Mars:
                    sb.AppendLine("The fourth planet from the Sun in the Solar System.");
                    sb.AppendLine("Equatorial Radius: 3.3962E6 m (0.533 Earths)");
                    sb.AppendLine("Mass: 6.418E23 (0.107 Earths)");
                    sb.AppendLine("Temperature: 186-268 K");
                    sb.AppendLine("Moons: two (Phobos and Deimos)");
                    sb.AppendLine("Source: http://www.wikipedia.com");
                    break;
                case CelestialObject.Jupiter:
                    sb.AppendLine("The fifth planet from the sun and largest planet in the Solar System.");
                    sb.AppendLine("Mean Radius: 7.1492E7 m (11.21 Earths)");
                    sb.AppendLine("Mass: 1.8986E27 kg (317.8 Earths)");
                    sb.AppendLine("Temperature: 112-165 K");
                    sb.AppendLine("Composition: 89.8% Hydrogen 10.2% Helium");
                    sb.AppendLine("Source: http://www.wikipedia.com");
                    break;
                case CelestialObject.Io:
                    sb.AppendLine("Is the innermost of the Galilean moon of Jupiter.");
                    sb.AppendLine("Mean Radius: 1.8213E6 m (0.286 Earths)");
                    sb.AppendLine("Mass: 8.9319E22 kg (0.015 Earths)");
                    sb.AppendLine("Temperature: 130-200 K");
                    sb.AppendLine("Volcanism: 400 active volcanoes (sulfur, sulfur dioxide)");
                    sb.AppendLine("Source: http://www.wikipedia.com");
                    break;
                case CelestialObject.Europa:
                    sb.AppendLine("The sixth and smallest Galilean moon of the planet Jupiter.");
                    sb.AppendLine("Mean Radius: 1.569E6 m (0.245 Earths)");
                    sb.AppendLine("Mass: 4.80E22 kg (0.008 Earths)");
                    sb.AppendLine("Temperature: 50-125 K");
                    sb.AppendLine("Surface: 0.64 high albedo icy crust");
                    sb.AppendLine("Source: http://www.wikipedia.com");
                    break;
                case CelestialObject.Ganymede:
                    sb.AppendLine("The seventh moon of the planet Jupiter and largest one in the Solar System.");
                    sb.AppendLine("Mean Radius: 2.6341E6 m (0.413 Earths)");
                    sb.AppendLine("Mass: 1.4819E23kg (0.025 Earths)");
                    sb.AppendLine("Temperature: 70-152 K");
                    sb.AppendLine("Participates in a 1:2:4 orbital resonance with the satellites Europa and Io.");
                    sb.AppendLine("Source: http://www.wikipedia.com");
                    break;
                case CelestialObject.Callisto:
                    sb.AppendLine("The fourth moon of the planet Jupiter and third larges one in the Solar System.");
                    sb.AppendLine("Mean Radius: 2.4103E6 m (0.378 Earths)");
                    sb.AppendLine("Mass: 1.0759E23 kg (0.018 Earths)");
                    sb.AppendLine("Temperature: 80-165 K");
                    sb.AppendLine("Surface: the oldest and most cratered in the Solar System");
                    sb.AppendLine("Source: http://www.wikipedia.com");
                    break;
                case CelestialObject.Saturn:
                    sb.AppendLine("The sixth planet from the Sun and second largest in the Solar System.");
                    sb.AppendLine("Equatorial Radius: 6.0268E7 m (9.45 Earths)");
                    sb.AppendLine("Mass: 5.6846E26 kg (763 Earths)");
                    sb.AppendLine("Temperature: 84-134 K");
                    sb.AppendLine("Cloud Features: hexagonal wave pattern around north pole");
                    sb.AppendLine("Source: http://www.wikipedia.com");
                    break;
                case CelestialObject.Uranus:
                    sb.AppendLine("The seventh planet from the Sun in the Solar System.");
                    sb.AppendLine("Mean Radius: 2.559E7 m (4.00 Earths)");
                    sb.AppendLine("Mass: 8.6810E25kg (14.54 Earths)");
                    sb.AppendLine("Temperature: 49-76 K");
                    sb.AppendLine("Orbit: period 84 years");
                    sb.AppendLine("Source: http://www.wikipedia.com");
                    break;
                case CelestialObject.Neptune:
                    sb.AppendLine("The eighth planet from the Sun in the Solar System.");
                    sb.AppendLine("Equatorial Radius: 2.4764E7 m (3.88 Earths)");
                    sb.AppendLine("Mass: 1.0243E26 kg (17.15 Earths)");
                    sb.AppendLine("Temperature: 55-72 K");
                    sb.AppendLine("Weather: strongest sustained wind speed of 2100 km/h");
                    sb.AppendLine("Source: http://www.wikipedia.com");
                    break;
                case CelestialObject.Pluto:
                    sb.AppendLine("Second larges known body in the Kuiper belt. ");
                    sb.AppendLine("Mean Radius: 1.195E6 m (0.19 Earths)");
                    sb.AppendLine("Mass: 1.305E22 kg (0.0021 Earths)");
                    sb.AppendLine("Temperature: 33-55 K");
                    sb.AppendLine("Orbit: highly inclined (17 deg) to the ecliptic");
                    sb.AppendLine("Source: http://www.wikipedia.com");
                    break;
            }

            return sb.ToString();
        }

        /// <summary>
        /// Get the information about the texture.
        /// </summary>
        /// <returns>A text describing the texture.</returns>
        public string GetTextureInfo()
        {
            StringBuilder sb = new StringBuilder();

            switch (this.celestialObject)
            {
                case CelestialObject.Sun:
                    sb.AppendLine("Artistic map based on images from SOHO.");
                    sb.AppendLine("(c) Calvin J. Hamilton, 2001. (educational use)");
                    sb.AppendLine("Source: http://solarviews.com");
                    break;
                case CelestialObject.Mercury:
                    sb.AppendLine("Map created from several sources including the Messenger spacecraft.");
                    sb.AppendLine("Contains fictional or cloned data.");
                    sb.AppendLine("Source: http://www.solarviews.com/eng/homepage.htm and http://home.arcor.de/jimpage");
                    break;
                case CelestialObject.Venus:
                    sb.AppendLine("Map created from 21 images taken by the Galileo spacecraft during its 1990 flyby.");
                    sb.AppendLine("The coloring is artificially enhanced to show features of the cloud cover.");
                    sb.AppendLine("Created by Bjorn Jonsson");
                    sb.AppendLine("Source: http://www.mmedia.is/~bjj/planetary_maps.html");
                    break;
                case CelestialObject.Earth:
                    sb.AppendLine("Map created from one years worth of monthly composites of images taken by the MODIS spacecraft in 2004.");
                    sb.AppendLine("The map set is also known as Blue Marble: Next Generation.");
                    sb.AppendLine("Cloud cover was removed via extensive automatic image processing.");
                    sb.AppendLine("Credits: NASA's Earth Observatory");
                    sb.AppendLine("Source: http://earthobservatory.nasa.gov/Features/BlueMarble");
                    break;
                case CelestialObject.Moon:
                    sb.AppendLine("Map created from USGS/Hubble and ESA Info Centre images.");
                    sb.AppendLine("Source: http://www.oera.net/How2/TextureMaps2.htm");
                    break;
                case CelestialObject.Mars:
                    sb.AppendLine("Map created from images taken by the Viking spacecraft and processes by the USGS.");
                    sb.AppendLine("Details added by incorporating topology data from the Mars Global Surveyor.");
                    sb.AppendLine("Manually corrected and color enhanced.");
                    sb.AppendLine("Source: http://planetpixelemporium.com/mars.html");
                    break;
                case CelestialObject.Jupiter:
                    sb.AppendLine("Map created from 100 images taken by the Cassini spacecraft during its Dec 2000 flyby.");
                    sb.AppendLine("Longitudinal position on left edge is arbitrary, as object has no fixed visible features.");
                    sb.AppendLine("Created by Bjorn Jonsson.");
                    sb.AppendLine("Source: http://www.mmedia.is/~bjj/planetary_maps.html");
                    break;
                case CelestialObject.Io:
                    sb.AppendLine("Map created from 1 high resolution greyscale image taken by the Voyager 1 spacecraft.");
                    sb.AppendLine("Color information comes from 3 low resolution images taken by the Voyager 2 spacecraft");
                    sb.AppendLine("Contains fictional or cloned data.");
                    sb.AppendLine("Created by Bjorn Jonsson.");
                    sb.AppendLine("Source: http://www.mmedia.is/~bjj/planetary_maps.html");
                    break;
                case CelestialObject.Europa:
                    sb.AppendLine("Map created from 14 images taken by the Galileo spacecraft.");
                    sb.AppendLine("Color information comes from several images taken by the Voyager 1 and 2 spacecraft");
                    sb.AppendLine("Created by Bjorn Jonsson.");
                    sb.AppendLine("Source: http://www.mmedia.is/~bjj/planetary_maps.html");
                    break;
                case CelestialObject.Ganymede:
                    sb.AppendLine("Map created from over 100 images taken by the Voyager and Galileo spacecraft.");
                    sb.AppendLine("Contains fictional or cloned data.");
                    sb.AppendLine("Created by Bjorn Jonsson.");
                    sb.AppendLine("Source: http://www.mmedia.is/~bjj/planetary_maps.html");
                    break;
                case CelestialObject.Callisto:
                    sb.AppendLine("Map created from several images taken by the Voyager and Galileo spacecraft.");
                    sb.AppendLine("Contains fictional or cloned data.");
                    sb.AppendLine("Created by Bjorn Jonsson.");
                    sb.AppendLine("Source: http://www.mmedia.is/~bjj/planetary_maps.html");
                    break;
                case CelestialObject.Saturn:
                    sb.AppendLine("Map created from 56 images taken by the Cassini spacecraft in Sept 2004.");
                    sb.AppendLine("Created by Bjorn Jonsson");
                    sb.AppendLine("Source: http://www.mmedia.is/~bjj/planetary_maps.html");
                    break;
                case CelestialObject.Neptune:
                    sb.AppendLine("Map created from 27 images taken by the Voyager 2 spacecraft in Aug 1989.");
                    sb.AppendLine("Contains fictional or cloned data.");
                    sb.AppendLine("Created by Bjorn Jonsson");
                    sb.AppendLine("Source: http://www.mmedia.is/~bjj/planetary_maps.html");
                    break;
                case CelestialObject.Uranus:
                    sb.AppendLine("Painted map based on various image sources including the Hubble space telescope.");
                    sb.AppendLine("Source: http://planetpixelemporium.com/uranus.html");
                    break;
                case CelestialObject.Pluto:
                    sb.AppendLine("Fictional rendering of object.");
                    break;
            }

            return sb.ToString();
        }

        /// <summary>
        /// Get the nominal known atmospheric density.
        /// </summary>
        /// <returns>A density number in (kg/m3)</returns>
        public double GetAtmosphericDensity()
        {
            switch (this.celestialObject)
            {

                case CelestialObject.Sun:
                    // Average coronal density
                    // Source: http://en.wikipedia.org/wiki/Sun
                    return 1.0E-12;

                case CelestialObject.Mercury:
                    // Essentially a vaccuum
                    // Source: http://nssdc.gsfc.nasa.gov/planetary/factsheet/mercuryfact.html
                    return 1.0E-18;

                case CelestialObject.Venus:
                    // Density at source 6.5% that of water
                    // Source: http://en.wikipedia.org/wiki/Venus#Atmosphere_and_climate
                    return 65.0;

                case CelestialObject.Earth:
                    return AirDensity.NominalAirDensity;

                case CelestialObject.Moon:
                    // Essentially a vaccuum
                    // Source: http://en.wikipedia.org/wiki/Moon#Atmosphere
                    return 1.0E-18;

                case CelestialObject.Mars:
                    // Nominal density
                    // Source: http://nssdc.gsfc.nasa.gov/planetary/factsheet/marsfact.html
                    return 0.02;

                case CelestialObject.Jupiter:
                    // Density at 1 bar
                    // Source: http://nssdc.gsfc.nasa.gov/planetary/factsheet/jupiterfact.html
                    return 0.16;

                case CelestialObject.Io:
                    // About 10 million times less than Earth
                    // Source: http://vega.lpl.arizona.edu/iotorus/io.html
                    return AirDensity.NominalAirDensity / 10000000;

                case CelestialObject.Europa:
                    // Assume a vaccuum
                    return 1.0E-18;

                case CelestialObject.Ganymede:
                    // Assume a vaccuum
                    return 1.0E-18;

                case CelestialObject.Callisto:
                    // Assume a vaccuum
                    return 1.0E-18;

                case CelestialObject.Saturn:
                    // Density at 1 bar
                    // Source: http://nssdc.gsfc.nasa.gov/planetary/factsheet/saturnfact.html
                    return 0.19;

                case CelestialObject.Uranus:
                    // Density at 1 bar
                    // Source: http://nssdc.gsfc.nasa.gov/planetary/factsheet/uranusfact.html
                    return 0.42;

                case CelestialObject.Neptune:
                    // Density at 1 bar
                    // Source: http://nssdc.gsfc.nasa.gov/planetary/factsheet/neptunefact.html
                    return 0.45;

                case CelestialObject.Pluto:
                    // Assume a vaccuum
                    return 1.0E-18;

                default:
                    // Default to vacuum
                    return 1.0E-18;
            }
        }

        /// <summary>
        /// Returns a texture name of the celestial body.
        /// </summary>
        /// <param name="variation">
        /// The variation of the texture. 
        /// This input variable is only used if there are any variations available. 
        /// If none are available the default texture (i.e. variation 0) is used.
        /// Variations are available for: Earth, Venus.
        /// </param>
        /// <returns>The name of the texture.</returns>
        public string GetTextureName(int variation)
        {
            string textureName = String.Empty;

            // Filenames of the earth textures by month.
            string[] earthTexturesByMonth = {
                                        "earth_jan",
                                        "earth_feb",
                                        "earth_mar",
                                        "earth_apr",
                                        "earth_may",
                                        "earth_jun",
                                        "earth_jul",
                                        "earth_aug",
                                        "earth_sep",
                                        "earth_oct",
                                        "earth_nov",
                                        "earth_dec"};

            switch (this.celestialObject)
            {
                case CelestialObject.Sun:
                    textureName = "sun";
                    break;
                case CelestialObject.Mercury:
                    textureName = "mercury";
                    break;
                case CelestialObject.Venus:
                    if (variation == 1)
                    {
                        textureName = "venus_surface";
                        break;
                    }
                    textureName = "venus_cloud";
                    break;
                case CelestialObject.Earth:
                    if (variation == 1)
                    {
                        textureName = "earth_map";
                        break;
                    }
                    textureName = earthTexturesByMonth[DateTime.Now.Month - 1];
                    break;
                case CelestialObject.Moon:
                    textureName = "moon";
                    break;
                case CelestialObject.Mars:
                    textureName = "mars";
                    break;
                case CelestialObject.Jupiter:
                    textureName = "jupiter";
                    break;
                case CelestialObject.Io:
                    textureName = "io";
                    break;
                case CelestialObject.Europa:
                    textureName = "europa";
                    break;
                case CelestialObject.Ganymede:
                    textureName = "ganymede";
                    break;
                case CelestialObject.Callisto:
                    textureName = "callisto";
                    break;
                case CelestialObject.Saturn:
                    textureName = "saturn";
                    break;
                case CelestialObject.Neptune:
                    textureName = "neptune";
                    break;
                case CelestialObject.Uranus:
                    textureName = "uranus";
                    break;
                default:
                    textureName = "planet";
                    break;
            }

            return "Textures/" + textureName;
        }
    }
}
