// --------------------------------------------------------
// DoubleSquarePendulumAcceleration.cs
// (A. Schiffler, 2009)
// --------------------------------------------------------

namespace NewGamePhysics.Physics
{
    using System;
    using System.Collections.Generic;
    using System.Text;

    using NewGamePhysics.Mathematics;

    /// <summary>
    /// An object defining the acceleration of a 2D double pendulum consisting of two square
    /// masses connected at corner under the influence of gravity.
    /// </summary>
    public class DoubleSquarePendulumAcceleration : ISecondDerivative
    {
        /// <summary>
        /// Size of squares.
        /// </summary>
        private double l;

        /// <summary>
        /// Mass of one square.
        /// </summary>
        private double m;

        /// <summary>
        /// Gravitational acceleration to apply.
        /// </summary>
        private double g;

        /// <summary>
        /// A constructor of the DoubleRegularPendulum object
        /// </summary>
        public DoubleSquarePendulumAcceleration(double l, double m, double g)
        {
            this.l = l;
            this.m = m;
            this.g = g;
        }

        /// <summary>
        /// Returns size of the squares.
        /// </summary>
        public double L
        {
            get
            {
                return this.l;
            }
        }

        /// <summary>
        /// Returns weight of one individual square.
        /// </summary>
        public double M
        {
            get
            {
                return this.m;
            }
        }

        /// <summary>
        /// Sets the gravitational acceleration.
        /// </summary>
        public double G
        {
            get
            {
                return this.g;
            }

            set
            {
                this.g = value;
            }
        }

        /// <summary>
        /// A method to calculate the vector of accelerations.
        /// </summary>
        /// <param name="t"> Time</param>
        /// <param name="theta"> VectorN of angle of squares in radians.
        /// 0.0 is vertically downwards, counter-clockwise is positive.</param>
        /// <param name="omega"> VectorN of angular velocities of squares</param>
        /// <returns> VectorN of accelerations</returns>
        VectorN ISecondDerivative.GetValue(double t, VectorN theta, VectorN omega)
        {
            double theta1 = theta[0];
            double theta2 = theta[1];
            double omega1 = omega[0];
            double omega2 = omega[1];

            VectorN acceleration = new VectorN(2);
            double sqrt2 = Math.Sqrt(2.0);
            acceleration[0] = (-3 * (8 * g * Math.Sin((Math.PI - 4 * theta1) / 4.0) - 4 * sqrt2 * g * Math.Sin(theta1) +
            6 * g * Math.Cos(Math.PI / 4.0 - theta1 + theta2) * Math.Sin(theta2) + 3 * l * Math.Cos(2 * (theta1 - theta2)) * Math.Pow(omega1, 2) +
            4 * sqrt2 * l * Math.Sin(Math.PI / 4.0 - theta1 + theta2) * Math.Pow(omega2, 2))) / (2.0 * l * (-20 + 9 * Math.Pow(Math.Cos(Math.PI / 4.0 - theta1 + theta2), 2)));

            acceleration[1] = (3 * (g * (3 * Math.Cos(Math.PI / 4.0 - theta1 + theta2) * (Math.Cos(theta1) - 2 * Math.Sin(theta1)) + 5 * sqrt2 * Math.Sin(theta2)) +
            5 * sqrt2 * l * Math.Sin(Math.PI / 4.0 - theta1 + theta2) * Math.Pow(omega1, 2) +
            (3 * l * Math.Cos(2 * (theta1 - theta2)) * Math.Pow(omega2, 2)) / 2.0)) / (l * (-20 + 9 * Math.Pow(Math.Cos(Math.PI / 4.0 - theta1 + theta2), 2)));

            return acceleration;
        }
    }
}
