// --------------------------------------------------------
// FrictionModel.cs
// (A. Schiffler, 2009)
// --------------------------------------------------------

namespace NewGamePhysics.Physics
{
    using System;
    using System.Collections.Generic;
    using System.Text;

    /// <summary>
    /// Enumeration for the various rotational friction models.
    /// </summary>
    public enum RotationalFrictionType
    {
        /// <summary>
        /// No friction.
        /// </summary>
        None,

        /// <summary>
        /// Linear angular velocity dependent friction.
        /// </summary>
        Linear,

        /// <summary>
        /// Non-linear angular velocity dependent friction using the Stribeck model.
        /// </summary>
        Stribeck
    }

    /// <summary>
    /// Model that simulates friction on a hinge
    /// (i.e. rotational friction) by reducing the angular
    /// velocity during the integration step.
    /// </summary>
    public class RotationalFrictionModel
    {
        /// <summary>
        /// Friction type of the current model.
        /// </summary>
        private RotationalFrictionType frictionType;

        /// <summary>
        /// Coefficient of friction for linear case.
        /// </summary>
        private double linearFrictionCoefficient;

        /// <summary>
        /// Static friction cutoff for Stribeck curve.
        /// </summary>
        private double stribeckStaticFriction;


        /// <summary>
        /// Scale for Stribeck friction curve.
        /// </summary>
        private double stribeckFrictionScale;

        /// <summary>
        /// Gets the linear friction coefficient for the 
        /// linear friction model.
        /// </summary>
        public double LinearFrictionCoefficient
        {
            get { return linearFrictionCoefficient; }
        }

        /// <summary>
        /// Gets the static friction coefficient for 
        /// the Stribeck friction model.
        /// </summary>
        public double StribeckStaticFriction
        {
            get { return stribeckStaticFriction; }
        }

        /// <summary>
        /// Gets the scale coefficient for the Stribeck
        /// friction model.
        /// </summary>
        public double StribeckFrictionScale
        {
            get { return stribeckFrictionScale; }
        }

        /// <summary>
        /// Create a friction model object and initialie 
        /// the type to 'None'.
        /// </summary>
        public RotationalFrictionModel()
        {
            this.frictionType = RotationalFrictionType.None;
        }

        /// <summary>
        /// Create a friction model object and initialie 
        /// to the given type with some default parameters.
        /// </summary>
        public RotationalFrictionModel(RotationalFrictionType f)
        {
            this.frictionType = f;
            switch (f)
            {
                case RotationalFrictionType.Linear:
                    this.linearFrictionCoefficient = 0.0005;
                    break;
                case RotationalFrictionType.Stribeck:
                    this.stribeckStaticFriction = 0.0001;
                    this.stribeckFrictionScale = 0.0002;
                    break;
            }
        }

        /// <summary>
        /// Set the None friction type.
        /// </summary>
        public void SetNone()
        {
            this.frictionType = RotationalFrictionType.None;
        }

        /// <summary>
        /// Set the Linear friction type.
        /// </summary>
        public void SetLinear(double linearFrictionCoefficient)
        {
            this.frictionType = RotationalFrictionType.Linear;
            this.linearFrictionCoefficient = linearFrictionCoefficient;
        }

        /// <summary>
        /// Set the Stribeck friction type.
        /// </summary>
        /// <param name="stribeckStaticFriction">The Stribeck static friction coefficient.</param>
        /// <param name="stribeckFrictionScale">The Stribeck friction scale.</param>
        public void SetStribeck(double stribeckStaticFriction, double stribeckFrictionScale)
        {
            this.frictionType = RotationalFrictionType.Stribeck;
            this.stribeckStaticFriction = stribeckStaticFriction;
            this.stribeckFrictionScale = stribeckFrictionScale;
        }

        /// <summary>
        /// Apply friction to a rotational element by changing the rotational speed.
        /// </summary>
        /// <param name="omega">Angular velocity without friction applied.</param>
        /// <returns>Angular velocity after friction was applied.</returns>
        public double ApplyFriction(double omega)
        {
            // Friction difference
            double deltaOmega = 0.0;
            double o = Math.Abs(omega);

            // Adjust by type
            switch (this.frictionType)
            {
                case RotationalFrictionType.None:
                    // No friction
                    deltaOmega = 0.0;
                    break;
                case RotationalFrictionType.Linear:
                    // Linear friction  
                    deltaOmega = linearFrictionCoefficient * o;
                    break;
                case RotationalFrictionType.Stribeck:
                    // Stribeck friction
                    if (o < stribeckStaticFriction)
                    {
                        // Hinge stuck
                        return 0.0;
                    }
                    else
                    {
                        // Calculate friction as curve
                        deltaOmega = stribeckFrictionScale * (o - Math.Exp(-Math.Pow((o / 0.025), 0.3)) + Math.Pow(2.0 * o, (1 - 0.3)));
                    }
                    break;
            }

            // Handle discontinuity around zero
            if (omega > 0.0)
            {
                omega -= deltaOmega;
                if (omega < 0.0)
                {
                    omega = 0.0;
                }
            }
            else
            {
                omega += deltaOmega;
                if (omega > 0.0)
                {
                    omega = 0.0;
                }
            }

            return omega;
        }
    }
}
