﻿// --------------------------------------------------------
// GravityCalculator.cs
// (A. Schiffler, 2009)
// --------------------------------------------------------

namespace NewGamePhysics.Physics
{
    using System;
    using System.Text;

    /// <summary>
    /// Enum specifying various mathematical models for the 
    /// gravitational field of the earth or mars.
    /// </summary>
    /// <remarks>
    /// Sources: wikipedia "Earth's gravity", http://gsc.nrcan.gc.ca
    /// </remarks>
    public enum EarthGravityModel
    {
        /// <summary>
        /// No modelling; fixed average value.
        /// </summary>
        None,

        /// <summary>
        /// No modelling; commonly known value of 9.81.
        /// </summary>
        FolkPhysics,

        /// <summary>
        /// Geodetic Reference System from 1967 (Version provided by IGF).
        /// </summary>
        /// <remarks>
        /// Source: geophysics.ou.edu/solid_earth/notes/potential/igf.htm
        /// </remarks>
        Igf67,

        /// <summary>
        /// The GRS 80 geodetic system.
        /// </summary>
        /// <remarks>
        /// Source: http://bgi.cnes.fr:8110
        /// </remarks>
        Grs80,

        /// <summary>
        /// High-resolution calculation of the 
        /// geodetic reference system (GRS 80) with
        /// height correction.
        /// </summary>
        /// <remarks>
        /// Source: Hofmann-Wellenhof/Moritz, 
        /// Physical Godesy, 2006.
        /// </remarks>
        Grs80Hd,

        /// <summary>
        /// World Geodetic System 84 Ellipsoidal Gravity Formula
        /// </summary>
        /// <remarks>
        /// Source: wikipedia "Earth's gravity"
        /// </remarks>
        Wgs84,

        /// <summary>
        /// High-resolution calculation of the 
        /// World Geodetic System (WGS 84) with height
        /// correction.
        /// </summary>
        /// <remarks>
        /// Source: Hofmann-Wellenhof/Moritz, 
        /// Physical Godesy, 2006.
        /// </remarks>
        Wgs84Hd,

        /// <summary>
        /// Grid interpolator of earths gravity based on data from 
        /// the International Center for Global Earth Models (ICGEM).
        /// </summary>
        /// <remarks>
        /// Source: http://icgem.gfz-potsdam.de/ICGEM/ICGEM.html
        /// </remarks>
        GfcGrid,

        /// <summary>
        /// Satellite only mean Earth Gravity Model from GRACE
        /// </summary>
        /// <remarks>
        /// The GGM03 Mean Earth Gravity Model from GRACE
        /// B. Tapley, J. Ries, S. Bettadpur, D. Chambers, M. Cheng, F. Condi, S. Poole
        /// Eos Trans. AGU 88(52), Fall Meet. Suppl., Abstract G42A-03, 2007.
        /// Source: http://icgem.gfz-potsdam.de/ICGEM/ICGEM.html
        /// </remarks>
        Ggm03sModel,

        /// <summary>
        /// Global combined high-resolution GRACE-based gravity
        /// field model of the GFZ-GRGS cooperation.
        /// </summary>
        /// <remarks>
        /// EIGEN-GL05C - A new global combined high-resolution GRACE-based gravity
        /// field model of the GFZ-GRGS cooperation,
        /// Ch. Foerste (1), F. Flechtner (1), R. Schmidt (1), R. Stubenvoll (1), 
        /// M. Rothacher (1), J. Kusche (1), H. Neumayer (1), R. Biancale (2), 
        /// J.-M. Lemoine (2), F. Barthelmes (1), S. Bruinsma (2), R. Koenig (1) 
        /// and Ul. Meyer (1),        
        /// Geophysical Research Abstracts, Vol. 10, EGU2008-A-03426, 2008.
        /// Source: http://icgem.gfz-potsdam.de/ICGEM/ICGEM.html
        /// </remarks>
        Eigen5cModel,
    }

    /// <summary>
    /// Enum specifying various corrections to the static field.
    /// </summary>
    /// <remarks>
    /// Sources: wikipedia "Earth's gravity", http://gsc.nrcan.gc.ca
    /// </remarks>
    public enum EarthGravityAnomaly
    {
        /// <summary>
        /// No anomaly correction.
        /// </summary>
        None,

        /// <summary>
        /// Free-air anomaly correction for variations in elevation.
        /// Applied for models without height dependency.
        /// Depends on set height.
        /// </summary>
        Fac,

        /// <summary>
        /// Bouguer anomaly correction for the effects of lateral 
        /// variations ("plane plate") in the density of gravity. 
        /// Includes Fac correction when applicable. 
        /// Depends on set height and ground height.
        /// </summary>
        BouguerPlane,

        /// <summary>
        /// Bouguer anomaly correction for a truly "spherical"
        /// Bouguer plate. [Reference: Moritz (1990), p. 235].
        /// Includes Fac correction when applicable. 
        /// Depends on set height and ground height.
        /// </summary>
        BouguerSpherical,
    }


    /// <summary>
    /// Enum specifying various mathematical models for the 
    /// gravitational field of mars.
    /// </summary>
    public enum MarsGravityModel
    {
        /// <summary>
        /// No modelling; fixed average value.
        /// </summary>
        None,

        /// <summary>
        /// Normal Gravity based on ellipsoid.
        /// </summary>
        Normal,

        /// <summary>
        /// High resolution version based on the elliptical model
        /// and a digital map of the Mars gravity anomalies
        /// derived from spherical harmonic coefficients of the NASA GSFC
        /// MROMGM0032 Mars gravity field.
        /// </summary>
        /// <remarks>
        /// http://pds-geosciences.wustl.edu/mro/mro-m-rss-5-sdp-v1/mrors_1xxx/data/rsdmap/
        /// </remarks>
        GgMro,
    }

    /// <summary>
    /// Class to represent gravity.
    /// </summary>
    public class GravityCalculator
    {
        /// <summary>
        /// The gravitational constant in units of m^3 kg^-1 s^-2.
        /// </summary>
        public const double G = 6.67428e-11;

        /// <summary>
        /// Which celestial body are we modelling.
        /// </summary>
        private CelestialObject simulatedBody;

        /// <summary>
        /// Nominally averaged crustal density of the earth
        /// in kg m^-3 used in Bouguer correction.
        /// </summary>
        /// <remarks>
        /// Source: http://gravmag.ou.edu/reduct/reduce.html
        /// </remarks>
        public const double StandardBouguerDensity = 2670.0;

        /// <summary>
        /// The mathematical model to use when correcting
        /// for location when the simulated body is earth.
        /// </summary>
        private EarthGravityModel earthGravityModel;

        /// <summary>
        /// The mathematical model to use when correcting
        /// for location when the simulated body is mars.
        /// </summary>
        private MarsGravityModel marsGravityModel;

        /// <summary>
        /// The anomaly calculation to use to correct for
        /// height when the simulated body is earth.
        /// </summary>
        private EarthGravityAnomaly earthGravityAnomaly;

        /// <summary>
        /// Geodetic Reference System 1980 model.
        /// </summary>
        private GravityEarthGrsModel grsModel;

        /// <summary>
        /// World Geodetic System 1984 model.
        /// </summary>
        private GravityEarthWgsModel wgsModel;

        /// <summary>
        /// Grid-based gravity representation of the earth
        /// based on gridded ICGEM data products.
        /// </summary>
        private GravityEarthGfcGrid gfcGrid;

        /// <summary>
        /// Gfc coefficient base gravity model based on
        /// ICGEM data products.
        /// </summary>
        private GravityEarthGfcModel gfcModel;

        /// <summary>
        /// Grid-based gravity of Mars based on
        /// Mars Reonnaissance Orbiter (MRO) measurements.
        /// </summary>
        private GravityMarsMroModel mroModel;

        /// <summary>
        /// The gravity value for the selected body 
        /// and observer position.
        /// </summary>
        private double value;

        /// <summary>
        /// The latitude of the observer in degrees.
        /// </summary>
        private double latitude;

        /// <summary>
        /// The longitude of the observer in degrees.
        /// </summary>
        private double longitude;

        /// <summary>
        /// The height of the observer in meters above sea level
        /// in meters.
        /// </summary>
        private double elevation;

        /// <summary>
        /// The height of the ground in meters above see level in
        /// meters.
        /// Value set to lie between 0 and elevation.
        /// </summary>
        private double groundElevation;

        /// <summary>
        /// Last calculation - used as to limit rate calculations for slow models.
        /// </summary>
        private DateTime lastCalculation = DateTime.Now;

        /// <summary>
        /// Gets or sets the mathematical model used when correcting
        /// for location when the simulated body is earth.
        /// Sets the simulated body to be earth and updates g value.
        /// </summary>
        public EarthGravityModel EarthGravityModel
        {
            get
            {
                return this.earthGravityModel;
            }

            set
            {
                this.earthGravityModel = value;

                // Set body to mars
                this.simulatedBody = CelestialObject.Earth;

                // Update models
                this.UnloadModelData();
                switch (value)
                {
                    case EarthGravityModel.GfcGrid:
                        this.gfcGrid.LoadGravityMap(@"Physics\Grids\eigen5c.gdf");
                        break;
                    case EarthGravityModel.Ggm03sModel:
                        this.gfcModel.LoadGfcModel(@"Physics\Models\ggm03s.gfc");
                        break;
                    case EarthGravityModel.Eigen5cModel:
                        this.gfcModel.LoadGfcModel(@"Physics\Models\eigen5c.gfc");
                        break;
                }

                UpdateValue();
            }
        }

        /// <summary>
        /// Gets or sets the mathematical model used when correcting
        /// for location when the simulated body is mars.
        /// Sets the simulated body to be mars and updates g value.
        /// </summary>
        public MarsGravityModel MarsGravityModel
        {
            get
            {
                return this.marsGravityModel;
            }

            set
            {
                this.marsGravityModel = value;

                // Set body to mars
                this.simulatedBody = CelestialObject.Mars;

                // Update models
                this.UnloadModelData();
                switch (value)
                {
                    case MarsGravityModel.GgMro:
                        this.mroModel.LoadAnomalyMap(@"Physics\Models\ggmro_095a_anom_085.img");
                        break;
                }

                UpdateValue();
            }
        }

        /// <summary>
        /// Gets or sets the anomaly calculation to use when
        /// correcting for height and/or elevation when 
        /// the simulated body is earth.
        /// Some models ('Hd') already incorporate the 'fac' anomaly.
        /// </summary>
        public EarthGravityAnomaly EarthGravityAnomaly
        {
            get
            {
                return this.earthGravityAnomaly;
            }

            set
            {
                this.earthGravityAnomaly = value;
                if (this.simulatedBody == CelestialObject.Earth)
                {
                    UpdateValue();
                }
            }
        }

        /// <summary>
        /// Gets or sets the gravitational field value in units of m/sec^2.
        /// </summary>
        public double Value
        {
            get
            {
                return this.value;
            }
        }

        /// <summary>
        /// Gets or sets the latitude of the observer in degrees.
        /// Value range -90.0 to 90.0 degrees.
        /// </summary>
        public double Latitude
        {
            get { return this.latitude; }
            set
            {
                if ((value < -90.0) || (value > 90.0))
                {
                    throw new ArgumentOutOfRangeException();
                }

                this.latitude = value;
                UpdateValue();
            }
        }

        /// <summary>
        /// Gets or sets the longitude of the observer in degrees.
        /// Value range 0.0 to 360.0 degrees.
        /// </summary>
        public double Longitude
        {
            get { return this.longitude; }
            set
            {
                if ((value < 0.0) || (value > 360.0))
                {
                    throw new ArgumentOutOfRangeException();
                }

                this.longitude = value;
                UpdateValue();
            }
        }

        /// <summary>
        /// Gets or sets the elevation of the observer in meters above sea level.
        /// Maybe adjusts ground elevation for consistency.
        /// Value range 0 to 1000000 meters.
        /// </summary>
        public double Elevation
        {
            get { return this.elevation; }
            set
            {
                if ((value >= 0.0) &&
                    (value <= 1000000.0))
                {
                    this.elevation = value;

                    // Maybe adjust ground elevation as well
                    if (this.groundElevation > value)
                    {
                        this.groundElevation = value;
                    }

                    UpdateValue();
                }
                else
                {
                    throw new ArgumentOutOfRangeException();
                }
            }
        }

        /// <summary>
        /// Gets or sets the height of the ground in meters above sea level.
        /// Maybe adjusts elevation for consistency.
        /// Value range 0 to 10000 meters.
        /// </summary>
        public double GroundElevation
        {
            get { return this.groundElevation; }
            set
            {
                if ((value >= 0.0) &&
                    (value <= 10000.0))
                {
                    this.groundElevation = value;

                    // Maybe adjust elevation to be consistent
                    if (this.elevation < value)
                    {
                        this.elevation = value;
                    }

                    UpdateValue();
                }
                else
                {
                    throw new ArgumentOutOfRangeException();
                }
            }
        }

        /// <summary>
        /// Constructs the gravity object for earth.
        /// </summary>
        public GravityCalculator()
            : this(CelestialObject.Earth)
        {
        }

        /// <summary>
        /// Constructs the gravity object for a celestial object.
        /// </summary>
        /// <param name="body">The celestial object to represent.</param>
        public GravityCalculator(CelestialObject body)
        {
            // Initialize earth models
            this.wgsModel = new GravityEarthWgsModel();
            this.grsModel = new GravityEarthGrsModel();
            this.gfcGrid = new GravityEarthGfcGrid();
            this.gfcModel = new GravityEarthGfcModel();

            // Initialize mars models
            this.mroModel = new GravityMarsMroModel();

            // Initialize instance state
            this.simulatedBody = body;
            this.earthGravityModel = EarthGravityModel.None;
            this.earthGravityAnomaly = EarthGravityAnomaly.None;
            this.marsGravityModel = MarsGravityModel.None;
            this.latitude = 0.0;
            this.elevation = 0.0;
            this.groundElevation = 0.0;

            // Initialize current value 
            this.UpdateValue();
        }

        /// <summary>
        /// Returns information on the current gravity model for the 
        /// selected celestial object.
        /// </summary>
        /// <returns>Informational text.</returns>
        public string GetModelInfo()
        {
            StringBuilder sb = new StringBuilder();

            switch (this.simulatedBody)
            {
                case CelestialObject.Sun:
                    sb.AppendLine("Model: Fixed value.");
                    sb.AppendLine("Source: http://www.wikipedia.org [Earth's Gravity]");
                    break;
                case CelestialObject.Mercury:
                    sb.AppendLine("Model: Fixed value.");
                    sb.AppendLine("Source: http://www.wikipedia.org [Earth's Gravity]");
                    break;
                case CelestialObject.Venus:
                    sb.AppendLine("Model: Fixed value.");
                    sb.AppendLine("Source: http://www.wikipedia.org [Earth's Gravity]");
                    break;
                case CelestialObject.Earth:
                    switch (this.earthGravityModel)
                    {
                        case EarthGravityModel.None:
                            sb.AppendLine("Model: Fixed value.");
                            sb.AppendLine("Source: http://www.wikipedia.org [Earth's Gravity]");
                            break;
                        case EarthGravityModel.FolkPhysics:
                            sb.AppendLine("Model: Commonly used fixed value.");
                            break;
                        case EarthGravityModel.Igf67:
                            sb.AppendLine("Model: Geodetic Reference System from 1967 (Version provided by IGF).");
                            sb.AppendLine("Source: http://geophysics.ou.edu/solid_earth/notes/potential/igf.htm");
                            break;
                        case EarthGravityModel.Grs80:
                            sb.AppendLine("Model: The GRS 80 geodetic system.");
                            sb.AppendLine("Source: http://bgi.cnes.fr:8110");
                            break;
                        case EarthGravityModel.Grs80Hd:
                            sb.AppendLine("Model: High-res calculation of the geodetic reference system (GRS 80)");
                            sb.AppendLine("Applies height correction.");
                            sb.AppendLine("Source: Hofmann-Wellenhof/Moritz, Physical Godesy, 2006.");
                            break;
                        case EarthGravityModel.Wgs84:
                            sb.AppendLine("Model: World Geodetic System 84 Ellipsoidal Gravity Formula");
                            sb.AppendLine("Source: http://www.wikipedia.org [Earth's Gravity]");
                            break;
                        case EarthGravityModel.Wgs84Hd:
                            sb.AppendLine("Model: High-res calculation of the World Geodetic System (WGS 84)");
                            sb.AppendLine("Applies height correction.");
                            sb.AppendLine("Source: Hofmann-Wellenhof/Moritz, Physical Godesy, 2006.");
                            break;
                        case EarthGravityModel.GfcGrid:
                            sb.AppendLine("Model: Grid interpolator of earths gravity based on data from");
                            sb.AppendLine("the International Center for Global Earth Models (ICGEM).");
                            sb.AppendLine("Grid based on eigen5c model with a 1 degree resolution.");
                            sb.AppendLine("Source: http://icgem.gfz-potsdam.de/ICGEM/ICGEM.html");
                            break;
                        case EarthGravityModel.Ggm03sModel:
                            sb.AppendLine("Model: Satellite only mean Earth Gravity Model from GRACE satellite");
                            sb.AppendLine("GGM03S is an unconstrained global gravity model complete to degree and order 180");
                            sb.AppendLine("accelerometer data. 47 months of GRACE data for the period from January 2003");
                            sb.AppendLine("through December 2006 were used (January 2004 excluded). The GRACE atmosphere-ocean");
                            sb.AppendLine("de-aliasing product (AOD1B) was used, but the mean of the AOD1B for the 47 months");
                            sb.AppendLine("has been restored so that GGM03S represents the true mean field.");
                            sb.AppendLine("Source: http://icgem.gfz-potsdam.de/ICGEM/ICGEM.html");
                            break;
                        case EarthGravityModel.Eigen5cModel:
                            sb.AppendLine("Model: A new global combined high-resolution GRACE-based gravity");
                            sb.AppendLine("field model of the GFZ-GRGS cooperation.");
                            sb.AppendLine("The reference epoch of this gravity field model is 01 Oct 2004");
                            sb.AppendLine("Source: http://icgem.gfz-potsdam.de/ICGEM/ICGEM.html");
                            break;
                    }
                    break;
                case CelestialObject.Moon:
                    sb.AppendLine("Model: Fixed value.");
                    sb.AppendLine("Source: http://www.wikipedia.org [Earth's Gravity]");
                    break;
                case CelestialObject.Mars:
                    switch (this.marsGravityModel)
                    {
                        case MarsGravityModel.None:
                            sb.AppendLine("Model: Fixed value.");
                            sb.AppendLine("Source: http://www.wikipedia.org [Earth's Gravity]");
                            break;
                        case MarsGravityModel.Normal:
                            sb.AppendLine("Model: Ellipsoid.");
                            sb.AppendLine("Algorithm: Hofmann-Wellenhof/Moritz, Physical Godesy, 2006");
                            sb.AppendLine("Source: http://pds-geosciences.wustl.edu/mro/mro-m-rss-5-sdp-v1/mrors_1xxx/data/rsdmap/");
                            break;
                        case MarsGravityModel.GgMro:
                            sb.AppendLine("Model: GgMro 095A NASA GSFC Mars Reconnaissance Orbiter.");
                            sb.AppendLine("A digital map of the Mars gravity anomalies derived from");
                            sb.AppendLine("spherical harmonic coefficients of the NASA GSFC MROMGM0032 Mars gravity field");
                            sb.AppendLine("Source: http://pds-geosciences.wustl.edu/mro/mro-m-rss-5-sdp-v1/mrors_1xxx/data/rsdmap/");
                            break;
                    }
                    break;
                case CelestialObject.Jupiter:
                    sb.AppendLine("Model: Fixed value.");
                    sb.AppendLine("Source: http://www.wikipedia.org [Earth's Gravity]");
                    break;
                case CelestialObject.Io:
                    sb.AppendLine("Model: Fixed value.");
                    sb.AppendLine("Source: http://www.wikipedia.org [Io_(moon)]");
                    break;
                case CelestialObject.Europa:
                    sb.AppendLine("Model: Fixed value.");
                    sb.AppendLine("Source: http://www.wikipedia.org [Europa_(moon)]");
                    break;
                case CelestialObject.Ganymede:
                    sb.AppendLine("Model: Fixed value.");
                    sb.AppendLine("Source: http://www.wikipedia.org [Ganymede_(moon)]");
                    break;
                case CelestialObject.Callisto:
                    sb.AppendLine("Model: Fixed value.");
                    sb.AppendLine("Source: http://www.wikipedia.org [Callisto_(moon)]");
                    break;
                case CelestialObject.Saturn:
                    sb.AppendLine("Model: Fixed value.");
                    sb.AppendLine("Source: http://www.wikipedia.org [Earth's Gravity]");
                    break;
                case CelestialObject.Uranus:
                    sb.AppendLine("Model: Fixed value.");
                    sb.AppendLine("Source: http://www.wikipedia.org [Earth's Gravity]");
                    break;
                case CelestialObject.Neptune:
                    sb.AppendLine("Model: Fixed value.");
                    sb.AppendLine("Source: http://www.wikipedia.org [Earth's Gravity]");
                    break;
            }

            return sb.ToString();
        }

        /// <summary>
        /// Sets the static gravitational constant for various celestial objects.
        /// </summary>
        /// <remarks>
        /// Source: wikipedia "Earth's gravity" and others.
        /// </remarks>
        private void UpdateValue()
        {
            switch (this.simulatedBody)
            {
                case CelestialObject.Sun:
                    value = 274.1;
                    break;

                case CelestialObject.Mercury:
                    value = 3.703;
                    break;

                case CelestialObject.Venus:
                    value = 8.872;
                    break;

                case CelestialObject.Earth:
                    // Flag controlling usage of fac correction 
                    bool facApplicable = false;

                    // Model gravity and calculate a dynamic value
                    double sinLat = Math.Sin(this.latitude * Math.PI / 180.0);
                    double sin2Lat = Math.Sin(2.0 * this.latitude * Math.PI / 180.0);
                    double sinLat2 = sinLat * sinLat;
                    double sin2Lat2 = sin2Lat * sin2Lat;
                    switch (earthGravityModel)
                    {
                        case EarthGravityModel.None:
                            value = 9.80665;
                            break;

                        case EarthGravityModel.FolkPhysics:
                            value = 9.81;
                            break;

                        case EarthGravityModel.Igf67:
                            value = 9.78031846 * (1.0 + 0.0053024 * sinLat2 - 0.0000058 * sin2Lat2);
                            facApplicable = true;
                            break;

                        case EarthGravityModel.Grs80:
                            value = 9.7803267715 * (1.0 + 0.001931851353 * sinLat2) / Math.Sqrt(1.0 - 0.00669438002290 * sinLat2);
                            facApplicable = true;
                            break;

                        case EarthGravityModel.Grs80Hd:
                            value = grsModel.Calculate(this.latitude, this.elevation);
                            break;

                        case EarthGravityModel.Wgs84:
                            value = 9.780326774 * (1.0 + 0.00193185138639 * sinLat2) / Math.Sqrt(1.0 - 0.00669437999013 * sinLat2);
                            facApplicable = true;
                            break;

                        case EarthGravityModel.Wgs84Hd:
                            value = wgsModel.Calculate(this.latitude, this.elevation);
                            break;

                        case EarthGravityModel.GfcGrid:
                            value = gfcGrid.Calculate(this.latitude, this.longitude, this.elevation);
                            break;

                        case EarthGravityModel.Ggm03sModel:
                            value = gfcModel.Calculate(this.latitude, this.longitude, this.elevation);
                            break;

                        case EarthGravityModel.Eigen5cModel:
                            // Use rate limiter (5Hz max.) since calculation is slow 
                            if (DateTime.Now > lastCalculation.AddMilliseconds(200))
                            {
                                value = gfcModel.Calculate(this.latitude, this.longitude, this.elevation);
                                lastCalculation = DateTime.Now;
                            }

                            break;
                    }

                    // Anomaly correction for elevation
                    if ((facApplicable) &&
                        ((earthGravityAnomaly == EarthGravityAnomaly.Fac) ||
                         (earthGravityAnomaly == EarthGravityAnomaly.BouguerPlane)))
                    {
                        value -= (3.086e-6 * this.elevation);

                    }

                    if (earthGravityAnomaly == EarthGravityAnomaly.BouguerPlane)
                    {
                        // Anomaly correction for ground level (plane Bouguer plate
                        value += (2.0 * Math.PI * G * StandardBouguerDensity * this.groundElevation);
                    }
                    else if (earthGravityAnomaly == EarthGravityAnomaly.BouguerSpherical)
                    {
                        // Anomaly correction for ground level (spherical Bouguer plate)
                        value += (4.0 * Math.PI * G * StandardBouguerDensity * this.groundElevation);
                    }

                    break;

                case CelestialObject.Moon:
                    value = 1.625;
                    break;

                case CelestialObject.Mars:
                    // Model gravity and calculate a dynamic value
                    switch (marsGravityModel)
                    {
                        case MarsGravityModel.None:
                            value = 3.728;
                            break;

                        case MarsGravityModel.Normal:
                            value = GravityMarsNormalModel.Calculate(this.latitude);
                            break;

                        case MarsGravityModel.GgMro:
                            value = mroModel.Calculate(this.latitude, this.longitude, this.elevation);
                            break;
                    }

                    break;

                case CelestialObject.Jupiter:
                    value = 25.93;
                    break;

                case CelestialObject.Io:
                    value = 1.796;
                    break;

                case CelestialObject.Europa:
                    value = 1.314;
                    break;

                case CelestialObject.Ganymede:
                    value = 1.428;
                    break;

                case CelestialObject.Callisto:
                    value = 1.235;
                    break;

                case CelestialObject.Saturn:
                    value = 11.19;
                    break;

                case CelestialObject.Uranus:
                    value = 9.01;
                    break;

                case CelestialObject.Neptune:
                    value = 11.28;
                    break;

                case CelestialObject.Pluto:
                    value = 0.61;
                    break;
            }
        }

        /// <summary>
        /// Clear all model maps/grids/coefficients.
        /// </summary>
        private void UnloadModelData()
        {
            this.gfcGrid.UnloadGravityMap();
            this.gfcModel.UnloadGfcModel();
            this.mroModel.UnloadAnomalyMap();
        }
    }
}
