﻿// --------------------------------------------------------
// GravityEarthGfcGrid.cs
// (A. Schiffler, 2009)
// --------------------------------------------------------

namespace NewGamePhysics.Physics
{
    using System;
    using System.IO;

    using NewGamePhysics.Mathematics;
    using NewGamePhysics.Utilities;

    /// <summary>
    /// Grid interpolator of earths gravity based on data from 
    /// the International Center for Global Earth Models (ICGEM).
    /// Reference: http://icgem.gfz-potsdam.de/ICGEM/ICGEM.html
    /// </summary>
    public class GravityEarthGfcGrid
    {
        /// <summary>
        /// Minimum latitude of grid.
        /// </summary>
        private const int LatMin = -90;

        /// <summary>
        /// Maximum latitude of grid.
        /// </summary>
        private const int LatMax =  90;

        /// <summary>
        /// Minimum longitude of grid.
        /// </summary>
        private const int LonMin = 0;

        /// <summary>
        /// Maximum longitude of grid.
        /// </summary>
        private const int LonMax = 360;

        /// <summary>
        /// Gravity value grid at lat [-90,90] and long [0,360] 
        /// in one degree steps.
        /// </summary>
        private double[,] grid;

        /// <summary>
        /// Number of gridpoints that were loaded.
        /// </summary>
        private int size = 0;

        /// <summary>
        /// Constructs an uninitialized gravity earth object.
        /// </summary>
        public GravityEarthGfcGrid()
        {
        }

        /// <summary>
        /// Constructs a gravity earth object and initializes it by
        /// loading a model.
        /// </summary>
        /// <param name="modelFilename">The gdf grid file to load.</param>
        public GravityEarthGfcGrid(string gridFilename)
        {
            this.LoadGravityMap(gridFilename);
        }

        /// <summary>
        /// Gets the number of gridpoints loaded.
        /// </summary>
        public int Size
        {
            get { return this.size; }
        }

        /// <summary>
        /// Calculate the acceleration g from grid using bicubic 
        /// interpolation.
        /// Requires a map file to be loaded.
        /// </summary>
        /// <param name="lat">Latitude (deg) in the range [-90,90]</param>
        /// <param name="lon">Longitude (deg) in the range [0,360]</param>
        /// <param name="h">Height (m) above sealevel in the range [0,100000].</param>
        /// <returns>Acceleration g.</returns>
        public double Calculate(double lat, double lon, double h)
        {
            if (null == this.grid)
            {
                throw new ApplicationException(
                    "Gravity map not initialized. Load one first.");
            }

            if ((lat < -90.0) || (lat > 90.0))
            {
                throw new ArgumentOutOfRangeException("lat");
            }

            if ((lon < 0.0) || (lon > 360.0))
            {
                throw new ArgumentOutOfRangeException("lon");
            }

            if ((h < 0.0) || (h > 100000.0))
            {
                throw new ArgumentOutOfRangeException("h");
            }

            // Calculate index position
            int intLat = (int)Math.Truncate(lat);
            int intLon = (int)Math.Truncate(lon);

            // Create bicubic interpolator matrix
            double[,] G = new double[4, 4];
            for (int i = -1; i <= 2; i++)
            {
                for (int j = -1; j <= 2; j++)
                {
                    int indexLat = LatWrap(intLat + i) - LatMin;
                    int indexLon = LonWrap(intLon + j) - LonMin;                    
                    G[i+1,j+1] = grid[indexLat, indexLon];
                }
            }

            // Interpolate over grid
            Bicubic bicubic = new Bicubic(G);
            double x = Math.Abs(lat - (double)intLat);
            double y = lon - (double)intLon;
            double g = bicubic.Calc(x,y);

            // Scale from mgal units
            g *= 1e-5;

            // Adjust for elevation
            g -= (3.086e-6 * h);

            return g;
        }

        /// <summary>
        /// Populate the gravity value map from an external file.
        /// Assumes the grid is in .gdf format from over the value range
        /// lat [-90,90] and long [0,360] in one degree steps.
        /// </summary>
        /// <param name="gridFilename">The grid file to load.</param>
        public void LoadGravityMap(string gridFilename)
        {
            // Create new grid 
            size = 0;
            grid = new double[(LatMax - LatMin + 1), (LonMax - LonMin + 1)];

            // Prep data reader
            Scanf scanf = new Scanf();
            FileStream fileStream = null;
            StreamReader streamReader = null;
            try
            {
                fileStream = new FileStream(gridFilename, FileMode.Open);
                streamReader = new StreamReader(fileStream);
                string line;
                while ((line = streamReader.ReadLine()) != null)
                {
                    object[] o = scanf.Scan(line, " %lf %lf %lf");
                    if ((null != o) && (o.Length == 3))
                    {
                        int lon = (int)Math.Truncate((double)o[0]);
                        int lat = (int)Math.Truncate((double)o[1]);
                        double g = (double)o[2];

                        // Check if the grid indices are in the right range
                        if ((lat < LatMin) || (lat > LatMax) || (lon < LonMin) || (lon > LonMax))
                        {
                            throw new Exception("Invalid data: " + line); 
                        }

                        int latIndex = lat - LatMin;
                        int lonIndex = lon - LonMin;

                        // Make sure we don't double write a grid point 
                        if (grid[latIndex, lonIndex] != 0.0)
                        {
                            throw new Exception("Double grid value: " + line);
                        }

                        // Store value in grid                        
                        grid[latIndex, lonIndex] = g;
                        size++;
                    }
                }
            }
            finally
            {
                if (null != streamReader)
                {
                    streamReader.Close();
                }
                if (null != fileStream)
                {
                    fileStream.Close();
                }
            }
        }

        /// <summary>
        /// Dispose of the current gravity map.
        /// </summary>
        public void UnloadGravityMap()
        {
            this.grid = null;
        }

        /// <summary>
        /// Wrap latitude value into range [-90,90].
        /// </summary>
        /// <param name="lat">Integer latitude value.</param>
        /// <returns>Wrapped latitude value.</returns>
        private int LatWrap(int lat)
        {
            while (lat < LatMin) 
            {
                lat =+ (LatMax - LatMin);
            } 

            while (lat > LatMax)
            {
                lat -= (LatMax - LatMin);
            }
 
            return lat;
        }

        /// <summary>
        /// Wrap longitude value into range [0,360].
        /// </summary>
        /// <param name="lat">Integer longitude value.</param>
        /// <returns>Wrapped longitude value.</returns>
        private int LonWrap(int lon)
        {
            while (lon < LonMin)
            {
                lon += (LonMax - LonMin);
            }

            while (lon > LonMax)
            {
                lon -= (LonMax - LonMin);
            }

            return lon;
        }
    }
}
