﻿// --------------------------------------------------------
// GravityEarthWgsModel.cs
// (A. Schiffler, 2009)
// --------------------------------------------------------

namespace NewGamePhysics.Physics
{
    using System;

    /// <summary>
    /// World Geodetic System 1984 (WGS 1984)
    /// model of earth's gravity g.
    /// Reference: 
    /// Hofmann-Wellenhof, B., Moritz, H. (2006), 
    /// Physical Geodesy (2nd ed.), Springer-Verlag Wien.
    /// </summary>
    public class GravityEarthWgsModel
    {
        /// <summary>
        /// Semimajor axis of the ellipsoid in m.
        /// </summary>
        public static double a = 6378137.0;

        /// <summary>
        /// Geocentric gravitational constant of the 
        /// earth including the atmosphere in m^3*s^-2.
        /// </summary>
        public static double GM = 3986004.418e8;

        /// <summary>
        /// Angular velocity of the earth in rad*s^-1.
        /// </summary>
        public static double omega = 7292115.0e-11;

        /// <summary>
        /// Semiminor axis of the ellipsoid in m.
        /// </summary>
        public static double b = 6356752.3142;

        /// <summary>
        /// Linear eccentricity in m.
        /// </summary>
        public static double E = 5.2185400842339e5;

        /// <summary>
        /// Polar radius of curvature in m.
        /// </summary>
        public static double c = 6399593.6258;

        /// <summary>
        /// First eccentricity squared.
        /// </summary>
        public static double e2 = 6.69437999014e-3;

        /// <summary>
        /// Second eccenctricity squared.
        /// </summary>
        public static double ep2 = 6.73949674228e-3;

        /// <summary>
        /// Flattening.
        /// </summary>
        public static double f = 1.0/298.257223563;

        /// <summary>
        /// Normal potential of the ellipsoid in m^2*s^-2;
        /// </summary>
        public static double U0 = 62636851.7146;

        /// <summary>
        /// m = omega^2*a^2*b/GM
        /// </summary>
        public static double m = 0.00344978650684;

        /// <summary>
        /// Normal gravity at the equator in m*s^-2.
        /// </summary>
        public static double ya = 9.7803253359;

        /// <summary>
        /// Normal gravity at the pole in m*s^-2.
        /// </summary>
        public static double yb = 9.8321849378;

        /// <summary>
        /// Mass of earth in kg.
        /// </summary>
        public static double M = 5.9733328e24;

        /// <summary>
        /// Calculate the gravity outside the ellipsoid of the
        /// earth at a particular latitude in degrees and 
        /// height above the surface in m. Obtained from the 
        /// second order expansion of a small height.
        /// </summary>
        /// <param name="latitude">The latitude in degrees [-90,90]</param>
        /// <param name="height">The height above the ellipsoid 
        /// in meters [0,1000000].</param>
        /// <returns>The gravity in m*s^-2.</returns>
        public double Calculate(double latitude, double height)
        {
            if ((latitude < -90.0) || (latitude > 90.0))
            {
                throw new ArgumentOutOfRangeException("latitude");
            }

            if ((height < 0.0) || (height > 1000000.0))
            {
                throw new ArgumentOutOfRangeException("height");
            }

            // precalculations
            double phi = Math.PI * latitude / 180.0;
            double sl = Math.Sin(phi);
            double s2l = Math.Sin(2.0 * phi);
            double sl2 = sl * sl;
            double s2l2 = s2l * s2l;

            // calculate gravity flattening
            double fs = (yb - ya) / ya;
            double f4 = -0.5 * f * f + 2.5 * f * m;

            // calculate latitude dependent gravity
            double y = ya * (1.0 + fs * sl2 - 0.25 * f4 * s2l2);

            // calculate the height dependent gravity
            double yh = y * (1.0 - 2.0 / a * (1.0 + f + m - 2.0 * f * sl2) * height + 3.0 / (a * a) * (height * height));

            // result
            return yh;           
        }
    }
}
