// --------------------------------------------------------
// SimpleEntropyCollector.cs
// (A. Schiffler, 2009)
// --------------------------------------------------------

namespace NewGamePhysics.Physics
{
    using System;
    using System.Text;

    using NewGamePhysics.Utilities;

    /// <summary>
    /// Collects entropy from submitted floating point values of physical
    /// processes (measurements or simulations fed by physical processes).
    /// Uses the bitcount of the difference of submitted floating point 
    /// values and a von Neumann unbiaser.
    /// The submissions of the values will yield usable entropy if:
    /// * if the value changes sufficiently much between submissions 
    /// * the submissions are independent from the value generation
    /// * either the value source or the submission trigger is based on
    ///   a real physical random process (i.e. uncorrelated user trigger)
    /// </summary>
    public class SimpleEntropyCollector
    {
        /// <summary>
        /// Size of bit pool.
        /// </summary>
        private int poolSize;

        /// <summary>
        /// Array of collected bits.
        /// </summary>
        private bool[] bitPool;

        /// <summary>
        /// Number of valid entropy bits in array.
        /// </summary>
        private int bitsInPool;

        /// <summary>
        /// Number of values collected.
        /// Used in value differencing.
        /// </summary>
        private int numValues;

        /// <summary>
        /// Cached last submitted value.
        /// </summary>
        private double lastValue;

        /// <summary>
        /// Number of generated bits.
        /// Used in vonNeumann corrector.
        /// </summary>
        private int numBits;

        /// <summary>
        /// Cached last generated bit.
        /// </summary>
        private bool lastBit;

        /// <summary>
        /// Helper class to count the bits in a ulong.
        /// </summary>
        private BitCount bitCount;

        /// <summary>
        /// Default constructor.
        /// </summary>
        public SimpleEntropyCollector()
        {
            // Initialize bitcount helper
            bitCount = new BitCount();

            // Initialize collector state
            this.Reset();
        }

        /// <summary>
        /// Gets the current number of valid entropy bits in a the pool.
        /// </summary>
        public int BitsInPool
        {
            get { return bitsInPool; }
        }

        /// <summary>
        /// Reset collector to initial state.
        /// </summary>
        public void Reset()
        {
            // Reset pool and value trackers
            this.poolSize = 1024;
            this.bitsInPool = 0;
            this.bitPool = new bool[poolSize];
            this.numValues = 0;
            this.numBits = 0;
        }

        /// <summary>
        /// Adds a float value to the collector.
        /// </summary>
        /// <param name="value">Value to add.</param>
        public void AddValue(double value)
        {
            if (numValues == 1)
            {
                // Different between values
                double delta = this.lastValue - value;

                // Translate the double into a 64 bit long.
                long deltaBits = BitConverter.DoubleToInt64Bits(delta);

                // Get bitcount
                int totalBits = this.bitCount.FastBitcount(deltaBits);

                // Generate this bit
                bool newBit = (bool)((totalBits & 1) == 1);

                // von Neumann corrector to unbias bitstream
                if (numBits == 1)
                {
                    // Check for 10: add a 1
                    if ((!newBit) && (this.lastBit))
                    {
                        // Use LSB of bitcount
                        if (this.bitsInPool < 1024)
                        {
                            this.bitPool[this.bitsInPool] = true;
                            this.bitsInPool++;
                        }
                    }
                    // Check for 01: add a 0
                    else if ((newBit) && (!this.lastBit))
                    {
                        // Use LSB of bitcount
                        if (this.bitsInPool < 1024)
                        {
                            this.bitPool[this.bitsInPool] = false;
                            this.bitsInPool++;
                        }
                    }
                    else
                    {
                        // Discard 00 and 11 patterns
                    }
                    // Reset corrector counter
                    this.numBits = 0;
                }
                else
                {
                    // Store bit
                    this.lastBit = newBit;
                    this.numBits++;
                }

                // Reset value count to we cache the next submission
                this.numValues = 0;
            }
            else
            {
                // Just cache value for next call
                this.lastValue = value;

                // Set value count so we process the next submission
                this.numValues = 1;
            }
        }

        /// <summary>
        /// Return the current entropy collector bit-pool as 
        /// string of 0 and 1 characters.
        /// </summary>
        /// <returns>The current bit pool as string.</returns>
        public override string ToString()
        {
            StringBuilder sb = new StringBuilder();

            if (this.bitsInPool > 0)
            {
                for (int i = bitsInPool - 1; i >= 0; i--)
                {
                    if (bitPool[i])
                    {
                        sb.Append("1");
                    }
                    else
                    {
                        sb.Append("0");
                    }
                }
            }

            return sb.ToString();
        }

    }
}
