//-----------------------------------------------------------------------------
// InputState.cs
// (Based on XNA Community Game Platform Demo, Microsoft Corp., 2008)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.StateManager
{
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Input;

    using NewGamePhysics.Utilities;

    /// <summary>
    /// Helper for reading input from keyboard and gamepad. This class tracks both
    /// the current and previous state of both input devices, and implements query
    /// methods for high level input actions such as "move up through the menu"
    /// or "pause the game".
    /// </summary>
    public class InputState
    {
        #region Fields

        public const int MaxInputs = 4;

        public readonly KeyboardState[] CurrentKeyboardStates;
        public readonly GamePadState[] CurrentGamePadStates;

        public readonly KeyboardState[] LastKeyboardStates;
        public readonly GamePadState[] LastGamePadStates;

        public readonly bool[] GamePadWasConnected;

        /// <summary>
        /// DirectX driven gamepads of game
        /// </summary>
        private Gamepads gamepads;

        #endregion

        #region Initialization

        /// <summary>
        /// Constructs a new input state.
        /// </summary>
        public InputState()
        {
            CurrentKeyboardStates = new KeyboardState[MaxInputs];
            CurrentGamePadStates = new GamePadState[MaxInputs];

            LastKeyboardStates = new KeyboardState[MaxInputs];
            LastGamePadStates = new GamePadState[MaxInputs];

            GamePadWasConnected = new bool[MaxInputs];

            // New DirectX input object
            this.gamepads = new Gamepads();
        }


        #endregion

        #region Public Methods

        /// <summary>
        /// Reads the latest state of the keyboard and gamepad.
        /// </summary>
        public void Update()
        {
            for (int i = 0; i < MaxInputs; i++)
            {
                LastKeyboardStates[i] = CurrentKeyboardStates[i];
                LastGamePadStates[i] = CurrentGamePadStates[i];

                CurrentKeyboardStates[i] = Keyboard.GetState((PlayerIndex)i);
                CurrentGamePadStates[i] = GamePad.GetState((PlayerIndex)i);

                // Keep track of whether a gamepad has ever been
                // connected, so we can detect if it is unplugged.
                if (CurrentGamePadStates[i].IsConnected)
                {
                    GamePadWasConnected[i] = true;
                }
            }

            // Check gamepad
            if ((this.gamepads != null) && (this.gamepads.Items.Count > 0))
            {
            }
        }

        /// <summary>
        /// Helper for checking if a key is currently pressed. The
        /// controllingPlayer parameter specifies which player to read input for.
        /// If this is null, it will accept input from any player. When a keydown
        /// is detected, the output playerIndex reports which player pressed it.
        /// </summary>
        public bool IsKeyDown(Keys key, PlayerIndex? controllingPlayer,
                                            out PlayerIndex playerIndex)
        {
            if (controllingPlayer.HasValue)
            {
                // Read input from the specified player.
                playerIndex = controllingPlayer.Value;

                int i = (int)playerIndex;

                return (CurrentKeyboardStates[i].IsKeyDown(key));
            }
            else
            {
                // Accept input from any player.
                return (IsKeyDown(key, PlayerIndex.One, out playerIndex) ||
                        IsKeyDown(key, PlayerIndex.Two, out playerIndex) ||
                        IsKeyDown(key, PlayerIndex.Three, out playerIndex) ||
                        IsKeyDown(key, PlayerIndex.Four, out playerIndex));
            }
        }

        /// <summary>
        /// Helper for checking if a key was newly pressed during this update. The
        /// controllingPlayer parameter specifies which player to read input for.
        /// If this is null, it will accept input from any player. When a keypress
        /// is detected, the output playerIndex reports which player pressed it.
        /// </summary>
        public bool IsNewKeyPress(Keys key, PlayerIndex? controllingPlayer,
                                            out PlayerIndex playerIndex)
        {
            if (controllingPlayer.HasValue)
            {
                // Read input from the specified player.
                playerIndex = controllingPlayer.Value;

                int i = (int)playerIndex;

                return (CurrentKeyboardStates[i].IsKeyDown(key) &&
                        LastKeyboardStates[i].IsKeyUp(key));
            }
            else
            {
                // Accept input from any player.
                return (IsNewKeyPress(key, PlayerIndex.One, out playerIndex) ||
                        IsNewKeyPress(key, PlayerIndex.Two, out playerIndex) ||
                        IsNewKeyPress(key, PlayerIndex.Three, out playerIndex) ||
                        IsNewKeyPress(key, PlayerIndex.Four, out playerIndex));
            }
        }
        /// <summary>
        /// Helper for checking if a button is currently pressed.
        /// The controllingPlayer parameter specifies which player to read input for.
        /// If this is null, it will accept input from any player. When a button down
        /// is detected, the output playerIndex reports which player pressed it.
        /// </summary>
        public bool IsButtonDown(Buttons button, PlayerIndex? controllingPlayer,
                                                     out PlayerIndex playerIndex)
        {
            if (controllingPlayer.HasValue)
            {
                // Read input from the specified player.
                playerIndex = controllingPlayer.Value;

                int i = (int)playerIndex;

                return (CurrentGamePadStates[i].IsButtonDown(button));
            }
            else
            {
                // Accept input from any player.
                return (IsButtonDown(button, PlayerIndex.One, out playerIndex) ||
                        IsButtonDown(button, PlayerIndex.Two, out playerIndex) ||
                        IsButtonDown(button, PlayerIndex.Three, out playerIndex) ||
                        IsButtonDown(button, PlayerIndex.Four, out playerIndex));
            }
        }

        /// <summary>
        /// Helper for checking if a button was newly pressed during this update.
        /// The controllingPlayer parameter specifies which player to read input for.
        /// If this is null, it will accept input from any player. When a button press
        /// is detected, the output playerIndex reports which player pressed it.
        /// </summary>
        public bool IsNewButtonPress(Buttons button, PlayerIndex? controllingPlayer,
                                                     out PlayerIndex playerIndex)
        {
            if (controllingPlayer.HasValue)
            {
                // Read input from the specified player.
                playerIndex = controllingPlayer.Value;

                int i = (int)playerIndex;

                return (CurrentGamePadStates[i].IsButtonDown(button) &&
                        LastGamePadStates[i].IsButtonUp(button));
            }
            else
            {
                // Accept input from any player.
                return (IsNewButtonPress(button, PlayerIndex.One, out playerIndex) ||
                        IsNewButtonPress(button, PlayerIndex.Two, out playerIndex) ||
                        IsNewButtonPress(button, PlayerIndex.Three, out playerIndex) ||
                        IsNewButtonPress(button, PlayerIndex.Four, out playerIndex));
            }
        }

        /// <summary>
        /// Checks for a "menu select" input action.
        /// Enables on Space, Enter, A and Start.
        /// The controllingPlayer parameter specifies which player to read input for.
        /// If this is null, it will accept input from any player. When the action
        /// is detected, the output playerIndex reports which player pressed it.
        /// </summary>
        public bool IsMenuSelect(PlayerIndex? controllingPlayer,
                                 out PlayerIndex playerIndex)
        {
            return IsNewKeyPress(Keys.Space, controllingPlayer, out playerIndex) ||
                   IsNewKeyPress(Keys.Enter, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.A, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.Start, controllingPlayer, out playerIndex);
        }


        /// <summary>
        /// Checks for a "menu cancel" input action.
        /// Enables on Escape, B and Back.
        /// The controllingPlayer parameter specifies which player to read input for.
        /// If this is null, it will accept input from any player. When the action
        /// is detected, the output playerIndex reports which player pressed it.
        /// </summary>
        public bool IsMenuCancel(PlayerIndex? controllingPlayer,
                                 out PlayerIndex playerIndex)
        {
            return IsNewKeyPress(Keys.Escape, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.B, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.Back, controllingPlayer, out playerIndex);
        }

        /// <summary>
        /// Checks for an "up" input state.
        /// Enables on Up (cursor, dpad and thumbstick).
        /// The controllingPlayer parameter specifies which player to read
        /// input for. If this is null, it will accept input from any player.
        /// </summary>
        public bool IsInputUp(PlayerIndex? controllingPlayer)
        {
            PlayerIndex playerIndex;

            return IsKeyDown(Keys.Up, controllingPlayer, out playerIndex) ||
                   IsButtonDown(Buttons.DPadUp, controllingPlayer, out playerIndex) ||
                   IsButtonDown(Buttons.LeftThumbstickUp, controllingPlayer, out playerIndex);
        }

        /// <summary>
        /// Checks for a "up" input action (key up toggle).
        /// Enables on Up (cursor, dpad and thumbstick).
        /// The controllingPlayer parameter specifies which player to read
        /// input for. If this is null, it will accept input from any player.
        /// </summary>
        public bool IsNewInputUp(PlayerIndex? controllingPlayer)
        {
            PlayerIndex playerIndex;

            return IsNewKeyPress(Keys.Up, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.DPadUp, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.LeftThumbstickUp, controllingPlayer, out playerIndex);
        }

        /// <summary>
        /// Checks if an "down" input state.
        /// Enables on Down (cursor, dpad and thumbstick).
        /// The controllingPlayer parameter specifies which player to read
        /// input for. If this is null, it will accept input from any player.
        /// </summary>
        public bool IsInputDown(PlayerIndex? controllingPlayer)
        {
            PlayerIndex playerIndex;

            return IsKeyDown(Keys.Down, controllingPlayer, out playerIndex) ||
                   IsButtonDown(Buttons.DPadDown, controllingPlayer, out playerIndex) ||
                   IsButtonDown(Buttons.LeftThumbstickDown, controllingPlayer, out playerIndex);
        }

        /// <summary>
        /// Checks for a "down" input action (key down toggle).
        /// Enables on Down (cursor, dpad and thumbstick).
        /// The controllingPlayer parameter specifies which player to read
        /// input for. If this is null, it will accept input from any player.
        /// </summary>
        public bool IsNewInputDown(PlayerIndex? controllingPlayer)
        {
            PlayerIndex playerIndex;

            return IsNewKeyPress(Keys.Down, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.DPadDown, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.LeftThumbstickDown, controllingPlayer, out playerIndex);
        }

        /// <summary>
        /// Checks for a "left" input state.
        /// Enables on Left (cursor, dpad and thumbstick).
        /// The controllingPlayer parameter specifies which player to read
        /// input for. If this is null, it will accept input from any player.
        /// </summary>
        public bool IsInputLeft(PlayerIndex? controllingPlayer)
        {
            PlayerIndex playerIndex;

            return IsKeyDown(Keys.Left, controllingPlayer, out playerIndex) ||
                   IsButtonDown(Buttons.DPadLeft, controllingPlayer, out playerIndex) ||
                   IsButtonDown(Buttons.LeftThumbstickLeft, controllingPlayer, out playerIndex);
        }

        /// <summary>
        /// Checks for a "left" input action (key left toggle).
        /// Enables on Left (cursor, dpad and thumbstick).
        /// The controllingPlayer parameter specifies which player to read
        /// input for. If this is null, it will accept input from any player.
        /// </summary>
        public bool IsNewInputLeft(PlayerIndex? controllingPlayer)
        {
            PlayerIndex playerIndex;

            return IsNewKeyPress(Keys.Left, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.DPadLeft, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.LeftThumbstickLeft, controllingPlayer, out playerIndex);
        }

        /// <summary>
        /// Checks for a "right" input state.
        /// Enables on Right (cursor, dpad and thumbstick).
        /// The controllingPlayer parameter specifies which player to read
        /// input for. If this is null, it will accept input from any player.
        /// </summary>
        public bool IsInputRight(PlayerIndex? controllingPlayer)
        {
            PlayerIndex playerIndex;

            return IsKeyDown(Keys.Right, controllingPlayer, out playerIndex) ||
                   IsButtonDown(Buttons.DPadRight, controllingPlayer, out playerIndex) ||
                   IsButtonDown(Buttons.LeftThumbstickRight, controllingPlayer, out playerIndex);
        }

        /// <summary>
        /// Checks for a "right" input action (key right toggle).
        /// Enables on Right (cursor, dpad and thumbstick).
        /// The controllingPlayer parameter specifies which player to read
        /// input for. If this is null, it will accept input from any player.
        /// </summary>
        public bool IsNewInputRight(PlayerIndex? controllingPlayer)
        {
            PlayerIndex playerIndex;

            return IsNewKeyPress(Keys.Right, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.DPadRight, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.LeftThumbstickRight, controllingPlayer, out playerIndex);
        }

        /// <summary>
        /// Checks for a "cancel" input action (key toggle).
        /// Enables on Escape, Back and Start.
        /// The controllingPlayer parameter specifies which player to read
        /// input for. If this is null, it will accept input from any player.
        /// </summary>
        public bool IsInputCancel(PlayerIndex? controllingPlayer)
        {
            PlayerIndex playerIndex;

            return IsNewKeyPress(Keys.Escape, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.Back, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.Start, controllingPlayer, out playerIndex);
        }

        /// <summary>
        /// Checks for a "select" input action. 
        /// Enables on Space, Enter, A and RightTrigger.
        /// The controllingPlayer parameter specifies which player to read
        /// input for. If this is null, it will accept input from any player.
        /// </summary>
        public bool IsInputSelect(PlayerIndex? controllingPlayer)
        {
            PlayerIndex playerIndex;

            return IsNewKeyPress(Keys.Space, controllingPlayer, out playerIndex) ||
                   IsNewKeyPress(Keys.Enter, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.A, controllingPlayer, out playerIndex) ||
                   IsNewButtonPress(Buttons.RightTrigger, controllingPlayer, out playerIndex);
        }

        /// <summary>
        /// Checks for a disconnection occurence (i.e. was connected, but is not anymore).
        /// The controllingPlayer parameter specifies which player to read
        /// input for. If this is null, it will accept input from any player.
        /// </summary>
        public bool IsDisconnected(PlayerIndex? controllingPlayer)
        {
            if (controllingPlayer.HasValue)
            {
                // Check state from the specified player.
                PlayerIndex playerIndex = controllingPlayer.Value;
                int i = (int)playerIndex;
                return (!this.CurrentGamePadStates[i].IsConnected && this.GamePadWasConnected[i]);
            }
            else
            {
                // Check state from any player.
                return  (!this.CurrentGamePadStates[(int)PlayerIndex.One].IsConnected && this.GamePadWasConnected[(int)PlayerIndex.One]) ||
                        (!this.CurrentGamePadStates[(int)PlayerIndex.Two].IsConnected && this.GamePadWasConnected[(int)PlayerIndex.Two]) ||
                        (!this.CurrentGamePadStates[(int)PlayerIndex.Three].IsConnected && this.GamePadWasConnected[(int)PlayerIndex.Three]) ||
                        (!this.CurrentGamePadStates[(int)PlayerIndex.Four].IsConnected && this.GamePadWasConnected[(int)PlayerIndex.Four]);                    
            }
        }
        #endregion
    }
}
