//-----------------------------------------------------------------------------
// BitCount.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Utilities
{
    using System;

    /// <summary>
    /// Counts the bits in integer values.
    /// </summary>
    class BitCount
    {
        /// <summary>
        /// Cached lookup table which stores 64K bytes worth of
        /// bitcounts, each entry holding the number of bits of its index. 
        /// Must be initialized in the constructor.
        /// </summary>
        static byte[] bitcountCache;

        /// <summary>
        /// Constructor which generates a bit count lookup array.
        /// </summary>
        public BitCount()
        {
            // Initialize cache
            bitcountCache = new byte[65536];
            for (int i = 0; i < 65536; i++)
            {
                bitcountCache[i] = SparseBitcount(i);
            }
        }

        /// <summary>
        /// Find the number of bits in the parameter ushort. 
        /// </summary>
        /// <param name="testThis">The number to count the bits of.</param>
        /// <returns>The number of bits in the parameter.</returns>
        public int FastBitcount(ushort testThis)
        {
            return bitcountCache[testThis];
        }

        /// <summary>
        /// Find the number of bits in the parameter short. 
        /// </summary>
        /// <param name="testThis">The number to count the bits of.</param>
        /// <returns>The number of bits in the parameter.</returns>
        public int FastBitcount(short testThis)
        {
            return FastBitcount((ushort)testThis);
        }

        /// <summary>
        /// Find the number of bits in the parameter uint. 
        /// </summary>
        /// <param name="testThis">The number to count the bits of.</param>
        /// <returns>The number of bits in the parameter.</returns>
        public int FastBitcount(uint testThis)
        {
            return bitcountCache[(testThis >> (0 * 16)) & 65535] +
                   bitcountCache[(testThis >> (1 * 16)) & 65535];
        }

        /// <summary>
        /// Find the number of bits in the parameter int. 
        /// </summary>
        /// <param name="testThis">The number to count the bits of.</param>
        /// <returns>The number of bits in the parameter.</returns>
        public int FastBitcount(int testThis)
        {
            return FastBitcount((uint)testThis);
        }

        /// <summary>
        /// Find the number of bits in the parameter long. 
        /// </summary>
        /// <param name="testThis">The number to count the bits of.</param>
        /// <returns>The number of bits in the parameter.</returns>
        public int FastBitcount(ulong testThis)
        {
            return bitcountCache[(testThis >> (0 * 16)) & 65535L] +
                   bitcountCache[(testThis >> (1 * 16)) & 65535L] +
                   bitcountCache[(testThis >> (2 * 16)) & 65535L] +
                   bitcountCache[(testThis >> (3 * 16)) & 65535L];
        }

        /// <summary>
        /// Find the number of bits in the parameter int. 
        /// </summary>
        /// <param name="testThis">The number to count the bits of.</param>
        /// <returns>The number of bits in the parameter.</returns>
        public int FastBitcount(long testThis)
        {
            return FastBitcount((ulong)testThis);
        }

        /// <summary>
        /// Find the number of bits in the parameter double. 
        /// </summary>
        /// <param name="testThis">The number to count the bits of.</param>
        /// <returns>The number of bits in the parameter.</returns>
        public int FastBitcount(double testThis)
        {
            // Translate the double into a 64 bit long.
            long testThisBits = BitConverter.DoubleToInt64Bits(testThis);
            return FastBitcount((ulong)testThisBits);
        }

        /// <summary>
        /// Finds the number of bits in the parameter. Must return the same result
        /// as the fast bitcount. Don't call in performance-critical code, but can be
        /// used to precompute the bit count array.
        /// Reference:
        /// http://infolab.stanford.edu/~manku/bitcount/bitcount.html).
        /// </summary>
        static private byte SparseBitcount(int testThis)
        {
            int count = 0;
            while (testThis != 0)
            {
                count++;
                testThis &= (testThis - 1);
            }

            return (byte)count;
        }
    }
}
