﻿//-----------------------------------------------------------------------------
// HighResolutionTimer.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Utilities
{
    using System;
    using System.ComponentModel;
    using System.Runtime.InteropServices;
    using System.Threading;

    /// <summary>
    /// Class which provides access to the high-resolution system timer via
    /// Interop calls into the Win32 API.
    /// </summary>
    public class HighResolutionTimer
    {
        /// <summary>
        /// Query the performance counter.
        /// </summary>
        /// <param name="lpCounter">The counter parameter to store the retrieved value in.</param>
        /// <returns>A flag indicating if the counter was set in the parameter.</returns>
        [DllImport("Kernel32.dll")]
        private static extern bool QueryPerformanceCounter(
            out long lpCounter);

        /// <summary>
        /// Query the performance frequency.
        /// </summary>
        /// <param name="lpFrequency">The frequency parameter to store the retrieved value in.</param>
        /// <returns>A flag indicating if the frequency was set in the parameter.</returns>
        [DllImport("Kernel32.dll")]
        private static extern bool QueryPerformanceFrequency(
            out long lpFrequency);

        /// <summary>
        /// Queries the high-performance counter.
        /// </summary>
        /// <returns>Current count number.</returns>
        public static long Counter()
        {
            long counter;

            if (!QueryPerformanceCounter(out counter))
            {
                throw new Win32Exception("Error calling QueryPerformanceCounter()");
            }

            return counter;
        }

        /// <summary>
        /// Queries the high resolution-timer frequency.
        /// </summary>
        /// <returns>A frequency in Hz.</returns>
        public static long Frequency()
        {
            long frequency;

            if (!QueryPerformanceFrequency(out frequency))
            {
                throw new Win32Exception("Error calling QueryPerformanceFrequency()");
            }

            return frequency;
        }

        /// <summary>
        /// Number of seconds since the system was started.
        /// </summary>
        /// <returns>A relative time in seconds.</returns>
        public static double Seconds()
        {
            return (double)Counter() / (double)Frequency();
        }
    }
}
