﻿//-----------------------------------------------------------------------------
// IntersectionTest.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Utilities
{
    using System;
    using Microsoft.Xna.Framework;

    /// <summary>
    /// Class implementing various intersection test algorithms.
    /// </summary>
    public static class IntersectionTest
    {
        /// <summary>
        /// 2D Check if a circle is within another circle
        /// If the circles touch on a single point, they are 
        /// considered intersecting.
        /// </summary>
        /// <param name="centerA">Center of first circle.</param>
        /// <param name="radiusA">Radius of first circle. Test fails if negative.</param>
        /// <param name="centerB">Center of second circle.</param>
        /// <param name="radiusB">Radius of second circle. Test fails if negative.</param>
        /// <returns>
        /// A boolean flag indicating if the circles intersect.
        /// </returns>
        public static bool CircleInCircle2D(Vector2 centerA, double radiusA, Vector2 centerB, double radiusB)
        {
            if ((radiusA < 0.0) || (radiusB < 0.0))
            {
                return false;
            }

            double dX = (double)(centerB.X - centerA.X);
            double dY = (double)(centerB.Y - centerA.Y);
            double d = Math.Sqrt(dX * dX + dY * dY);

            // circles do not intersect if they are too far apart
            if (d > (radiusA + radiusB))
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// 2D Check if a point lies withing a polygon.
        /// </summary>
        /// <param name="polygonVertex">The points of the polygon.</param>
        /// <param name="testVertex">The point to check.</param>
        /// <returns>
        /// A boolean flag indicating if the test vertex
        /// is inside the polygon.
        /// </returns>
        public static bool PointInPolygon2D(Vector2[] polygonVertex, Vector2 testVertex)
        {
            bool c = false;
            int nvert = polygonVertex.Length;
            if (nvert > 2)
            {
                int i, j;
                for (i = 0, j = nvert - 1; i < nvert; j = i++)
                {
                    if (((polygonVertex[i].Y > testVertex.Y) != (polygonVertex[j].Y > testVertex.Y)) &&
                     (testVertex.X < (polygonVertex[j].X - polygonVertex[i].X) *
                     (testVertex.Y - polygonVertex[i].Y) /
                     (polygonVertex[j].Y - polygonVertex[i].Y) + polygonVertex[i].X))
                    {
                        c = !c;
                    }
                }
            }

            return c;
        }

        /// <summary>
        /// 2D Triangle-Triangle Intersection Test.
        /// If the triangles touch in a single point or a line, they are 
        /// considered intersecting.
        /// </summary>
        /// <param name="p1">First vertex of first triangle.</param>
        /// <param name="q1">Second vertex of first triangle.</param>
        /// <param name="r1">Third vertex of first triangle.</param>
        /// <param name="p2">First vertex of second triangle.</param>
        /// <param name="q2">Second vertex of second triangle.</param>
        /// <param name="r2">Third vertex of second triangle.</param>
        /// <returns>A boolean flag indicating if the two triangles
        /// intersect.</returns>
        /// <remarks>
        /// Based on the C implementation of algorithms for                
        /// performing two and three-dimensional triangle-triangle intersection test
        /// "Fast and Robust Triangle-Triangle Overlap Test Using Orientation Predicates"  
        /// by P. Guigue - O. Devillers, July 2002 - December 2003                                                
        /// Journal of Graphics Tools, 8(1), 2003  
        /// </remarks>
        public static bool TriangleTriangleIntersect2D(
            Vector2 p1,
            Vector2 q1,
            Vector2 r1,
            Vector2 p2,
            Vector2 q2,
            Vector2 r2)
        {
            if (Orientation2D(p1, q1, r1) < 0.0f)
            {
                if (Orientation2D(p2, q2, r2) < 0.0f)
                {
                    return CcwTriTriIntersection2D(p1, r1, q1, p2, r2, q2);
                }
                else
                {
                    return CcwTriTriIntersection2D(p1, r1, q1, p2, q2, r2);
                }
            }
            else
            {
                if (Orientation2D(p2, q2, r2) < 0.0f)
                {
                    return CcwTriTriIntersection2D(p1, q1, r1, p2, r2, q2);
                }
                else
                {
                    return CcwTriTriIntersection2D(p1, q1, r1, p2, q2, r2);
                }
            }
        }


        /// <summary>
        /// 2D test for counterclockwise tri-tri intersection.
        /// </summary>
        /// <param name="p1">First vertex of first triangle.</param>
        /// <param name="q1">Second vertex of first triangle.</param>
        /// <param name="r1">Third vertex of first triangle.</param>
        /// <param name="p2">First vertex of second triangle.</param>
        /// <param name="q2">Second vertex of second triangle.</param>
        /// <param name="r2">Third vertex of second triangle.</param>
        /// <returns>True if intersecting</returns>
        private static bool CcwTriTriIntersection2D(
            Vector2 p1,
            Vector2 q1,
            Vector2 r1,
            Vector2 p2,
            Vector2 q2,
            Vector2 r2)
        {
            if (Orientation2D(p2, q2, p1) >= 0.0f)
            {
                if (Orientation2D(q2, r2, p1) >= 0.0f)
                {
                    if (Orientation2D(r2, p2, p1) >= 0.0f)
                    {
                        return true;
                    }
                    else
                    {
                        return TestEdgeIntersection(p1, q1, r1, p2, q2, r2);
                    }
                }
                else
                {
                    if (Orientation2D(r2, p2, p1) >= 0.0f)
                    {
                        return TestEdgeIntersection(p1, q1, r1, r2, p2, q2);
                    }
                    else
                    {
                        return TestVertexIntersection2D(p1, q1, r1, p2, q2, r2);
                    }
                }
            }
            else
            {
                if (Orientation2D(q2, r2, p1) >= 0.0f)
                {
                    if (Orientation2D(r2, p2, p1) >= 0.0f)
                    {
                        return TestEdgeIntersection(p1, q1, r1, q2, r2, p2);
                    }
                    else
                    {
                        return TestVertexIntersection2D(p1, q1, r1, q2, r2, p2);
                    }
                }
                else
                {
                    return TestVertexIntersection2D(p1, q1, r1, r2, p2, q2);
                }
            }
        }

        /// <summary>
        /// Calculates the orientation of triangle.
        /// </summary>
        /// <param name="p">First vertex of triangle.</param>
        /// <param name="q">Second vertex of triangle.</param>
        /// <param name="r">Third vertex of triangle.</param>
        /// <returns>Orientation value</returns>
        private static float Orientation2D(Vector2 p, Vector2 q, Vector2 r)
        {
            return ((p.X - r.X) * (q.Y - r.Y) - (p.Y - r.Y) * (q.X - r.X));
        }

        /// <summary>
        /// 2D test for tri-tri edge intersection.
        /// </summary>
        /// <param name="p1">First vertex of first triangle.</param>
        /// <param name="q1">Second vertex of first triangle.</param>
        /// <param name="r1">Third vertex of first triangle.</param>
        /// <param name="p2">First vertex of second triangle.</param>
        /// <param name="q2">Second vertex of second triangle.</param>
        /// <param name="r2">Third vertex of second triangle.</param>
        /// <returns>True if intersecting</returns>
        private static bool TestEdgeIntersection(
            Vector2 p1, 
            Vector2 q1, 
            Vector2 r1, 
            Vector2 p2, 
            Vector2 q2, 
            Vector2 r2)
        {
            if (Orientation2D(r2, p2, q1) >= 0.0f)
            {
                if (Orientation2D(p1, p2, q1) >= 0.0f)
                {
                    if (Orientation2D(p1, q1, r2) >= 0.0f)
                    {
                        return true;
                    }
                    else
                    {
                        return false;
                    }
                }
                else
                {
                    if (Orientation2D(q1, r1, p2) >= 0.0f)
                    {
                        if (Orientation2D(r1, p1, p2) >= 0.0f)
                        {
                            return true;
                        }
                        else
                        {
                            return false;
                        }
                    }
                    else
                    {
                        return false;
                    }
                }
            }
            else
            {
                if (Orientation2D(r2, p2, r1) >= 0.0f)
                {
                    if (Orientation2D(p1, p2, r1) >= 0.0f)
                    {
                        if (Orientation2D(p1, r1, r2) >= 0.0f)
                        {
                            return true;
                        }
                        else
                        {
                            if (Orientation2D(q1, r1, r2) >= 0.0f)
                            {
                                return true;
                            }
                            else
                            {
                                return false;
                            }
                        }
                    }
                    else
                    {
                        return false;
                    }
                }
                else
                {
                    return false;
                }
            }
        }

        /// <summary>
        /// 2D test for tri-tri vertex intersection.
        /// </summary>
        /// <param name="p1">First vertex of first triangle.</param>
        /// <param name="q1">Second vertex of first triangle.</param>
        /// <param name="r1">Third vertex of first triangle.</param>
        /// <param name="p2">First vertex of second triangle.</param>
        /// <param name="q2">Second vertex of second triangle.</param>
        /// <param name="r2">Third vertex of second triangle.</param>
        /// <returns>True if intersecting</returns>
        private static bool TestVertexIntersection2D(
            Vector2 p1,
            Vector2 q1,
            Vector2 r1,
            Vector2 p2,
            Vector2 q2,
            Vector2 r2)
        {
            if (Orientation2D(r2, p2, q1) >= 0.0f)
            {
                if (Orientation2D(r2, q2, q1) <= 0.0f)
                {
                    if (Orientation2D(p1, p2, q1) > 0.0f)
                    {
                        if (Orientation2D(p1, q2, q1) <= 0.0f)
                        {
                            return true;
                        }
                        else
                        {
                            return false;
                        }
                    }
                    else
                    {
                        if (Orientation2D(p1, p2, r1) >= 0.0f)
                        {
                            if (Orientation2D(q1, r1, p2) >= 0.0f)
                            {
                                return true;
                            }
                            else
                            {
                                return false;
                            }
                        }
                        else
                        {
                            return false;
                        }
                    }
                }
                else
                {
                    if (Orientation2D(p1, q2, q1) <= 0.0f)
                    {
                        if (Orientation2D(r2, q2, r1) <= 0.0f)
                        {
                            if (Orientation2D(q1, r1, q2) >= 0.0f)
                            {
                                return true;
                            }
                            else
                            {
                                return false;
                            }
                        }
                        else
                        {
                            return false;
                        }
                    }
                    else
                    {
                        return false;
                    }
                }
            }
            else
            {
                if (Orientation2D(r2, p2, r1) >= 0.0f)
                {
                    if (Orientation2D(q1, r1, r2) >= 0.0f)
                    {
                        if (Orientation2D(p1, p2, r1) >= 0.0f)
                        {
                            return true;
                        }
                        else
                        {
                            return false;
                        }
                    }
                    else
                    {
                        if (Orientation2D(q1, r1, q2) >= 0.0f)
                        {
                            if (Orientation2D(r2, r1, q2) >= 0.0f)
                            {
                                return true;
                            }
                            else
                            {
                                return false;
                            }
                        }
                        else
                        {
                            return false;
                        }
                    }
                }
                else
                {
                    return false;
                }
            }
        }
    }
}