//-----------------------------------------------------------------------------
// PrimitiveBatch.cs
// (Based on XNA Community Game Platform Demo, Microsoft Corp., 2008)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Utilities
{
    using System;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Graphics;

    /// <summary>
    /// Class to handle efficient rendering of lines, points,
    /// and triangles to the screen. 
    /// </summary>
    public class PrimitiveBatch : IDisposable
    {
        #region Constants and Fields

        /// <summary>
        /// this constant controls how large the vertices buffer is. Larger buffers will
        /// require flushing less often, which can increase performance. However, having
        /// buffer that is unnecessarily large will waste memory.
        /// 
        /// </summary>
        const int DefaultBufferSize = 1024;

        /// <summary>
        /// a block of vertices that calling AddVertex will fill. Flush will draw using
        /// this array, and will determine how many primitives to draw from
        /// positionInBuffer.
        /// </summary>
        VertexPositionColor[] vertices = new VertexPositionColor[DefaultBufferSize];

        /// <summary>
        /// keeps track of how many vertices have been added. this value increases until
        /// we run out of space in the buffer, at which time Flush is automatically
        /// called.
        /// </summary>
        int positionInBuffer = 0;

        /// <summary>
        /// the vertex declaration that will be set on the device for drawing. this is 
        /// created automatically using VertexPositionColor's vertex elements.
        /// </summary>
        VertexDeclaration vertexDeclaration;

        /// <summary>
        /// a basic effect, which contains the shaders that we will use to draw our
        /// primitives.
        /// </summary>
        BasicEffect basicEffect;

        /// <summary>
        /// the device that we will issue draw calls to.
        /// </summary>
        GraphicsDevice device;

        /// <summary>
        /// this value is set by Begin, and is the type of primitives that we are
        /// drawing.
        /// </summary>
        PrimitiveType primitiveType;

        /// <summary>
        /// how many verts does each of these primitives take up? points are 1,
        /// lines are 2, and triangles are 3.
        /// </summary>
        int numVertsPerPrimitive;

        /// <summary>
        /// hasBegun is flipped to true once Begin is called, and is used to make
        /// sure users don't call End before Begin is called.
        /// </summary>
        bool hasBegun = false;

        /// <summary>
        /// Flag tracking object disposal.
        /// </summary>
        bool isDisposed = false;

        #endregion

        /// <summary>
        /// Creates a new PrimitiveBatch instance.
        /// </summary>
        public PrimitiveBatch(GraphicsDevice graphicsDevice)
        {
            if (graphicsDevice == null)
            {
                throw new ArgumentNullException("graphicsDevice");
            }

            device = graphicsDevice;

            // create a vertex declaration, which tells the graphics card what kind of
            // data to expect during a draw call. We're drawing using
            // VertexPositionColors, so we'll use those vertex elements.
            vertexDeclaration = new VertexDeclaration(graphicsDevice,
                VertexPositionColor.VertexElements);

            // set up a new basic effect, and enable vertex colors.
            basicEffect = new BasicEffect(graphicsDevice, null);
            basicEffect.VertexColorEnabled = true;

            // projection uses CreateOrthographicOffCenter to create 2d projection
            // matrix with 0,0 in the upper left.
            basicEffect.Projection = Matrix.CreateOrthographicOffCenter
                (0, graphicsDevice.Viewport.Width,
                graphicsDevice.Viewport.Height, 0,
                0, 1);
        }

        /// <summary>
        /// Dispose interface.
        /// </summary>
        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// Dispose interface with flag.
        /// </summary>
        /// <param name="disposing">Flag indicating if we are disposing the object.</param>
        protected virtual void Dispose(bool disposing)
        {
            if (disposing && !isDisposed)
            {
                if (vertexDeclaration != null)
                {
                    vertexDeclaration.Dispose();
                }

                if (basicEffect != null)
                {
                    basicEffect.Dispose();
                }

                isDisposed = true;
            }
        }

        /// <summary>
        /// Begin is called to tell the PrimitiveBatch what kind of primitives will be
        /// drawn, and to prepare the graphics card to render those primitives.
        /// </summary>
        /// <param name="primitiveType">The primitive types to draw.</param>
        public void Begin(PrimitiveType primitiveType)
        {
            if (hasBegun)
            {
                throw new InvalidOperationException
                    ("End must be called before Begin can be called again.");
            }

            // these three types reuse vertices, so we can't flush properly without more
            // complex logic.
            if (primitiveType == PrimitiveType.LineStrip ||
                primitiveType == PrimitiveType.TriangleFan ||
                primitiveType == PrimitiveType.TriangleStrip)
            {
                throw new NotSupportedException
                    ("The specified primitiveType is not supported by PrimitiveBatch.");
            }

            this.primitiveType = primitiveType;

            // how many verts will each of these primitives require?
            this.numVertsPerPrimitive = NumVertsPerPrimitive(primitiveType);

            // prepare the graphics device for drawing by setting the vertex declaration
            // and telling our basic effect to begin.
            device.VertexDeclaration = vertexDeclaration;
            basicEffect.Begin();
            basicEffect.CurrentTechnique.Passes[0].Begin();

            // flip the error checking boolean. It's now ok to call AddVertex, Flush,
            // and End.
            hasBegun = true;
        }

        /// <summary>
        /// AddVertex is called to add another vertex to be rendered. To draw a point,
        /// AddVertex must be called once. for lines, twice, and for triangles 3 times.
        /// this function can only be called once begin has been called.
        /// if there is not enough room in the vertices buffer, Flush is called
        /// automatically.
        /// </summary>
        /// <param name="vertex">The vertex to add.</param>
        /// <param name="color">The color of this vertex.</param>
        public void AddVertex(Vector2 vertex, Color color)
        {
            if (!hasBegun)
            {
                throw new InvalidOperationException
                    ("Begin must be called before AddVertex can be called.");
            }

            // are we starting a new primitive? if so, and there will not be enough room
            // for a whole primitive, flush.
            bool newPrimitive = ((positionInBuffer % numVertsPerPrimitive) == 0);

            if (newPrimitive &&
                (positionInBuffer + numVertsPerPrimitive) >= vertices.Length)
            {
                Flush();
            }

            // once we know there's enough room, set the vertex in the buffer,
            // and increase position.
            vertices[positionInBuffer].Position = new Vector3(vertex, 0);
            vertices[positionInBuffer].Color = color;

            positionInBuffer++;
        }

        /// <summary>
        /// End is called once all the primitives have been drawn using AddVertex.
        /// it will call Flush to actually submit the draw call to the graphics card, and
        /// then tell the basic effect to end.
        /// </summary>
        public void End()
        {
            if (!hasBegun)
            {
                throw new InvalidOperationException
                    ("Begin must be called before End can be called.");
            }

            // Draw whatever the user wanted us to draw
            Flush();

            // and then tell basic effect that we're done.
            basicEffect.CurrentTechnique.Passes[0].End();
            basicEffect.End();
            hasBegun = false;
        }

        /// <summary>
        /// Flush is called to issue the draw call to the graphics card. Once the draw
        /// call is made, positionInBuffer is reset, so that AddVertex can start over
        /// at the beginning. End will call this to draw the primitives that the user
        /// requested, and AddVertex will call this if there is not enough room in the
        /// buffer.
        /// </summary>
        private void Flush()
        {
            if (!hasBegun)
            {
                throw new InvalidOperationException
                    ("Begin must be called before Flush can be called.");
            }

            // no work to do
            if (positionInBuffer == 0)
            {
                return;
            }

            // how many primitives will we draw?
            int primitiveCount = positionInBuffer / numVertsPerPrimitive;

            // submit the draw call to the graphics card
            device.DrawUserPrimitives<VertexPositionColor>(primitiveType, vertices, 0,
                primitiveCount);

            // now that we've drawn, it's ok to reset positionInBuffer back to zero,
            // and write over any vertices that may have been set previously.
            positionInBuffer = 0;
        }

        #region Helper functions

        /// <summary>
        /// NumVertsPerPrimitive is a boring helper function that tells how many vertices
        /// it will take to draw each kind of primitive.
        /// </summary>
        /// <param name="primitive">The primitive type.</param>
        /// <returns>The number of vertices per primitive.</returns>
        static private int NumVertsPerPrimitive(PrimitiveType primitive)
        {
            int numVertsPerPrimitive;
            switch (primitive)
            {
                case PrimitiveType.PointList:
                    numVertsPerPrimitive = 1;
                    break;
                case PrimitiveType.LineList:
                    numVertsPerPrimitive = 2;
                    break;
                case PrimitiveType.TriangleList:
                    numVertsPerPrimitive = 3;
                    break;
                default:
                    throw new InvalidOperationException("Primitive is not valid");
            }
            return numVertsPerPrimitive;
        }

        #endregion

    }
}
