//-----------------------------------------------------------------------------
// GamePlayer.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace PendulumGame
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Text;

    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Graphics;

    using NewGamePhysics.Utilities;
    using NewGamePhysics.Mathematics;
    using NewGamePhysics.Physics;
    using NewGamePhysics.PhysicalElements;

    /// <summary>
    /// Represents a player of the pendulum game.
    /// </summary>
    public class GamePlayer
    {
        /// <summary>
        /// Size of the action intensity tracking queue.
        /// </summary>
        private const int ActionIntensityQueueSize = 32;

        /// <summary>
        /// The points the player collected.
        /// </summary>
        private int points;

        /// <summary>
        /// The currently selected indicator.
        /// </summary>
        private int selectedIndicator;

        /// <summary>
        /// Which hinge is the actionable marker.
        /// </summary>
        private DoublePendulumHinges actionMarker;

        /// <summary>
        /// Intensity of the action on the hinge.
        /// Value range -1.0 to 1.0.
        /// </summary>
        private double actionIntensity;

        /// <summary>
        /// High-pass filter for activity detection.
        /// </summary>
        private ButterworthFilter filter;

        /// <summary>
        /// Rising-Edge detector for activity detection.
        /// </summary>
        private EdgeDetector detector;

        /// <summary>
        /// Last set of action intensity samples.
        /// </summary>
        private Queue<double> actionIntensityQueue;

        /// <summary>
        /// Default constructor.
        /// </summary>
        public GamePlayer()
        {
            this.selectedIndicator = 0;
            this.points = 0;
            this.actionIntensityQueue = 
                new Queue<double>(ActionIntensityQueueSize);
            this.filter =
                new ButterworthFilter(ButterworthFilterType.HighPass, 100, 20);
            this.detector =
                new EdgeDetector(EdgeDetectionType.Rising, 0.4);
        }

        #region Properties

        /// <summary>
        /// Gets or sets the indicator selected for display
        /// </summary>
        public int SelectedIndicator
        {
            get { return this.selectedIndicator; }
            set { this.selectedIndicator = value; }
        }

        /// <summary>
        /// Gets or sets the players score.
        /// </summary>
        public int Points
        {
            get { 
                return this.points; 
            }
            set 
            {
                this.points = value;
            }
        }

        /// <summary>
        /// Gets or sets the hinge to which force is applied.
        /// </summary>
        public DoublePendulumHinges ActionMarker
        {
            get { 
                return this.actionMarker; 
            }
            set { 
                this.actionMarker = value; 
            }
        }

        /// <summary>
        /// Gets or sets the action intensity which is applied.
        /// </summary>
        public double ActionIntensity
        {
            get { 
                return this.actionIntensity; 
            }
            set {
                if (value < -1.0)
                {
                    this.actionIntensity = -1.0;
                }
                else if (value > 1.0)
                {
                    this.actionIntensity = 1.0;
                }
                else
                {
                    this.actionIntensity = value;
                }
            }
        }

        #endregion

        /// <summary>
        /// Tracks action intensity by pushing it onto the stack of values.
        /// </summary>
        public void TrackActionIntensity()
        {
            // Keep queue the same size by dropping off items
            if (this.actionIntensityQueue.Count == ActionIntensityQueueSize)
            {
                this.actionIntensityQueue.Dequeue();
            }

            // Add new sample
            this.actionIntensityQueue.Enqueue(this.actionIntensity);
        }

        /// <summary>
        /// Returns flag indicating if the action intensity has changed 
        /// sufficiently.
        /// </summary>
        /// <returns>True or false depending on a detected change in the 
        /// action intensity.</returns>
        public bool ActionIntensityChanged()
        {
            // Only process if we have enough samples in our queue
            if (this.actionIntensityQueue.Count == ActionIntensityQueueSize)
            {
                double[] samples = this.actionIntensityQueue.ToArray();

                // High-pass filter samples
                double[] filteredSamples = this.filter.Calculate(samples);

                // Edge-detect samples
                double[] detectedSamples = this.detector.Calculate(samples);

                // Return result by checking the last sample
                if (detectedSamples[ActionIntensityQueueSize - 1] > 0.0)
                {
                    return true;
                }
            }

            return false;
        }
    }
}
