#region File Description
//-----------------------------------------------------------------------------
// PendulumBackgroundScreen.cs
//-----------------------------------------------------------------------------
#endregion

namespace PendulumGame
{
    using System;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Media;
    using Microsoft.Xna.Framework.Input;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Utilities;
    
    /// <summary>
    /// The background screen sits behind all the other menu screens.
    /// It draws a background image that remains fixed in place regardless
    /// of whatever transitions the screens on top of it may be doing.
    /// </summary>
    class PendulumBackgroundScreen : GameScreen
    {
        #region Fields

        /// <summary>
        /// The content manager associated with this screen.
        /// </summary>
        ContentManager content;

        /// <summary>
        /// The background texture.
        /// </summary>
        Texture2D backgroundTexture;

        /// <summary>
        /// The frame texture.
        /// </summary>
        Texture2D frameTexture;

        /// <summary>
        /// The video texture.
        /// </summary>
        Texture2D videoTexture;

        /// <summary>
        /// The video file.
        /// </summary>
        Video videoFile;

        /// <summary>
        /// Flag indicating that the video playback was started.
        /// </summary>
        bool videoStarted = false;

        #endregion

        #region Initialization


        /// <summary>
        /// Constructor.
        /// </summary>
        public PendulumBackgroundScreen()
        {
            TransitionOnTime = TimeSpan.FromSeconds(0.5);
            TransitionOffTime = TimeSpan.FromSeconds(0.5);
        }


        /// <summary>
        /// Loads graphics content for this screen. The background texture is quite
        /// big, so we use our own local ContentManager to load it. This allows us
        /// to unload before going from the menus into the game itself, wheras if we
        /// used the shared ContentManager provided by the Game class, the content
        /// would remain loaded forever.
        /// </summary>
        public override void LoadContent()
        {
            if (content == null)
            {
                content = new ContentManager(ScreenManager.Game.Services, "Content");
            }

            backgroundTexture = content.Load<Texture2D>(@"Sprites\background");
            frameTexture = content.Load<Texture2D>(@"Sprites\frame");
            videoFile = content.Load<Video>(@"Videos\DoublePendulumTitle-CBR");
        }

        /// <summary>
        /// Unloads graphics content for this screen.
        /// </summary>
        public override void UnloadContent()
        {
            ScreenManager.VideoPlayer.Stop();
            content.Unload();
        }


        #endregion

        #region Update and Draw


        /// <summary>
        /// Updates the background screen. Unlike most screens, this should not
        /// transition off even if it has been covered by another screen: it is
        /// supposed to be covered, after all! This overload forces the
        /// coveredByOtherScreen parameter to false in order to stop the base
        /// Update method wanting to transition off.
        /// </summary>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus,
                                                       bool coveredByOtherScreen)
        {
            base.Update(gameTime, otherScreenHasFocus, false);
            if (!videoStarted)
            {
                ScreenManager.VideoPlayer.Play(videoFile);
                videoStarted = true;
            }
            else
            {
                if (ScreenManager.VideoPlayer.State == MediaState.Playing)
                {
                    videoTexture = ScreenManager.VideoPlayer.GetTexture();
                }
                else if (ScreenManager.VideoPlayer.State == MediaState.Stopped)
                {
                    videoStarted = false;
                }            
            }
        }

        /// <summary>
        /// Draws the background screen.
        /// </summary>
        public override void Draw(GameTime gameTime)
        {
            SpriteBatch spriteBatch = ScreenManager.SpriteBatch;
            Viewport viewport = ScreenManager.GraphicsDevice.Viewport;
            Rectangle fullscreen = new Rectangle(0, 0, viewport.Width, viewport.Height);
            byte transparency = TransitionAlpha;
            byte intensity = 192;

            spriteBatch.Begin(SpriteBlendMode.None);
            spriteBatch.Draw(
                backgroundTexture, 
                fullscreen,
                Color.White);
            spriteBatch.End();

            spriteBatch.Begin(SpriteBlendMode.AlphaBlend);
            if (videoTexture != null)
            {
                spriteBatch.Draw(
                    videoTexture,
                    fullscreen,
                    new Color(intensity, intensity, intensity, transparency));
            }

            spriteBatch.Draw(
                frameTexture,
                fullscreen,
                Color.White);
            spriteBatch.End();

            // If the game is transitioning on or off, fade it out to black.
            if (TransitionPosition > 0)
            {
                ScreenManager.FadeBackBufferToBlack(255 - TransitionAlpha);
            }
        }

        #endregion
    }
}
