﻿//-----------------------------------------------------------------------------
// PendulumGameOverScreen.cs
//-----------------------------------------------------------------------------

namespace PendulumGame
{
    using System;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Media;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Utilities;

    /// <summary>
    /// Shows the GameOver screen.
    /// </summary>
    class PendulumGameOverScreen : MenuScreen
    {
        #region fields

        /// <summary>
        /// The content manager associated with this screen.
        /// </summary>
        private ContentManager content;

        /// <summary>
        /// The video file.
        /// </summary>
        private Video videoFile;

        /// <summary>
        /// Flag indicating that the video playback was started.
        /// </summary>
        private bool videoStarted = false;

        /// <summary>
        /// The chromakey effect pixel shader.
        /// </summary>
        private Effect effect;

        /// <summary>
        /// The reference color for chroma keying.
        /// </summary>
        private Color referenceColor;

        /// <summary>
        /// The frame texture.
        /// </summary>
        Texture2D frameTexture;

        #endregion

        #region Initialization

        /// <summary>
        /// Constructor.
        /// </summary>
        public PendulumGameOverScreen()
            : base("Game Over")
        {
            // Flag that there is no need for the game to transition
            // off when the pause menu is on top of it.
            IsPopup = true;

            // Create our menu entries.
            MenuEntry playGameMenuEntry = new MenuEntry("Play Again");
            MenuEntry returnMenuEntry = new MenuEntry("Return to Main Menu");

            // Hook up menu event handlers.
            playGameMenuEntry.Selected += PlayGameMenuEntrySelected;
            returnMenuEntry.Selected += OnContinue;

            // Add entries to the menu.
            MenuEntries.Add(playGameMenuEntry);
            MenuEntries.Add(returnMenuEntry);
        }

        /// <summary>
        /// Loads graphics content for this screen. The background texture is quite
        /// big, so we use our own local ContentManager to load it. This allows us
        /// to unload before going from the menus into the game itself, wheras if we
        /// used the shared ContentManager provided by the Game class, the content
        /// would remain loaded forever.
        /// </summary>
        public override void LoadContent()
        {
            if (content == null)
            {
                content = new ContentManager(ScreenManager.Game.Services, "Content");
            }

            frameTexture = content.Load<Texture2D>(@"Sprites\frame");

            // Pick random video
            int videoNumber = PendulumGame.State.PhysicalRandomNumberGenerator.Next(0, 12);

            videoFile = content.Load<Video>(@"Videos\Pendulum" + videoNumber);
            effect = content.Load<Effect>(@"Effects\ColorChromaKey");

            // Determine chromakey reference color by averaging
            // the color over the reference image
            Texture2D backgroundReference =
                content.Load<Texture2D>(@"Textures\PendulumBackgroundReference");
            Color[] colors = TextureHelpers.TextureToColors(backgroundReference);
            ulong rSum = 0;
            ulong gSum = 0;
            ulong bSum = 0;
            ulong num = (ulong)colors.Length;
            foreach (Color color in colors)
            {
                rSum += (ulong)color.R;
                gSum += (ulong)color.G;
                bSum += (ulong)color.B;
            }

            this.referenceColor = new Color(
                (byte)(rSum / num),
                (byte)(gSum / num),
                (byte)(bSum / num));
        }

        /// <summary>
        /// Unloads graphics content for this screen.
        /// </summary>
        public override void UnloadContent()
        {
            content.Unload();
        }

        #endregion

        #region Handle Input

        /// <summary>
        /// Event handler for Continue menu.
        /// </summary>
        void OnContinue(object sender, PlayerIndexEventArgs e)
        {
            PendulumLoadingScreen.Load(ScreenManager, false, null, new PendulumBackgroundScreen(),
                                                           new PendulumMainMenuScreen());
        }

        /// <summary>
        /// Event handler for when the Play Game menu entry is selected.
        /// </summary>
        void PlayGameMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            PendulumLoadingScreen.Load(ScreenManager, true, e.PlayerIndex,
                               new PendulumGameplayScreen());
        }

        #endregion

        #region Draw

        /// <summary>
        /// Updates the background screen. Unlike most screens, this should not
        /// transition off even if it has been covered by another screen: it is
        /// supposed to be covered, after all! This overload forces the
        /// coveredByOtherScreen parameter to false in order to stop the base
        /// Update method wanting to transition off.
        /// </summary>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus,
                                                       bool coveredByOtherScreen)
        {
            if (!videoStarted)
            {
                ScreenManager.VideoPlayer.Play(videoFile);
                videoStarted = true;
            }
            else
            {
                if (ScreenManager.VideoPlayer.State == MediaState.Stopped)
                {
                    // Return to main menu
                    PendulumLoadingScreen.Load(ScreenManager, false, null, new PendulumBackgroundScreen(),
                                               new PendulumMainMenuScreen());
                }
            }

            base.Update(gameTime, otherScreenHasFocus, false);
        }

        /// <summary>
        /// Draws the menu screen. This darkens down the gameplay screen
        /// that is underneath us, and then chains to the base MenuScreen.Draw.
        /// </summary>
        public override void Draw(GameTime gameTime)
        {
            // Dim background
            ScreenManager.FadeBackBufferToBlack(3 * TransitionAlpha / 4);

            // Messages
            string[] messages = new string[5];
            if (PendulumGame.State.Players[0].Points > PendulumGame.State.Players[1].Points)
            {
                messages[0] = "Player A wins!";
            }
            else if (PendulumGame.State.Players[0].Points == PendulumGame.State.Players[1].Points)
            {
                messages[0] = "It is a Tie!";
            }
            else
            {
                messages[0] = "Player B wins!";
            }

            messages[0] = "===[ " + messages[0] + " ]===";
            messages[1] = "Player A got " + PendulumGame.State.Players[0].Points + " points";
            messages[2] = "Player B got " + PendulumGame.State.Players[1].Points + " points";
            messages[3] = PendulumGame.State.SubmittedEntropyBits + " bits of Entropy extracted from last round";
            messages[4] = 
                PendulumGame.State.PhysicalRandomNumberGenerator.RetrievedEntropyBits + " bits of " +
                PendulumGame.State.PhysicalRandomNumberGenerator.EntropySource.ToString() + " Entropy used in total";

            // Draw score summary
            SpriteBatch spriteBatch = ScreenManager.SpriteBatch;
            SpriteFont font = ScreenManager.Fonts["retro"];
            Viewport viewport = ScreenManager.GraphicsDevice.Viewport;
            Rectangle fullscreen = new Rectangle(0, 0, viewport.Width, viewport.Height);
            Color color = Color.White;
            Vector2 origin = new Vector2(0, 0);
            Vector2 position = new Vector2(viewport.Width * 0.3f, viewport.Height * 0.65f);
            spriteBatch.Begin();
            foreach (string message in messages)
            {
                spriteBatch.DrawString(
                    font,
                    message,
                    position,
                    color,
                    0,
                    origin,
                    1.0f,
                    SpriteEffects.None,
                    0);
                position.Y += (viewport.Height * 0.05f);
            }

            spriteBatch.Draw(
                frameTexture,
                fullscreen,
                Color.White);
            spriteBatch.End();

            // Render the video in it's orginal resolution 
            // to the screen using SpriteBatch
            if (ScreenManager.VideoPlayer.State == MediaState.Playing)
            {
                int frameSize = (int)((float)viewport.Width * 0.3f);
                Rectangle videoFrame = 
                    new Rectangle(
                        viewport.Width - frameSize - 10, 
                        viewport.Height / 2 - frameSize / 2 - 30,
                        frameSize, 
                        frameSize);
                spriteBatch.Begin(SpriteBlendMode.AlphaBlend, SpriteSortMode.Immediate, SaveStateMode.None);
                effect.Begin();
                float threshold = 0.33f;
                effect.Parameters["referenceColor"].SetValue(this.referenceColor.ToVector3());
                effect.Parameters["threshold"].SetValue(threshold);
                foreach (EffectPass pass in effect.CurrentTechnique.Passes)
                {
                    pass.Begin();
                    spriteBatch.Draw(
                        ScreenManager.VideoPlayer.GetTexture(),
                        videoFrame,
                        Color.White);
                    pass.End();
                }
                effect.End();
                spriteBatch.End();
            }

            // If the game is transitioning on or off, fade it out to black.
            if (TransitionPosition > 0)
            {
                ScreenManager.FadeBackBufferToBlack(255 - TransitionAlpha);
            }

            base.Draw(gameTime);
        }

        #endregion
    }
}

