//-----------------------------------------------------------------------------
// PendulumGameplayScreen.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace PendulumGame
{
    using System;
    using System.IO;
    using System.Text;
    using System.Threading;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Input;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Mathematics;
    using NewGamePhysics.Physics;
    using NewGamePhysics.GraphicalElements;
    using NewGamePhysics.PhysicalElements;
    using NewGamePhysics.Utilities;
    using NewGamePhysics.Networking;

    using Microsoft.DirectX.DirectSound;

    /// <summary>
    /// This screen implements the actual game logic.
    /// </summary>
    class PendulumGameplayScreen : GameScreen
    {
        #region Fields

        /// <summary>
        /// The minimum energy for entropy collection to occur. 
        /// </summary>
        private const double MinimumEnergyForEntropyCollection = 10.0;

        /// <summary>
        /// Content manager (cache from main game).
        /// </summary>
        private ContentManager contentManager;

        /// <summary>
        /// Realtime Audio player.
        /// </summary>
        private DirectXAudio audioPlayer;

        /// <summary>
        /// Pendulums for gameplay: regular Pendulum.
        /// </summary>
        private DoubleRegularPendulum animatedDoubleRegularPendulum;

        /// <summary>
        /// Pendulums for gameplay: square Pendulum.
        /// </summary>
        private DoubleSquarePendulum animatedDoubleSquarePendulum;

        /// <summary>
        /// Origins of pendulums.
        /// </summary>
        private VectorN[] pendulumOrigins = new VectorN[2];

        /// <summary>
        /// Sizes of pendulums.
        /// </summary>
        private double[] pendulumScales = new double[2];

        /// <summary>
        /// Pendulums for audio generation: regular Pendulum.
        /// </summary>
        private DoubleRegularPendulumSimulation audioPendulum1 = null;

        /// <summary>
        /// Pendulums for audio generation: square Pendulum.
        /// </summary>
        private DoubleSquarePendulumSimulation audioPendulum2 = null;

        /// <summary>
        /// Audio pendulum sample lifetime counter
        /// </summary>
        private int[] audioPendulumSamplesRemaining = new int[2];

        /// <summary>
        ///  Chart Plotters for Pendulums
        /// </summary>
        private DotPlotter[] dotPlotters;

        /// <summary>
        /// The displacement step per update for plotters.
        /// </summary>
        private Vector2[] dotPlotterDisplacement;

        /// <summary>
        /// Laplace for balls
        /// </summary>
        private LaplaceOverlay laplaceOverlay;

        /// <summary>
        /// Blooming for plotters
        /// </summary>
        private BloomOverlay bloomOverlay;

        /// <summary>
        /// Energy displays
        /// </summary>
        private EnergyIndicator[] energyIndicators;

        /// <summary>
        /// Value displays
        /// </summary>
        private ValueIndicator[] valueIndicators;

        /// <summary>
        /// The shooting gallery
        /// </summary>
        private ShootingGallery shootingGallery;

        /// <summary>
        /// Entropy collectors for players.
        /// </summary>
        private AdvancedEntropyCollector[] entropyCollectors;

        /// <summary>
        /// Parameter indicators for pendulum state.
        /// </summary>
        private RotationalActionIndicator[] actionIndicators;

        /// <summary>
        /// The origin of the display with respect to the simulation.
        /// </summary>
        private Vector2 screenOrigin;

        /// <summary>
        /// The scale of the display with respect to the simulation.
        /// </summary>
        private double screenScale;

        /// <summary>
        /// Triger playback of audio from pendulum 1 when on onDemand mode.
        /// </summary>
        private bool triggerAudio1 = false;

        /// <summary>
        /// Triger playback of audio from pendulum 2 when on onDemand mode.
        /// </summary>
        private bool triggerAudio2 = false;

        #endregion

        #region Initialization

        /// <summary>
        /// Constructor of the screen instance.
        /// </summary>
        public PendulumGameplayScreen()
        {
            TransitionOnTime = TimeSpan.FromSeconds(1.0);
            TransitionOffTime = TimeSpan.FromSeconds(1.0);

            // Create playback streamer
            this.audioPlayer = new DirectXAudio((Device)null, 22050, (short)16, (short)2);
        }

        /// <summary>
        /// Load graphics content for the game.
        /// </summary>
        public override void LoadContent()
        {
            if (contentManager == null)
            {
                contentManager = new ContentManager(ScreenManager.Game.Services, "Content");
            }

            // Local viewport
            Viewport viewport = ScreenManager.GraphicsDevice.Viewport;

            // Initialize the game players
            for (int i = 0; i < PendulumGameState.NumPlayers; i++)
            {
                PendulumGame.State.Players[i] = new GamePlayer();
                PendulumGame.State.Players[i].ActionMarker = DoublePendulumHinges.Hinge1;
            }

            // Set gravity
            GravityCalculator gravityCalculator = new GravityCalculator(PendulumGame.State.CurrentCelestialObject);
            PendulumGame.State.CurrentGravity = gravityCalculator.Value;

            // Create physical element(s)
            double size = 1.0;
            double size2 = 1.0;
            double mass = 1.0;
            double mass2 = 1.0;

            this.screenOrigin =
                new Vector2((float)(viewport.Width / 2.0), (float)(viewport.Height / 2.0));
            this.screenScale = 140.0;

            // Pendulums
            this.animatedDoubleRegularPendulum = new DoubleRegularPendulum(
                ScreenManager,
                new Vector2(-2.0f, 0.0f),
                size,
                mass,
                size2,
                mass2,
                PendulumGame.State.CurrentGravity,
                PendulumGame.State.CurrentRotationalFrictionType,
                screenOrigin,
                screenScale);

            this.animatedDoubleSquarePendulum = new DoubleSquarePendulum(
                ScreenManager,
                new Vector2(2.0f, 0.0f),
                size * (82.8427 / 100.0),
                mass,
                PendulumGame.State.CurrentGravity,
                PendulumGame.State.CurrentRotationalFrictionType,
                screenOrigin,
                screenScale);

            // Plotters
            dotPlotters = new DotPlotter[2];
            dotPlotters[0] = new DotPlotter(2 * 1024);
            dotPlotters[1] = new DotPlotter(2 * 1024);
            dotPlotterDisplacement = new Vector2[2];
            dotPlotterDisplacement[0] =
                new Vector2(0.5f * PendulumGame.State.Scale.X, 0.0f);
            dotPlotterDisplacement[1] =
                new Vector2(-0.5f * PendulumGame.State.Scale.X, 0.0f);

            // Energy Indicators
            energyIndicators = new EnergyIndicator[2];
            energyIndicators[0] = new EnergyIndicator(ScreenManager, "E");
            energyIndicators[1] = new EnergyIndicator(ScreenManager, "E");

            // Indicators
            valueIndicators = new ValueIndicator[8];
            valueIndicators[0] = new ValueIndicator(ScreenManager, "A Angle Inner", "{0,20:####.##} deg", -180, 180);
            valueIndicators[1] = new ValueIndicator(ScreenManager, "B Angle Inner", "{0,20:####.##} deg", -180, 180);
            valueIndicators[2] = new ValueIndicator(ScreenManager, "A Angle Outer", "{0,20:####.##} deg", -180, 180);
            valueIndicators[3] = new ValueIndicator(ScreenManager, "B Angle Outer", "{0,20:####.##} deg", -180, 180);
            valueIndicators[4] = new ValueIndicator(ScreenManager, "A Speed Inner", "{0,20:####.##} 1/s", -20, 20);
            valueIndicators[5] = new ValueIndicator(ScreenManager, "B Speed Inner", "{0,20:####.##} 1/s", -20, 20);
            valueIndicators[6] = new ValueIndicator(ScreenManager, "A Speed Outer", "{0,20:####.##} 1/s", -20, 20);
            valueIndicators[7] = new ValueIndicator(ScreenManager, "B Speed Outer", "{0,20:####.##} 1/s", -20, 20);

            // Place all indicators at the bottom in the same spot
            float yPos;
            yPos = viewport.Height - EnergyIndicator.Height - 10.0f;
            energyIndicators[0].SetPosition(new Vector2(5.0f, yPos));
            energyIndicators[1].SetPosition(new Vector2(viewport.Width - EnergyIndicator.Width - 5.0f, yPos));
            yPos = viewport.Height - ValueIndicator.Height - 10.0f + 1.0f;
            for (int i = 0; i < 4; i++)
            {
                valueIndicators[2 * i].SetPosition(new Vector2(25.0f, yPos));
                valueIndicators[2 * i + 1].SetPosition(new Vector2(viewport.Width - ValueIndicator.Width - 25.0f, yPos));
            }

            // Create shooting gallery across screen
            float simulationHeight = 0.3f;
            float simulationWidth = 8.0f;
            this.shootingGallery =
                new ShootingGallery(
                    ScreenManager,
                    new Vector2(-simulationWidth / 2.0f, -2.0f),
                    simulationWidth,
                    simulationHeight,
                    screenOrigin,
                    screenScale);
            this.shootingGallery.TargetConveyerSpeed = 0.005;
            CelestialBody celestialBody =
                new CelestialBody(PendulumGame.State.CurrentCelestialObject);
            string[] targetTypes = new string[12];
            string[] targetChoices = Enum.GetNames(typeof(ShootingGalleryTargetType));
            for (int i = 0; i < targetTypes.Length; i++)
            {
                int randomChoice = PendulumGame.State.PhysicalRandomNumberGenerator.Next(
                        0,
                        targetChoices.Length - 1);
                targetTypes[i] = targetChoices[randomChoice];
            }

            shootingGallery.Reset(
                targetTypes,
                PendulumGame.State.CurrentGravity,
                celestialBody.GetAtmosphericDensity());

            PendulumGame.State.PhysicalRandomNumberGenerator.Next(0, 2);

            // Action indicators
            actionIndicators = new RotationalActionIndicator[2];
            actionIndicators[0] = new RotationalActionIndicator(ScreenManager);
            actionIndicators[1] = new RotationalActionIndicator(ScreenManager);

            // Random number generators
            entropyCollectors = new AdvancedEntropyCollector[2];
            entropyCollectors[0] = new AdvancedEntropyCollector();
            entropyCollectors[1] = new AdvancedEntropyCollector();

            // Reset bit count
            PendulumGame.State.SubmittedEntropyBits = 0;

            // Pick random info link on 'randomness' and send it
            Program.Game.SendRandomInfoLink("Randomness");

            // Load laplace effect
            this.laplaceOverlay = new LaplaceOverlay(this.ScreenManager);
            this.laplaceOverlay.LoadContent();

            // Load bloom effect
            this.bloomOverlay = new BloomOverlay(this.ScreenManager);
            this.bloomOverlay.LoadContent();

            // Reset game time
            ScreenManager.Game.ResetElapsedTime();

            // Start audio playback
            this.audioPlayer.Play(new PullAudioCallback(PendulumFiller));
        }

        /// <summary>
        /// Unload graphics content used by the game screen.
        /// </summary>
        public override void UnloadContent()
        {
            // Stop playback
            this.audioPlayer.Stop();

            // Unload laplace effect
            this.laplaceOverlay.UnloadContent();

            // Unload bloom effect
            this.bloomOverlay.UnloadContent();

            // Pick random info link on 'randomness' and send it 
            Program.Game.SendRandomInfoLink("Randomness");

            // Maybe send entropy to service
            CheckAndSubmitEntropy(0, 1);
            CheckAndSubmitEntropy(1, 1);

            // Unload all content
            contentManager.Unload();

            // Unload base 
            base.UnloadContent();
        }

        #endregion

        #region Update and Draw


        /// <summary>
        /// Updates the state of the game. This method checks the GameScreen.IsActive
        /// property, so the game will stop updating when the pause menu is active,
        /// or if you tab away to a different application.
        /// </summary>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus,
                                                       bool coveredByOtherScreen)
        {
            base.Update(gameTime, otherScreenHasFocus, coveredByOtherScreen);

            if (IsActive)
            {
                Vector2[] pendulumPoints;
                double rotation;

                // ----------- Update simulations ---------

                Viewport viewport = ScreenManager.GraphicsDevice.Viewport;

                // Move plotter points
                dotPlotters[0].MoveDots(dotPlotterDisplacement[0]);
                dotPlotters[1].MoveDots(dotPlotterDisplacement[1]);

                // Player A: Animate double pendulum

                // Maybe drive pendulum
                rotation = animatedDoubleRegularPendulum.Pendulum.Drive(
                        PendulumGame.State.Players[0].ActionMarker,
                        PendulumGame.State.Players[0].ActionIntensity);

                // Update action indicator
                actionIndicators[0].Animate(rotation);

                // Update pendulum animation
                animatedDoubleRegularPendulum.Pendulum.Animate();

                // Get pendulum locations in screen coordinates
                pendulumPoints =
                    animatedDoubleRegularPendulum.Pendulum.GetPosition(this.screenOrigin, this.screenScale);

                // Updated Plotter
                dotPlotters[0].AddDot(pendulumPoints[2], new Color(255, 0, 0), 0);

                // Update action indicator position
                switch (PendulumGame.State.Players[0].ActionMarker)
                {
                    case DoublePendulumHinges.Hinge1:
                        actionIndicators[0].SetPosition(pendulumPoints[0]);
                        break;
                    case DoublePendulumHinges.Hinge2:
                        actionIndicators[0].SetPosition(pendulumPoints[1]);
                        break;
                }

                // Get pendulum locations in simulation coordinates
                pendulumPoints =
                    animatedDoubleRegularPendulum.Pendulum.GetPosition();

                // Check if Player A hits targets
                int hitsA = shootingGallery.ShootAtTargets(pendulumPoints[2], 0.01);
                if (hitsA > 0)
                {
                    PendulumGame.State.Players[0].Points += hitsA;

                    // Pick random info link on 'pendulums' and send it 
                    Program.Game.SendRandomInfoLink("DoublePendulum");
                }

                // Player B: Animate double square pendulum

                // Maybe drive pendulum
                rotation = animatedDoubleSquarePendulum.Pendulum.Drive(
                        PendulumGame.State.Players[1].ActionMarker,
                        PendulumGame.State.Players[1].ActionIntensity);

                // Update action indicator
                actionIndicators[1].Animate(rotation);

                // Update pendulum animation
                animatedDoubleSquarePendulum.Pendulum.Animate();

                // Get pendulum locations in screen coordinates
                pendulumPoints =
                    animatedDoubleSquarePendulum.Pendulum.GetPosition(this.screenOrigin, this.screenScale);

                // Updated Plotter
                dotPlotters[1].AddDot(pendulumPoints[6], new Color(0, 0, 192), 0);

                // Update action indicator position
                switch (PendulumGame.State.Players[1].ActionMarker)
                {
                    case DoublePendulumHinges.Hinge1:
                        actionIndicators[1].SetPosition(pendulumPoints[0]);
                        break;
                    case DoublePendulumHinges.Hinge2:
                        actionIndicators[1].SetPosition(pendulumPoints[4]);
                        break;
                }

                // Get pendulum locations in simulation coordinates
                pendulumPoints =
                    animatedDoubleSquarePendulum.Pendulum.GetPosition();

                // Check if Player B hits targets
                int hitsB = shootingGallery.ShootAtTargets(pendulumPoints[6], 0.01);
                if (hitsB > 0)
                {
                    PendulumGame.State.Players[1].Points += hitsB;

                    // Pick random info link on 'pendulums' and send it 
                    Program.Game.SendRandomInfoLink("DoublePendulum");
                }

                // Animate shooting gallery
                shootingGallery.Update(gameTime);

                // Check for game over
                if (shootingGallery.NumActiveTargets == 0)
                {
                    // Maybe send entropy to service
                    CheckAndSubmitEntropy(0, 1);
                    CheckAndSubmitEntropy(1, 1);

                    ScreenManager.AddScreen(
                        new PendulumGameOverScreen(), null);
                }

                // Update all indicator values
                double e1 = animatedDoubleRegularPendulum.Pendulum.GetEnergy();
                energyIndicators[0].SetValue(e1);

                double e2 = animatedDoubleSquarePendulum.Pendulum.GetEnergy();
                energyIndicators[1].SetValue(e2);

                double[] a1 = animatedDoubleRegularPendulum.Pendulum.GetAngle();
                valueIndicators[0].SetValue(ToDegree(a1[0]));
                valueIndicators[2].SetValue(ToDegree(a1[1]));

                double[] a2 = animatedDoubleSquarePendulum.Pendulum.GetAngle();
                valueIndicators[1].SetValue(ToDegree(a2[0]));
                valueIndicators[3].SetValue(ToDegree(a2[1]));

                double[] v1 = animatedDoubleRegularPendulum.Pendulum.GetVelocity();
                valueIndicators[4].SetValue(v1[0]);
                valueIndicators[6].SetValue(v1[1]);

                double[] v2 = animatedDoubleSquarePendulum.Pendulum.GetVelocity();
                valueIndicators[5].SetValue(v2[0]);
                valueIndicators[7].SetValue(v2[1]);

                // Pick up randomness

                // Enough energy in the pendulum of player A?
                if (e1 > MinimumEnergyForEntropyCollection)
                {
                    if (PendulumGame.State.Players[1].ActionIntensityChanged())
                    {
                        CollectEntropyFromPlayerA();
                    }
                }

                // Enough energy in the pendulum of player B
                if (e2 > MinimumEnergyForEntropyCollection)
                {
                    if (PendulumGame.State.Players[0].ActionIntensityChanged())
                    {
                        CollectEntropyFromPlayerB();
                    }
                }

                // Active pendulum 1 sound
                if ((this.audioPendulum1 == null) &&
                    (this.audioPendulumSamplesRemaining[0] == 0))
                {
                    if ((!PendulumGame.State.RealtimeAudioOnDemand) || (triggerAudio1))
                    {
                        // Reset trigger
                        triggerAudio1 = false;

                        // Current state
                        double[] angles = this.animatedDoubleRegularPendulum.Pendulum.GetAngle();
                        double[] velocities = this.animatedDoubleRegularPendulum.Pendulum.GetVelocity();

                        float theta1 = MathHelper.WrapAngle((float)angles[0]);
                        float theta2 = MathHelper.WrapAngle((float)angles[1]);

                        // New Pendulum
                        DoubleRegularPendulumSimulation audioPendulum =
                            new DoubleRegularPendulumSimulation(
                                new Vector2(),
                                0.1, 1.0, 0.1, 1.0,
                                PendulumGame.State.CurrentGravity,
                                PendulumGame.State.CurrentRotationalFrictionType);
                        audioPendulum.SetInitialConditions(
                                (double)theta1, velocities[0], (double)theta2, velocities[1]);

                        // Activate
                        this.audioPendulumSamplesRemaining[0] = 38000;
                        this.audioPendulum1 = audioPendulum;
                    }
                }
                else
                {
                    if (this.audioPendulumSamplesRemaining[0] == 0)
                    {
                        this.audioPendulum1 = null;
                    }
                }

                // Active pendulum 2 sound
                if ((this.audioPendulum2 == null) &&
                    (this.audioPendulumSamplesRemaining[1] == 0))
                {
                    if ((!PendulumGame.State.RealtimeAudioOnDemand) || (triggerAudio2))
                    {
                        // Reset trigger
                        triggerAudio2 = false;

                        // Current state
                        double[] angles = this.animatedDoubleSquarePendulum.Pendulum.GetAngle();
                        double[] velocities = this.animatedDoubleSquarePendulum.Pendulum.GetVelocity();

                        float theta1 = MathHelper.WrapAngle((float)angles[0]);
                        float theta2 = MathHelper.WrapAngle((float)angles[1]);

                        // New Pendulum
                        DoubleSquarePendulumSimulation audioPendulum =
                            new DoubleSquarePendulumSimulation(
                                new Vector2(),
                                0.1, 1.0,
                                PendulumGame.State.CurrentGravity,
                                PendulumGame.State.CurrentRotationalFrictionType);
                        audioPendulum.SetInitialConditions(
                                (double)theta1, velocities[0], (double)theta2, velocities[1]);

                        // Activate
                        this.audioPendulumSamplesRemaining[1] = 44000;
                        this.audioPendulum2 = audioPendulum;
                    }
                }
                else
                {
                    if (this.audioPendulumSamplesRemaining[1] == 0)
                    {
                        this.audioPendulum2 = null;
                    }
                }
            }
        }

        /// <summary>
        /// Lets the game respond to player input. Unlike the Update method,
        /// this will only be called when the gameplay screen is active.
        /// </summary>
        public override void HandleInput(InputState input)
        {
            if (input == null)
            {
                throw new ArgumentNullException("input");
            }

            PlayerIndex playerIndex;

            if (input.IsInputCancel(null) || input.IsDisconnected(null))
            {
                ScreenManager.AddScreen(new PendulumPauseMenuScreen(), null);
            }
            else
            {
                // Keep track of action intensity
                for (int i = 0; i < PendulumGameState.NumPlayers; i++)
                {
                    PendulumGame.State.Players[i].TrackActionIntensity();
                }

                // Player 1 - top actuator
                if (input.IsNewKeyPress(Keys.Q, null, out playerIndex) ||
                    input.IsNewButtonPress(Buttons.B, PlayerIndex.One, out playerIndex))
                {
                    PendulumGame.State.Players[0].ActionMarker = DoublePendulumHinges.Hinge1;
                }

                // Player 1 - bottom actuator
                if (input.IsNewKeyPress(Keys.A, null, out playerIndex) ||
                    input.IsNewButtonPress(Buttons.A, PlayerIndex.One, out playerIndex))
                {
                    PendulumGame.State.Players[0].ActionMarker = DoublePendulumHinges.Hinge2;
                }

                // Player 2 - top actuator
                if (input.IsNewKeyPress(Keys.Up, null, out playerIndex) ||
                    input.IsNewButtonPress(Buttons.B, PlayerIndex.Two, out playerIndex) ||
                    (PendulumGame.State.UseOnlyOneGamepad && input.IsNewButtonPress(Buttons.Y, PlayerIndex.One, out playerIndex)))
                {
                    PendulumGame.State.Players[1].ActionMarker = DoublePendulumHinges.Hinge1;
                }

                // Player 2 - bottom actuator
                if (input.IsNewKeyPress(Keys.Down, null, out playerIndex) ||
                    input.IsNewButtonPress(Buttons.A, PlayerIndex.Two, out playerIndex) ||
                    (PendulumGame.State.UseOnlyOneGamepad && input.IsNewButtonPress(Buttons.X, PlayerIndex.One, out playerIndex)))
                {
                    PendulumGame.State.Players[1].ActionMarker = DoublePendulumHinges.Hinge2;
                }

                // Player 1 - swing left or right
                if (input.IsKeyDown(Keys.Z, null, out playerIndex))
                {
                    PendulumGame.State.Players[0].ActionIntensity = -1.0;
                }
                else if (input.IsKeyDown(Keys.X, null, out playerIndex))
                {
                    PendulumGame.State.Players[0].ActionIntensity = 1.0;
                }
                else
                {
                    PendulumGame.State.Players[0].ActionIntensity =
                        input.CurrentGamePadStates[0].ThumbSticks.Left.X;
                }

                // Player 2 - swing left or right
                if (input.IsKeyDown(Keys.Left, null, out playerIndex))
                {
                    PendulumGame.State.Players[1].ActionIntensity = -1.0;
                }
                else if (input.IsKeyDown(Keys.Right, null, out playerIndex))
                {
                    PendulumGame.State.Players[1].ActionIntensity = 1.0;
                }
                else
                {

                    // Player 2 - swing with gamepad
                    if (PendulumGame.State.UseOnlyOneGamepad)
                    {
                        PendulumGame.State.Players[1].ActionIntensity =
                            input.CurrentGamePadStates[0].ThumbSticks.Right.X;
                    }
                    else
                    {
                        PendulumGame.State.Players[1].ActionIntensity =
                            input.CurrentGamePadStates[1].ThumbSticks.Left.X;
                    }
                }

                // Keep track of action intensity
                for (int i = 0; i < PendulumGameState.NumPlayers; i++)
                {
                    PendulumGame.State.Players[i].TrackActionIntensity();
                }

                // Player 1 - display switch
                if (input.IsNewKeyPress(Keys.LeftShift, null, out playerIndex) ||
                    input.IsNewButtonPress(Buttons.LeftTrigger, PlayerIndex.One, out playerIndex))
                {
                    PendulumGame.State.Players[0].SelectedIndicator++;
                    PendulumGame.State.Players[0].SelectedIndicator %= (valueIndicators.Length / 2);
                    triggerAudio1 = true;
                }
                else if (input.IsNewButtonPress(Buttons.RightTrigger, PlayerIndex.One, out playerIndex))
                {
                    PendulumGame.State.Players[0].SelectedIndicator--;
                    if (PendulumGame.State.Players[0].SelectedIndicator < 0)
                    {
                        PendulumGame.State.Players[0].SelectedIndicator += (valueIndicators.Length / 2);
                    }
                }

                // Player 2 - display switch audio trigger
                if (input.IsNewKeyPress(Keys.RightShift, null, out playerIndex) ||
                    input.IsNewButtonPress(Buttons.LeftTrigger, PlayerIndex.Two, out playerIndex) ||
                    (PendulumGame.State.UseOnlyOneGamepad &&
                     input.IsNewButtonPress(Buttons.RightTrigger, PlayerIndex.One, out playerIndex)))
                {
                    PendulumGame.State.Players[1].SelectedIndicator++;
                    PendulumGame.State.Players[1].SelectedIndicator %= (valueIndicators.Length / 2);
                    triggerAudio2 = true;
                }
                else if (input.IsNewButtonPress(Buttons.RightTrigger, PlayerIndex.Two, out playerIndex))
                {
                    PendulumGame.State.Players[1].SelectedIndicator--;
                    if (PendulumGame.State.Players[1].SelectedIndicator < 0)
                    {
                        PendulumGame.State.Players[1].SelectedIndicator += (valueIndicators.Length / 2);
                    }
                }

                // Early game-over
                if (input.IsNewKeyPress(Keys.G, null, out playerIndex))
                {
                    ScreenManager.AddScreen(
                        new PendulumGameOverScreen(), null);
                }
            }
        }

        /// <summary>
        /// Draws the gameplay screen.
        /// </summary>
        public override void Draw(GameTime gameTime)
        {
            // This game has a blue background. Why? Because!
            ScreenManager.GraphicsDevice.Clear(ClearOptions.Target,
                                               Color.Black, 0, 0);

            SpriteBatch spriteBatch = ScreenManager.SpriteBatch;
            PrimitiveBatch primitiveBatch = ScreenManager.PrimitiveBatch;
            Viewport viewport = ScreenManager.GraphicsDevice.Viewport;

            // Draw physical elements
            // Plotter            
            dotPlotters[0].Draw(gameTime, primitiveBatch);
            dotPlotters[1].Draw(gameTime, primitiveBatch);

            // Bloom dot plotters using shaders
            if (!PendulumGame.State.DisablePixelShaderEffects)
            {
                this.bloomOverlay.Draw(gameTime);
            }

            // Laplace-render the targets using shaders
            if ((!PendulumGame.State.DisablePixelShaderEffects) && 
                (this.shootingGallery.NumActiveTargets > 0))
            {
                shootingGallery.Draw(gameTime);
                this.laplaceOverlay.Draw(gameTime);
            }

            // Draw the two pendulums
            animatedDoubleRegularPendulum.Draw(
                gameTime);
            animatedDoubleSquarePendulum.Draw(
                gameTime);

            // Draw action markers
            actionIndicators[0].Draw(gameTime);
            actionIndicators[1].Draw(gameTime);

            // Draw targets again
            shootingGallery.Draw(gameTime);

            // Indicators for both players
            energyIndicators[0].Draw(gameTime);
            energyIndicators[1].Draw(gameTime);
            valueIndicators[PendulumGame.State.Players[0].SelectedIndicator * 2].Draw(gameTime);
            valueIndicators[PendulumGame.State.Players[1].SelectedIndicator * 2 + 1].Draw(gameTime);

            // Separator
            primitiveBatch.Begin(PrimitiveType.LineList);
            primitiveBatch.AddVertex(new Vector2(viewport.Width * 0.5f, viewport.Height - 50.0f), Color.White);
            primitiveBatch.AddVertex(new Vector2(viewport.Width * 0.5f, viewport.Height - 10.0f), Color.White);
            primitiveBatch.End();

            // Draw entropy state
            DrawEntropyStatus(spriteBatch);

            // Draw player score
            DrawPlayers(spriteBatch);

            // If the game is transitioning on or off, fade it out to black.
            if (TransitionPosition > 0)
            {
                ScreenManager.FadeBackBufferToBlack(255 - TransitionAlpha);
            }
        }

        #endregion

        /// <summary>
        /// Use angle from pendulum of player A as value for entropy collector.
        /// </summary>
        private void CollectEntropyFromPlayerA()
        {
            entropyCollectors[1].AddValue(
                animatedDoubleRegularPendulum.Pendulum.GetAngle(PendulumGame.State.Players[1].ActionMarker));
            dotPlotters[0].SetTypeAtHead(1);
        }

        /// <summary>
        /// Use angle from pendulum of player B as value for entropy collector.
        /// </summary>       
        private void CollectEntropyFromPlayerB()
        {
            entropyCollectors[0].AddValue(
                animatedDoubleSquarePendulum.Pendulum.GetAngle(PendulumGame.State.Players[0].ActionMarker));
            dotPlotters[1].SetTypeAtHead(1);
        }

        /// <summary>
        /// Check and submit entropy bits for a player. Sends
        /// collected entropy bits in background thread to webservice.
        /// </summary>
        /// <param name="playerIndex">
        /// Index of the player/collector to process.</param>
        /// <param name="checkLimit">
        /// Submit bits if at least this number of bits are available.</param>
        private void CheckAndSubmitEntropy(int playerIndex, int checkLimit)
        {
            // Get bits
            string entropy = entropyCollectors[playerIndex].ToString();

            // Maybe submit
            if (entropy.Length >= checkLimit)
            {
                PendulumGame.State.SubmittedEntropyBits += entropy.Length;
                Program.Game.SubmitEntropyBits(entropy);
                entropyCollectors[playerIndex].Reset();
            }
        }

        /// <summary>
        /// Show currently collected values for entropy as status for both players.
        /// </summary>
        /// <param name="spriteBatch">The sprite batch to use for drawing</param>
        private void DrawEntropyStatus(SpriteBatch spriteBatch)
        {
            // Get counts
            int countA = entropyCollectors[0].ValuePoolSize;
            int countB = entropyCollectors[1].ValuePoolSize;

            SpriteFont font = ScreenManager.Fonts["small"];
            Viewport viewport = ScreenManager.GraphicsDevice.Viewport;
            Color color = Color.White;
            Vector2 origin = new Vector2(0, 0);
            Vector2 position = new Vector2(viewport.Width * 0.28f, viewport.Height - font.LineSpacing - 10.0f);
            string message = "A Collected: " + countA.ToString();
            spriteBatch.Begin();
            spriteBatch.DrawString(
                font,
                message,
                position,
                color,
                0,
                origin,
                1.0f,
                SpriteEffects.None,
                0);
            position = new Vector2(viewport.Width * 0.52f, viewport.Height - font.LineSpacing - 10.0f);
            message = "B Collected: " + countB.ToString();
            spriteBatch.DrawString(
                font,
                message,
                position,
                color,
                0,
                origin,
                1.0f,
                SpriteEffects.None,
                0);
            spriteBatch.End();
        }

        /// <summary>
        /// Show the player score and stats.
        /// </summary>
        /// <param name="spriteBatch">The sprite batch to use for drawing</param>
        private void DrawPlayers(SpriteBatch spriteBatch)
        {
            SpriteFont font = ScreenManager.Fonts["small"];
            Viewport viewport = ScreenManager.GraphicsDevice.Viewport;
            Color color = Color.White;
            Vector2 origin = new Vector2(0, 0);
            Vector2 position = new Vector2(viewport.Width * 0.28f, viewport.Height - 2 * font.LineSpacing - 20.0f);
            string message = "Player A: " + PendulumGame.State.Players[0].Points + " points";
            spriteBatch.Begin();
            spriteBatch.DrawString(
                font,
                message,
                position,
                color,
                0,
                origin,
                1.0f,
                SpriteEffects.None,
                0);
            position = new Vector2(viewport.Width * 0.52f, viewport.Height - 2 * font.LineSpacing - 20.0f);
            message = "Player B: " + PendulumGame.State.Players[1].Points + " points";
            spriteBatch.DrawString(
                font,
                message,
                position,
                color,
                0,
                origin,
                1.0f,
                SpriteEffects.None,
                0);
            spriteBatch.End();
        }

        /// <summary>
        /// Convert a value to degrees and keep in a range.
        /// </summary>
        /// <param name="value">Input radians.</param>
        /// <returns>Output degrees in range.</returns>
        private double ToDegree(double value)
        {
            value = value * 180.0 / Math.PI;
            if (value < 0.0)
            {
                while (value < -180.0)
                {
                    value += 360.0;
                }
            }
            else
            {
                while (value > 180.0)
                {
                    value -= 360.0;
                }
            }

            return value;
        }

        /// <summary>
        /// Audio filler for generating sounds from each pendulum.
        /// </summary>
        /// <param name="dest">Pointer to byte data to update.</param>
        /// <param name="size">Number of bytes to update.</param>
        private void PendulumFiller(IntPtr dest, int size)
        {
            // maximum wave amplitude
            const float maxAmplitude = 0.90f;

            // Calculate number fo 16bit samples
            int samples = size / 2;

            // Create local buffer
            short[] buffer = new short[samples];

            // Maybe mix in audio pendulum 1 values into left channel
            if (this.audioPendulum1 != null)
            {
                int samplePos = 0; // offset for left channel
                short sampleValue = 0;
                double[] angles;
                double[] velocities;
                float theta;
                float omega;
                float fvalue = 0.0f;
                while ((this.audioPendulumSamplesRemaining[0] > 0) && (samplePos < samples))
                {
                    // Update audio pendulum
                    this.audioPendulum1.Animate();

                    // Get values depending on the value indicator which was selected
                    switch (PendulumGame.State.Players[0].SelectedIndicator)
                    {
                        case 0:
                            angles = audioPendulum1.GetAngle();
                            theta = MathHelper.WrapAngle((float)angles[0]);
                            fvalue = theta / (float)Math.PI * maxAmplitude * (float)(short.MaxValue);
                            break;
                        case 1:
                            angles = audioPendulum1.GetAngle();
                            theta = MathHelper.WrapAngle((float)angles[1]);
                            fvalue = theta / (float)Math.PI * maxAmplitude * (float)(short.MaxValue);
                            break;
                        case 2:
                            velocities = audioPendulum1.GetVelocity();
                            omega = (float)velocities[0];
                            fvalue = (float)omega * 0.1f;
                            if (fvalue < -maxAmplitude)
                            {
                                fvalue = -maxAmplitude;
                            }
                            else if (fvalue > maxAmplitude)
                            {
                                fvalue = maxAmplitude;
                            }

                            fvalue *= (float)(short.MaxValue);
                            break;
                        case 3:
                            velocities = audioPendulum1.GetVelocity();
                            omega = (float)velocities[1];
                            fvalue = (float)omega * 0.1f;
                            if (fvalue < -maxAmplitude)
                            {
                                fvalue = -maxAmplitude;
                            }
                            else if (fvalue > maxAmplitude)
                            {
                                fvalue = maxAmplitude;
                            }

                            fvalue *= (float)(short.MaxValue);
                            break;
                        default:
                            fvalue = 0.0f;
                            break;
                    }

                    sampleValue = Convert.ToInt16(fvalue);
                    buffer[samplePos] = sampleValue;
                    samplePos += 2;

                    this.audioPendulumSamplesRemaining[0]--;
                }
            }

            // Maybe mix in audio pendulum 2 values into right channel
            if (this.audioPendulum2 != null)
            {
                int samplePos = 1; // offset for right channel
                short sampleValue = 0;
                double[] angles;
                double[] velocities;
                float theta;
                float omega;
                float fvalue = 0.0f;
                while ((this.audioPendulumSamplesRemaining[1] > 0) && (samplePos < samples))
                {
                    // Update audio pendulum
                    this.audioPendulum2.Animate();

                    // Get values depending on the value indicator which was selected
                    switch (PendulumGame.State.Players[1].SelectedIndicator)
                    {
                        case 0:
                            angles = audioPendulum2.GetAngle();
                            theta = MathHelper.WrapAngle((float)angles[0]);
                            fvalue = theta / (float)Math.PI * maxAmplitude * (float)(short.MaxValue);
                            break;
                        case 1:
                            angles = audioPendulum2.GetAngle();
                            theta = MathHelper.WrapAngle((float)angles[1]);
                            fvalue = theta / (float)Math.PI * maxAmplitude * (float)(short.MaxValue);
                            break;
                        case 2:
                            velocities = audioPendulum2.GetVelocity();
                            omega = (float)velocities[0];
                            fvalue = (float)omega * 0.1f;
                            if (fvalue < -maxAmplitude)
                            {
                                fvalue = -maxAmplitude;
                            }
                            else if (fvalue > maxAmplitude)
                            {
                                fvalue = maxAmplitude;
                            }

                            fvalue *= (float)(short.MaxValue);
                            break;
                        case 3:
                            velocities = audioPendulum2.GetVelocity();
                            omega = (float)velocities[1];
                            fvalue = (float)omega * 0.1f;
                            if (fvalue < -maxAmplitude)
                            {
                                fvalue = -maxAmplitude;
                            }
                            else if (fvalue > maxAmplitude)
                            {
                                fvalue = maxAmplitude;
                            }

                            fvalue *= (float)(short.MaxValue);
                            break;
                        default:
                            fvalue = 0.0f;
                            break;
                    }


                    sampleValue = Convert.ToInt16(fvalue);
                    buffer[samplePos] = sampleValue;
                    samplePos += 2;

                    this.audioPendulumSamplesRemaining[1]--;
                }
            }

            System.Runtime.InteropServices.Marshal.Copy(buffer, 0, dest, samples);
        }
    }
}
