﻿//-----------------------------------------------------------------------------
// PendulumHelpScreen.cs
//-----------------------------------------------------------------------------

namespace PendulumGame
{
    using System;
    using System.Threading;
    using System.Collections.Generic;

    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Input;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Utilities;
    using NewGamePhysics.GraphicalElements;
    using NewGamePhysics.PhysicalElements;

    /// <summary>
    /// This screen implements the game help
    /// </summary>
    class PendulumHelpScreen : GameScreen
    {
        /// <summary>
        /// The text to display.
        /// </summary>
        private List<string> messages = new List<string>();

        /// <summary>
        /// Display of info messages.
        /// </summary>
        private InfoMessages infoMessages;

        /// <summary>
        /// Unicolored texture as background pane for text
        /// </summary>
        private Texture2D paneTexture;

        #region Initialization

        /// <summary>
        /// Constructor of the screen.
        /// </summary>
        /// <param name="screenMessage">The help message to display.</param>
        public PendulumHelpScreen(string[] screenMessage)
        {
            // Store screen message
            this.messages.AddRange(screenMessage);

            // Add exit message
            string[] exitMessage = { "", "", "[Key] or (Button) to return to Main Menu" };
            this.messages.AddRange(exitMessage);

            TransitionOnTime = TimeSpan.FromSeconds(0.5);
            TransitionOffTime = TimeSpan.FromSeconds(0.5);
        }

        /// <summary>
        /// Load graphics content for the game.
        /// </summary>
        public override void LoadContent()
        {
            // Create message display object
            Vector2 messagesOrigin = new Vector2(
                ScreenManager.GraphicsDevice.Viewport.Width * 0.05f,
                ScreenManager.GraphicsDevice.Viewport.Height * 0.05f);
            this.infoMessages = new InfoMessages(
                this.ScreenManager,
                this.messages,
                messagesOrigin);
            this.infoMessages.Scale = 0.5f;

            // Reset game time
            ScreenManager.Game.ResetElapsedTime();

            // Create translucent pane rectangle 
            if (this.paneTexture == null)
            {
                // New background texture
                this.paneTexture = TextureHelpers.Create(ScreenManager.GraphicsDevice, new Color(64, 64, 64));
            }
        }

        /// <summary>
        /// Unload graphics content used by the screen.
        /// </summary>
        public override void UnloadContent()
        {
        }

        #endregion

        #region Update and Draw

        /// <summary>
        /// Updates the state of the screen.
        /// </summary>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus,
                                                       bool coveredByOtherScreen)
        {
            base.Update(gameTime, otherScreenHasFocus, coveredByOtherScreen);

            if (IsActive)
            {
                this.infoMessages.Update(gameTime);
            }
        }

        /// <summary>
        /// Lets the game respond to player input. Unlike the Update method,
        /// this will only be called when the gameplay screen is active.
        /// </summary>
        public override void HandleInput(InputState input)
        {
            if (input == null)
            {
                throw new ArgumentNullException("input");
            }

            if (input.IsInputCancel(null) || input.IsDisconnected(null))
            {
                ScreenManager.AddScreen(new PendulumPauseMenuScreen(), null);
            }
            else
            {
                PlayerIndex playerIndex;

                // Back to main menu
                if (input.IsMenuSelect(null, out playerIndex))
                {
                    ScreenManager.RemoveScreen(this);
                    ScreenManager.AddScreen(new PendulumMainMenuScreen(), null);
                }
            }
        }

        /// <summary>
        /// Draws the gameplay screen.
        /// </summary>
        public override void Draw(GameTime gameTime)
        {
            Color paneColor = new Color(4, 4, 4);
            Viewport viewport = ScreenManager.GraphicsDevice.Viewport;
            Rectangle backgroundPane =
                new Rectangle(
                    10, 10, viewport.Width - 20, viewport.Height - 20);
            ScreenManager.SpriteBatch.Begin();
            ScreenManager.SpriteBatch.Draw(this.paneTexture, backgroundPane, new Color(paneColor, 128));
            ScreenManager.SpriteBatch.End();

            // Draw messages
            this.infoMessages.Draw(gameTime);

            // If the game is transitioning on or off, fade it out to black.
            if (TransitionPosition > 0)
            {
                ScreenManager.FadeBackBufferToBlack(255 - TransitionAlpha);
            }

            base.Draw(gameTime);
        }

        #endregion
    }
}

