#region File Description
//-----------------------------------------------------------------------------
// PendulumMainMenuScreen.cs
//-----------------------------------------------------------------------------
#endregion

namespace PendulumGame
{
    using System;
    using Microsoft.Xna.Framework;
    using NewGamePhysics.StateManager;

    /// <summary>
    /// The main menu screen is the first thing displayed when the game starts up.
    /// </summary>
    class PendulumMainMenuScreen : MenuScreen
    {
        #region Initialization

        /// <summary>
        /// Constructor fills in the menu contents.
        /// </summary>
        public PendulumMainMenuScreen()
            : base("Main Menu")
        {
            TransitionOnTime = TimeSpan.FromSeconds(1.0);
            TransitionOffTime = TimeSpan.FromSeconds(1.0);

            // Create our menu entries.
            MenuEntry playGameMenuEntry = new MenuEntry("Start Game");
            MenuEntry helpMenuEntry = new MenuEntry("Game Controls Help");
            MenuEntry infoMenuEntry = new MenuEntry("Game Information");
            MenuEntry videoMenuEntry = new MenuEntry("View Pendulum Video");
            MenuEntry exitMenuEntry = new MenuEntry("Exit Game");

            // Hook up menu event handlers.
            playGameMenuEntry.Selected += PlayGameMenuEntrySelected;
            helpMenuEntry.Selected += HelpMenuEntrySelected;
            infoMenuEntry.Selected += InfoMenuEntrySelected;
            videoMenuEntry.Selected += VideoMenuEntrySelected;
            exitMenuEntry.Selected += OnCancel;

            // Add entries to the menu.
            MenuEntries.Add(playGameMenuEntry);
            MenuEntries.Add(helpMenuEntry);
            MenuEntries.Add(infoMenuEntry);
            MenuEntries.Add(videoMenuEntry);
            MenuEntries.Add(exitMenuEntry);
        }

        #endregion

        #region Handle Input

        /// <summary>
        /// Event handler for when the Play Game menu entry is selected.
        /// </summary>
        void PlayGameMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            ScreenManager.AddScreen(new PendulumOptionsMenuScreen(), null);
        }

        /// <summary>
        /// When the user cancels the main menu, ask if they want to exit the program.
        /// </summary>
        protected override void OnCancel(PlayerIndex playerIndex)
        {
            const string message = "Are you sure you want to exit the program?";

            PendulumMessageBoxScreen confirmExitMessageBox = new PendulumMessageBoxScreen(message);

            confirmExitMessageBox.Accepted += ConfirmExitMessageBoxAccepted;

            ScreenManager.AddScreen(confirmExitMessageBox, playerIndex);
        }

        /// <summary>
        /// Event handler for when the user selects ok on the "are you sure
        /// you want to exit" message box.
        /// </summary>
        void ConfirmExitMessageBoxAccepted(object sender, PlayerIndexEventArgs e)
        {
            ScreenManager.Game.Exit();
        }

        /// Event handler for when the Info menu entry is selected.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        void InfoMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            // Text to display
            string[] messages = { 
            "Information on The Pendulum Game", 
            "",
            "The Pendulum Game is a simple dynamics simulation of double",
            "regular + square pendulums (hence the name). The goal is to",
            "touch the 'targets' at the top of the screen for points.",
            "",
            "A player needs to control the friction actuators in the hinges of",
            "the pendulums to inject energy into the system and make the",            
            "pendulum arms swing high enough to reach the 'targets'.",
            "",
            "Options include various gravity environments and friction models.",
            "",
            "The oscillatory systems together with the player interactions ",
            "are used to extract physical entropy (randomness) during gameplay.",
            };

            ScreenManager.AddScreen(new PendulumHelpScreen(messages), null);            

            // Reset back to play entry
            this.SelectedEntry = 0;

            // Set flag to update text when we return
            WaitForUncover = true;
        }

        /// <summary>
        /// Event handler for when the Help menu entry is selected.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        void HelpMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            // Text to display
            string[] messages = { 
            "Controls for The Pendulum Game", 
            "",
            "--- Player A ---",
            "   [Q]/[A] or (A)/(B)- select actuator hinge",
            "   [Z]/[X] or [Left Stick] - inject energy at hinge",
            "   [Left Shift] or [Left Trigger] - switch data display",
            "--- Player B ---",
            "   [Up]/[Down] or (X)/(Y) - select actuator hinge",
            "   [Left]/[Right] or [Right Stick] - inject energy at hinge",
            "   [Right Shift] or [Right Trigger] - switch data display",
            " [Space] or (A)",
            "    continue in menu, game or help screens",
            " [Escape] or (Back)",
            "    pause game or quit to main menu",
            };

            ScreenManager.AddScreen(new PendulumHelpScreen(messages), null);

            // Reset back to play entry
            this.SelectedEntry = 0;

            // Set flag to update text when we return
            WaitForUncover = true;
        }

        /// <summary>
        /// Event handler for when the Video menu entry is selected.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        void VideoMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {

            PendulumLoadingScreen.Load(
                ScreenManager, true, e.PlayerIndex,
                new PendulumVideoScreen());
        }
        #endregion
    }
}
