//-----------------------------------------------------------------------------
// PendulumOptionsMenuScreen.cs
//-----------------------------------------------------------------------------

namespace PendulumGame
{
    using System;
    using System.Text;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Graphics;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Utilities;
    using NewGamePhysics.Physics;
    using NewGamePhysics.GraphicalElements;
    using NewGamePhysics.PhysicalElements;
    using NewGamePhysics.Networking;

    /// <summary>
    /// The options for the gravity chooser.
    /// </summary>
    class PendulumOptionsMenuScreen : MenuScreen
    {
        /// <summary>
        /// Text scroller for informational message.
        /// </summary>
        private ScrollingText scrollingInfoText;

        /// <summary>
        /// Menu entry for choosing gravity
        /// </summary>
        private MenuEntry chooseGravityMenuEntry;

        /// <summary>
        /// The celestial object chooser option.
        /// </summary>
        private MenuEntry celestialObjectMenuEntry;

        /// <summary>
        /// The gravity model chooser option.
        /// </summary>
        private MenuEntry gravityModelMenuEntry;

        /// <summary>
        /// The rotational friction type chooser option.
        /// </summary>
        private MenuEntry rotationalFrictionTypeMenuEntry;

        /// <summary>
        /// The entropy source type chooser option.
        /// </summary>
        private MenuEntry entropySourceTypeMenuEntry;

        #region constructor

        /// <summary>
        /// Constructor for the options menu screen of the gravity chooser.
        /// </summary>
        public PendulumOptionsMenuScreen()
            : base("Game Options")
        {
            this.chooseGravityMenuEntry = new MenuEntry("*"); // text will be set later
            this.chooseGravityMenuEntry.Selected += ChooseGravityMenuEntrySelected;
            MenuEntries.Add(this.chooseGravityMenuEntry);

            this.celestialObjectMenuEntry = new MenuEntry(string.Empty);
            this.celestialObjectMenuEntry.Selected += CelestialObjectMenuEntrySelected;
            MenuEntries.Add(this.celestialObjectMenuEntry);

            this.gravityModelMenuEntry = new MenuEntry(string.Empty);
            this.gravityModelMenuEntry.Selected += GravityModelMenuEntrySelected;
            MenuEntries.Add(this.gravityModelMenuEntry);

            this.rotationalFrictionTypeMenuEntry = new MenuEntry(string.Empty);
            this.rotationalFrictionTypeMenuEntry.Selected += RotationalFrictionTypeMenuEntrySelected;
            MenuEntries.Add(this.rotationalFrictionTypeMenuEntry);

            this.entropySourceTypeMenuEntry = new MenuEntry(string.Empty);
            this.entropySourceTypeMenuEntry.Selected += EntropySourceTypeMenuEntrySelected;
            MenuEntries.Add(this.entropySourceTypeMenuEntry);

            MenuEntry backMenuEntry = new MenuEntry("Back to Main Menu");
            backMenuEntry.Selected += OnCancel;
            MenuEntries.Add(backMenuEntry);

            // Update menu+info text
            UpdateAllTexts();
        }

        /// <summary>
        /// Load content and create graphical elements needed for this screen.
        /// </summary>
        public override void LoadContent()
        {
            // Load base
            base.LoadContent();

            // Create scroller
            SpriteFont font = ScreenManager.Fonts["game"];
            int width = ScreenManager.GraphicsDevice.Viewport.Width;
            int yPos = ScreenManager.GraphicsDevice.Viewport.Height - 32;

            // Text will be set later
            scrollingInfoText = new ScrollingText("-", font, width, yPos);
            scrollingInfoText.TextScale = 0.5f;
            scrollingInfoText.ScrollerSpeed = 200.0f;

            // Pick random info link on 'gravity' and send it 
            Program.Game.SendRandomInfoLink("Gravity");
        }

        /// <summary>
        /// Unload graphics content used by the game screen.
        /// </summary>
        public override void UnloadContent()
        {
            // Unload base
            base.UnloadContent();
        }

        /// <summary>
        /// Update the main menu screen.
        /// </summary>
        /// <param name="gameTime">The current game time.</param>
        /// <param name="otherScreenHasFocus">
        /// Flag indicating of another screen has the focus.
        /// </param>
        /// <param name="coveredByOtherScreen">
        /// Flag indicating of the screen is covered by another screen.
        /// </param>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus, bool coveredByOtherScreen)
        {
            if (WaitForUncover)
            {
                if ((!coveredByOtherScreen) || (!otherScreenHasFocus))
                {
                    // Update the texts
                    UpdateAllTexts();

                    // Update menu text
                    WaitForUncover = false;
                }
            }

            // Update scrollers
            scrollingInfoText.Update(gameTime);

            base.Update(gameTime, otherScreenHasFocus, coveredByOtherScreen);
        }

        /// <summary>
        /// Draw the options screen.
        /// </summary>
        /// <param name="gameTime">The current game time.</param>
        public override void Draw(GameTime gameTime)
        {
            base.Draw(gameTime);

            // Draw scroller
            SpriteBatch spriteBatch = ScreenManager.SpriteBatch;
            scrollingInfoText.Draw(gameTime, spriteBatch);
        }

        #endregion

        #region private_methods

        /// <summary>
        /// Fills in the options screen texts from the game state.
        /// </summary>
        private void UpdateAllTexts()
        {
            UpdateMenuText();
            UpdateInfoText();
        }

        /// <summary>
        /// Fills in the options screen menu text.
        /// </summary>
        private void UpdateMenuText()
        {
            // Menu text
            this.chooseGravityMenuEntry.Text =
                "Choose Gravity and Play on " + PendulumGame.State.CurrentCelestialObject.ToString();

            // Menu text
            celestialObjectMenuEntry.Text =
                "Celestial Object: " + PendulumGame.State.CurrentCelestialObject.ToString();

            gravityModelMenuEntry.Text =
                    "Gravity Model: ";
            if (PendulumGame.State.CurrentCelestialObject == CelestialObject.Earth)
            {
                gravityModelMenuEntry.Text +=
                    PendulumGame.State.CurrentEarthGravityModel.ToString();
            }
            else if (PendulumGame.State.CurrentCelestialObject == CelestialObject.Mars)
            {
                gravityModelMenuEntry.Text +=
                    PendulumGame.State.CurrentMarsGravityModel.ToString();
            }
            else
            {
                gravityModelMenuEntry.Text +=
                    "-not applicable-";
            }

            this.rotationalFrictionTypeMenuEntry.Text =
                "Rotational Friction Model: " + PendulumGame.State.CurrentRotationalFrictionType.ToString();

            this.entropySourceTypeMenuEntry.Text =
                "Entropy Source: " + PendulumGame.State.CurrentEntropySourceType.ToString();
        }

        /// <summary>
        /// Fills in the options screen info text.
        /// </summary>
        private void UpdateInfoText()
        {
            // Scroller text
            if (null != scrollingInfoText)
            {
                string infoText =                
                    "Current Gravity: g = " + PendulumGame.State.CurrentGravity.ToString() + " m/s^-2\n";

                CelestialBody celestialBody = new CelestialBody(PendulumGame.State.CurrentCelestialObject);                
                infoText += 
                    PendulumGame.State.CurrentCelestialObject.ToString() +
                    " Info: " +
                    celestialBody.GetBodyInfo() +
                    "\n";
                infoText = infoText.Replace("\n", "  *  ");
                scrollingInfoText.Text = infoText;
            }
        }

        /// <summary>
        /// Update gravity value based on current settings.
        /// </summary>
        private void UpdateGravity()
        {
            // Update gravity
            GravityCalculator gravityCalculator = new GravityCalculator(PendulumGame.State.CurrentCelestialObject);
            gravityCalculator.EarthGravityModel = PendulumGame.State.CurrentEarthGravityModel;
            PendulumGame.State.CurrentGravity = gravityCalculator.Value;
        }

        #endregion

        #region event_handlers

        /// <summary>
        /// Event handler for when the select gravity menu entry is selected.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        void ChooseGravityMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            PendulumLoadingScreen.Load(
                ScreenManager, 
                true, 
                e.PlayerIndex,                
                new PendulumGravityChooserScreen());

            // Set flag to update text when we return
            WaitForUncover = true;
        }

        /// <summary>
        /// Event handler for the selection of the celestial object menu entry.
        /// </summary>
        private void CelestialObjectMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            // Select next item form the celestial body enumeration
            int currentIndex = EnumConvert.ChangeTo<int>(PendulumGame.State.CurrentCelestialObject);
            int maxIndex = Enum.GetValues(typeof(CelestialObject)).Length;
            currentIndex++;
            if (currentIndex == maxIndex)
            {
                currentIndex = 0;
            }

            PendulumGame.State.CurrentCelestialObject = EnumConvert.ChangeTo<CelestialObject>(currentIndex);

            // New gravity
            UpdateGravity();

            // Update menu
            UpdateAllTexts();
        }

        /// <summary>
        /// Event handler for the selection of the celestial object menu entry.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        private void GravityModelMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            // Select next item form the model body enumeration for the current model
            int currentIndex;
            int maxIndex;
            if (PendulumGame.State.CurrentCelestialObject == CelestialObject.Earth)
            {
                currentIndex = EnumConvert.ChangeTo<int>(PendulumGame.State.CurrentEarthGravityModel);
                maxIndex = Enum.GetValues(typeof(EarthGravityModel)).Length;
                
                // BUG: remove grace models by shortening maxIndex
                maxIndex -= 2;

                currentIndex = (currentIndex + 1) % maxIndex;
                PendulumGame.State.CurrentEarthGravityModel = EnumConvert.ChangeTo<EarthGravityModel>(currentIndex);
            }
            else if (PendulumGame.State.CurrentCelestialObject == CelestialObject.Mars)
            {
                currentIndex = EnumConvert.ChangeTo<int>(PendulumGame.State.CurrentMarsGravityModel);
                maxIndex = Enum.GetValues(typeof(MarsGravityModel)).Length;
                currentIndex = (currentIndex + 1) % maxIndex;
                PendulumGame.State.CurrentMarsGravityModel = EnumConvert.ChangeTo<MarsGravityModel>(currentIndex);
            }
            else
            {
                currentIndex = 0;
            }

            // New gravity
            UpdateGravity();

            // Update menu
            UpdateAllTexts();
        }

        /// <summary>
        /// Event handler for the selection of the rotational friction type menu entry.
        /// </summary>
        private void RotationalFrictionTypeMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            // Select next item from the friction types
            int currentIndex = EnumConvert.ChangeTo<int>(PendulumGame.State.CurrentRotationalFrictionType);
            int maxIndex = Enum.GetValues(typeof(RotationalFrictionType)).Length;
            currentIndex++;
            if (currentIndex == maxIndex)
            {
                currentIndex = 0;
            }

            PendulumGame.State.CurrentRotationalFrictionType = EnumConvert.ChangeTo<RotationalFrictionType>(currentIndex);

            // Update menu
            UpdateMenuText();
        }

        /// <summary>
        /// Event handler for the selection of the entropy Source menu entry.
        /// </summary>
        private void EntropySourceTypeMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            // Select next item from the friction types
            int currentIndex = EnumConvert.ChangeTo<int>(PendulumGame.State.CurrentEntropySourceType);
            int maxIndex = Enum.GetValues(typeof(EntropySourceType)).Length;
            currentIndex++;
            if (currentIndex == maxIndex)
            {
                currentIndex = 0;
            }

            PendulumGame.State.CurrentEntropySourceType = EnumConvert.ChangeTo<EntropySourceType>(currentIndex);
            PendulumGame.State.PhysicalRandomNumberGenerator.EntropySource = PendulumGame.State.CurrentEntropySourceType;

            // Update menu
            UpdateMenuText();
        }

        #endregion
    }
}
