﻿namespace Tests
{
    using System;
    using System.IO;
    using System.Text;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using NewGamePhysics.Mathematics;
    using NewGamePhysics.Utilities;

    /// <summary>
    /// Summary description for UnitTestLegendre
    /// </summary>
    [TestClass]
    public class UnitTestLegendre
    {
        public UnitTestLegendre()
        {
        }

        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        
        /// <summary>
        /// Tests the Legendre Polynomial.
        /// </summary>
        [TestMethod]
        public void LegendrePolynomial()
        {
            // Reference:
            // Mathematica N[LegendreP[n,x],20]
            // n x Pn(x)
            string[] testdata = {
                                    "3 0 0",
                                    "2 0.5 -0.125",
                                    "10 0.3333333333333333 0.23026638893122660841",
                                    "21 -0.75 0.12110918341200137011"                    
                                };

            Scanf scanf = new Scanf();
            object[] t;
            for (int i = 0; i < testdata.Length; i++)
            {
                t = scanf.Scan(testdata[i],"%i %lf %lf");
                int n = (int)t[0];
                double x = (double)t[1];
                double expected = (double)t[2];
                double value = Legendre.Polynomial(n, x);
                double error = Math.Abs(value - expected);
                Console.WriteLine(
                    "Legendre.Polynomial({0},{1}): expected {2} got {3} error {4}",
                    n,
                    x,
                    expected,
                    value,
                    error);
                Assert.IsTrue(error <= 1e-10);
            }
        }

        /// <summary>
        /// Tests the Associated Legendre Function.
        /// </summary>
        [TestMethod]
        public void LegendreAssociatedFunction()
        {
            // Reference:
            // Mathematica N[LegendreP[n,m,x],20]
            // n m x Pnm(x)
            string[] testdata ={
             "1 0 0.0 0.00000",    
             "1 0 0.5 0.500000",    
             "1 0 0.75 0.75",    
             "1 0 1.0 1.0",    
             "1 1 0.5000 -0.86602540378443864676",    
             "2 0 0.5000 -0.125",    
             "2 1 0.5000 -1.2990381056766579701",    
             "2 2 0.5000 2.25",    
             "3 0 0.5000 -0.4375",    
             "3 1 0.5000 -0.32475952641916449254",    
             "3 2 0.5000 5.625",    
             "3 3 0.5000 -9.7427857925749347761",    
             "4 2 0.5000 4.21875",    
             "5 2 0.5000 -4.921875",    
             "6 3 0.5000 12.787406352754601894",    
             "7 3 0.5000 116.68508296888574228",    
             "8 4 0.5000 -1050.66650390625",    
             "9 4 0.5000 -2078.492431640625",    
             "10 5 0.5000 30086.169706116174977",
             "9 5 -0.5 9771.5764857471337533",
             "12 8 0.2 -2.9361651664394649600E07",
             "20 18 0.25 3.2974276572126079987E21"
            };

            Scanf scanf = new Scanf();
            object[] t;
            for (int i = 0; i < testdata.Length; i++)
            {
                t = scanf.Scan(testdata[i], "%i %i %lf %lf");
                int n = (int)t[0];
                int m = (int)t[1];
                double x = (double)t[2];
                double expected = (double)t[3];
                double value = Legendre.AssociatedFunction(n, m, x);
                double error;
                if (Math.Abs(expected) > 0.0)
                {
                    error = Math.Abs(value - expected) / Math.Abs(expected);
                }
                else
                {
                    error = Math.Abs(value - expected);
                }
                Console.WriteLine(
                    "Legendre.AssociatedFunction({0},{1},{2}): expected {3} got {4} rel_error {5}",
                    n,
                    m,
                    x,
                    expected,
                    value,
                    error);
                Assert.IsTrue(error <= 1e-15);
            }
        }

        /// <summary>
        /// Tests the Spherical Associated Legendre Function.
        /// </summary>
        [TestMethod]
        public void LegendreSphericalAssociatedFunction()
        {
            // Reference: 
            // Mathematica N[SphericalHarmonicY[n,m,theta,0],20]
            // n m theta Ynm(theta,0)
            string[] testdata ={
             "1 0 0.0 0.48860251190291992159",    
             "1 0 0.5 0.42878904414183579379",    
             "1 0 0.75 0.35750501926315508333",    
             "1 0 1.0 0.26399306383411281647",    
             "1 1 0.5000 -0.16563871869489602941",    
             "2 0 0.5000 0.41330596756220761898",    
             "2 1 0.5000 -0.32503853318233770541",    
             "2 2 0.5000 0.088784679986342305024",    
             "3 0 0.5000 0.27861659336351639787",    
             "3 1 0.5000 -0.44169847523833372956",    
             "3 2 0.5000 0.20614605996878713307",    
             "3 3 0.5000 -0.045976149181370300989",    
             "4 2 0.5000 0.33762752561059228279",    
             "5 2 0.5000 0.44798449902914186851"    
            };

            Scanf scanf = new Scanf();
            object[] t;
            for (int i = 0; i < testdata.Length; i++)
            {
                t = scanf.Scan(testdata[i], "%i %i %lf %lf");
                int n = (int)t[0];
                int m = (int)t[1];
                double x = (double)t[2];
                double expected = (double)t[3];
                double value = Legendre.SphericalAssociatedFunction(n, m, x);
                double error = Math.Abs(value - expected);
                Console.WriteLine(
                    "Legendre.SphericalAssociatedFunction({0},{1},{2}): expected {3} got {4} error {5}",
                    n,
                    m,
                    x,
                    expected,
                    value,
                    error);
                Assert.IsTrue(error <= 1e-10);
            }                
        }

        /// <summary>
        /// Tests the Normalized Associated Legendre Function with
        /// Deriative as matrix.
        /// </summary>
        [TestMethod]
        public void LegendreNormalizedAssociatedFunctionAndDerivative()
        {
            // Reference: 
            // SHTOOLS generated sample dataset provided by
            // Mark Wieczorek, private communications, 
            // Format:
            // lmax z
            // l m p_lm dp_lm
            string[] testfiles = {
                 @"normalizedLegendre1.dat",
                 @"normalizedLegendre2.dat"
            };

            Scanf scanf = new Scanf();
            object[] t;
            for (int i = 0; i < testfiles.Length; i++)
            {
                string[] testdata = null;

                // Load file
                testdata = File.ReadAllLines(testfiles[i], Encoding.ASCII);
                Console.WriteLine("Read {0} lines from file {1}", testdata.Length, testfiles[i]);

                // Load first row
                t = scanf.Scan(testdata[0], "%i %lf");
                int lmax = (int)t[0];
                double z = (double)t[1];
                Console.WriteLine("lmax={0} z={1}", lmax, z);

                // Load remaining rows
                int n_expected = (lmax + 1 ) * (lmax + 2 ) / 2;
                Console.WriteLine("n_expected={0}", n_expected);
                int[] l = new int[n_expected + 1];
                int[] m = new int[n_expected + 1];
                double[] p_expected = new double[n_expected + 1];
                double[] dp_expected = new double[n_expected + 1];
                for (int j = 1; j <= n_expected; j++)
                {
                    t = scanf.Scan(testdata[j], " %i %i %lf %lf");
                    l[j] = (int)t[0];
                    m[j] = (int)t[1];
                    p_expected[j] = (double)t[2];
                    dp_expected[j] = (double)t[3];
                }

                // Calculate
                double[][] p;
                double[][] dp;
                Legendre.NormalizedAssociatedFunctionAndDerivative(lmax, z, out p, out dp);
                Console.WriteLine(
                    "Legendre.NormalizedAssociatedFunctionAndDerivative({0},{1},p,dp)",
                    lmax,
                    z);

                // Compare
                double error;
                for (int k = 1; k < l.Length; k++)
                {
                    error = Math.Abs(p[l[k]][m[k]] - p_expected[k]);
                    Console.WriteLine("l={0} m={1} p_lm={2} expected {3} error {4}", l[k], m[k], p[l[k]][m[k]], p_expected[k], error);
                    Assert.IsTrue(error <= 1e-12);
                }
                for (int k = 1; k < l.Length; k++)
                {
                    error = Math.Abs(dp[l[k]][m[k]] - dp_expected[k]);
                    Console.WriteLine("l={0} m={1} dp_lm={2} expected {3} error {4}", l[k], m[k], dp[l[k]][m[k]], dp_expected[k], error);
                    Assert.IsTrue(error <= 1e-12);
                }
            }
        }
    }
}
