﻿namespace Tests
{
    using System;
    using System.Text;
    using System.Threading;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using NewGamePhysics.Mathematics;

    /// <summary>
    /// Summary description for UnitTestUnbiaser
    /// </summary>
    [TestClass]
    public class UnitTestUnbiaser
    {
        const int NumBits = 1024;
        const int InputBiasPercent = 25;

        public UnitTestUnbiaser()
        {
        }

        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        [TestMethod]
        public void BitUnbiaserVonNeuman()
        {
            BitUnbiaser unbiaser = new BitUnbiaser(BitUnbiasAlgorithm.VonNeuman);

            string input = RandomBitStream(NumBits, InputBiasPercent);
            string output = unbiaser.Process(input);

            Console.WriteLine("vonNeuman Test");
            Console.WriteLine("Input  = {0}", input);
            Console.WriteLine("Input Bias  = {0}%", CalculateBiasPercent(input));

            Console.WriteLine("Output = {0}", output);
            Console.WriteLine("Output Bias  = {0}%", CalculateBiasPercent(output));

            Console.WriteLine("vonNeuman Yield = {0}%", 100.0 * (double)output.Length / (double)input.Length);
            Console.WriteLine("");
        }

        [TestMethod]
        public void BitUnbiaserMLS()
        {
            BitUnbiaser unbiaser = new BitUnbiaser(BitUnbiasAlgorithm.MLS);

            string input = RandomBitStream(NumBits, InputBiasPercent);
            string output = unbiaser.Process(input);

            Console.WriteLine("MLS Test");
            Console.WriteLine("Input  = {0}", input);
            Console.WriteLine("Input Bias  = {0}%", CalculateBiasPercent(input));

            Console.WriteLine("Output = {0}", output);
            Console.WriteLine("Output Bias  = {0}%", CalculateBiasPercent(output));

            Console.WriteLine("MLS Yield = {0}%", 100.0 * (double)output.Length / (double)input.Length);
            Console.WriteLine("");
        }

        [TestMethod]
        public void BitUnbiaserAMLS()
        {

            BitUnbiaser unbiaser = new BitUnbiaser(BitUnbiasAlgorithm.AMLS);

            string input = RandomBitStream(NumBits, InputBiasPercent);
            string output = unbiaser.Process(input);

            Console.WriteLine("AMLS Test");
            Console.WriteLine("Input  = {0}", input);
            Console.WriteLine("Input Bias  = {0}%", CalculateBiasPercent(input));

            Console.WriteLine("Output = {0}", output);
            Console.WriteLine("Output Bias  = {0}%", CalculateBiasPercent(output));

            Console.WriteLine("AMLS Yield = {0}%", 100.0 * (double)output.Length / (double)input.Length);
            Console.WriteLine("");
        }

        [TestMethod]
        public void ValueUnbiaserPairwise()
        {

            ValueUnbiaser unbiaser = new ValueUnbiaser(ValueUnbiasAlgorithm.Pairwise);

            int dataSize = 40;
            double[] data = new double[dataSize];

            Random rand = new Random();
            Console.WriteLine("PairWise Test");
            for (int i = 0; i< dataSize; i++)
            {
                data[i] = rand.NextDouble() + rand.NextDouble();
                Console.WriteLine(i + " = " + data[i]);
            }

            string output = unbiaser.Extract(data);

            Console.WriteLine("Output = {0}", output);
            Console.WriteLine("Yield = {0}%", 100.0 * (double)output.Length / (double)data.Length);
            Console.WriteLine("");
        }

        private double CalculateBiasPercent(string bitStream)
        {
            int length = bitStream.Length;
            char[] bitStreamC = bitStream.ToCharArray();
            int count = 0;
            for (int i = 0; i < length; i++)
            {
                if (bitStreamC[i] == '1')
                {
                    count++;
                }
            }

            return 100.0 * (double)count / (double)length;
        }

        private string RandomBitStream(int size, int bias)
        {
            Thread.Sleep(100);
            Random rand = new Random();
            StringBuilder sb = new StringBuilder(size);
            for (int i = 0; i < size; i++)
            {
                sb.Append( (rand.Next() % 100 < bias) ? '1' : '0');
            }

            return sb.ToString();
        }
    }
}



