//-----------------------------------------------------------------------------
// GravityChooserGameplayScreen.cs
// (A. Schiffler, 2009-2011)
//-----------------------------------------------------------------------------

namespace GravityChooser
{
    using System;
    using System.Threading;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Input;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Physics;
    using NewGamePhysics.GraphicalElements;
    using NewGamePhysics.PhysicalElements;

    /// <summary>
    /// This screen implements the actual game logic for the gravity chooser.
    /// </summary>
    class GravityChooserGameplayScreen : GameScreen
    {
        #region Fields

        /// <summary>
        /// The content manager.
        /// </summary>
        private ContentManager contentManager;

        /// <summary>
        /// The animated gravity selector shown on this screen.
        /// </summary>
        private GravitySelector gravitySelector;

        /// <summary>
        /// Speed change from user input.
        /// </summary>
        private double speedStep = 0.1;

        /// <summary>
        /// Maximum speed.
        /// </summary>
        private double speedMax = 1.5;

        /// <summary>
        /// Drag on speed change.
        /// </summary>
        private double speedDragFactor = 0.95;

        /// <summary>
        /// Residual longitudonal rotation speed for user interface.
        /// </summary>
        private double longitudeSpeed = 0.0;

        /// <summary>
        /// Residual latitudinal rotation speed for user interface.
        /// </summary>
        private double latitudeSpeed = 0.0;

        #endregion

        #region Initialization

        /// <summary>
        /// Constructor of the screen.
        /// </summary>
        public GravityChooserGameplayScreen()
        {
            TransitionOnTime = TimeSpan.FromSeconds(1.5);
            TransitionOffTime = TimeSpan.FromSeconds(0.5);
        }

        /// <summary>
        /// Load graphics content for the game.
        /// </summary>
        public override void LoadContent()
        {
            if (this.contentManager == null)
            {
                this.contentManager = new ContentManager(ScreenManager.Game.Services, "Content");
            }

            // Create physical element(s)
            GravityCalculator gravityCalculator = 
                new GravityCalculator(GravityChooser.state.CurrentCelestialObject);
            if (GravityChooser.state.CurrentCelestialObject == CelestialObject.Earth)
            {
                gravityCalculator.EarthGravityModel = GravityChooser.state.CurrentEarthGravityModel;
            }

            this.gravitySelector = new GravitySelector(
                ScreenManager, 
                GravityChooser.state.CurrentCelestialObject,
                gravityCalculator);

            // Reset game time
            ScreenManager.Game.ResetElapsedTime();
        }


        /// <summary>
        /// Unload graphics content used by the game.
        /// </summary>
        public override void UnloadContent()
        {
            this.contentManager.Unload();
        }

        #endregion

        #region Update and Draw

        /// <summary>
        /// Updates the state of the game. This method checks the GameScreen.IsActive
        /// property, so the game will stop updating when the pause menu is active,
        /// or if you tab away to a different application.
        /// </summary>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus,
                                                       bool coveredByOtherScreen)
        {
            base.Update(gameTime, otherScreenHasFocus, coveredByOtherScreen);

            if (IsActive)
            {
                // Animate gravity selector
                gravitySelector.Update(gameTime);
            }
        }


        /// <summary>
        /// Lets the game respond to player input. Unlike the Update method,
        /// this will only be called when the gameplay screen is active.
        /// </summary>
        public override void HandleInput(InputState input)
        {
            if (input == null)
            {
                throw new ArgumentNullException("input");
            }

            // Look up inputs for the active player profile.
            PlayerIndex playerIndex;
            if (ControllingPlayer != null)
            {
                playerIndex = ControllingPlayer.Value;
            }
            else
            {
                playerIndex = 0;
            }

            KeyboardState keyboardState = input.CurrentKeyboardStates[(int)playerIndex];
            GamePadState gamePadState = input.CurrentGamePadStates[(int)playerIndex];

            // The game pauses either if the user presses the pause button, or if
            // they unplug the active gamepad. This requires us to keep track of
            // whether a gamepad was ever plugged in, because we don't want to pause
            // on PC if they are playing with a keyboard and have no gamepad at all!
            bool gamePadDisconnected = !gamePadState.IsConnected &&
                                       input.GamePadWasConnected[(int)playerIndex];

            if (input.IsInputCancel(ControllingPlayer) || gamePadDisconnected)
            {
                ScreenManager.AddScreen(new GravityChooserPauseMenuScreen(), ControllingPlayer);
            }
            else
            {
                if (input.IsInputSelect(ControllingPlayer))
                {
                    // Capture selected value
                    GravityChooser.state.CurrentGravity = gravitySelector.Gravity;

                    // Quit to main menu
                    GravityChooserLoadingScreen.Load(
                        ScreenManager, 
                        false, 
                        null, 
                        new GravityChooserBackgroundScreen(),
                        new GravityChooserMainMenuScreen());
                }

                // Keyboard speed adjustment
                if (input.IsInputLeft(null))
                {
                    longitudeSpeed += speedStep;
                }
                else if (input.IsInputRight(null))

                {
                    longitudeSpeed -= speedStep;
                }

                if (input.IsInputUp(null))
                {
                    latitudeSpeed += speedStep;
                }
                else if (input.IsInputDown(null))
                {
                    latitudeSpeed -= speedStep;
                }

                // Speed limiter
                if (longitudeSpeed < -speedMax)
                {
                    longitudeSpeed = -speedMax;
                }
                else if (longitudeSpeed > speedMax)
                {
                    longitudeSpeed = speedMax;
                }

                if (latitudeSpeed < -speedMax)
                {
                    latitudeSpeed = -speedMax;
                }
                else if (latitudeSpeed > speedMax)
                {
                    latitudeSpeed = speedMax;
                }

                // Update rotation
                gravitySelector.Longitude += longitudeSpeed;
                gravitySelector.Latitude += latitudeSpeed;

                // Drag speed
                longitudeSpeed *= speedDragFactor;
                latitudeSpeed *= speedDragFactor;
            }
        }

        /// <summary>
        /// Draws the gameplay screen.
        /// </summary>
        public override void Draw(GameTime gameTime)
        {
            // Match transition to selector size
            gravitySelector.DisplaySize = 1.0f - TransitionPosition;

            // Draw gravity selector
            gravitySelector.Draw(gameTime);

            // If the game is transitioning on or off, fade it out to black.
            if (TransitionPosition > 0)
            {
                ScreenManager.FadeBackBufferToBlack(255 - TransitionAlpha);
            }
        }


        #endregion
    }
}
