﻿using System;
using System.IO;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

using NewGamePhysics.Networking;

namespace InfoBrowser
{
    public partial class InfoBrowserForm : Form
    {
        /// <summary>
        /// Callback for receiver.
        /// </summary>
        /// <param name="infoLink">Infolink instances to receive.</param>
        delegate void AddInfoLinkCallback(InfoLink infoLink);

        /// <summary>
        /// The InfoLink packet receiver.
        /// </summary>
        InfoLinkReceiver infoLinkReceiver;

        /// <summary>
        /// Creates the info browser form.
        /// </summary>
        public InfoBrowserForm()
        {
            InitializeComponent();

            this.toolStripStatusLabel1.Text = string.Empty;

            this.toolStripButton1.Image = new Bitmap(@"Icons\back.gif");
            this.toolStripButton1.Enabled = false;
            this.webBrowser1.CanGoBackChanged += new EventHandler(webBrowser1_CanGoBackChanged);

            this.toolStripButton2.Image = new Bitmap(@"Icons\reload.gif");
            this.toolStripButton2.Enabled = false;

            this.toolStripButton3.Image = new Bitmap(@"Icons\forward.gif");
            this.toolStripButton3.Enabled = false;
            this.webBrowser1.CanGoForwardChanged += new EventHandler(webBrowser1_CanGoForwardChanged);

            this.toolStripButton4.Image = new Bitmap(@"Icons\stop.gif");
            this.toolStripButton4.Enabled = false;

            this.webBrowser1.Navigating += new WebBrowserNavigatingEventHandler(webBrowser1_Navigating);
            this.webBrowser1.Navigated += new WebBrowserNavigatedEventHandler(webBrowser1_Navigated);
            this.webBrowser1.ProgressChanged += new WebBrowserProgressChangedEventHandler(webBrowser1_ProgressChanged);
            this.webBrowser1.DocumentTitleChanged += new EventHandler(webBrowser1_DocumentTitleChanged);

            this.treeViewInfoLinks.NodeMouseClick += new TreeNodeMouseClickEventHandler(treeView1_NodeMouseClick);
            // this.treeView1.Font = new Font(FontFamily.GenericSansSerif, 8.0f, FontStyle.Bold);
            
            // Start the UDP background receiver
            this.infoLinkReceiver = new InfoLinkReceiver(ReceiverCallback);
            this.infoLinkReceiver.StartListener();

            // Add an initial link
            AddInfoLinkToTree(new InfoLink(
                "General Topics",
                "Physics",
                "What is Physics?", 
                new Uri("http://en.wikipedia.org/wiki/Physics")));

            // Write welcome message into browser and clear URL
            string html = File.ReadAllText("Welcome.htm");
            WriteHtmlIntoBrowser(html);
            this.textBox1.Text = string.Empty;

        }

        /// <summary>
        /// Write HTML into web browser.
        /// </summary>
        /// <param name="html">The HTMl to write.</param>
        private void WriteHtmlIntoBrowser(string html)
        {
            this.webBrowser1.AllowNavigation = false;
            if (this.webBrowser1.Document != null)
            {
                this.webBrowser1.Document.OpenNew(true);
            }
            else
            {
                this.webBrowser1.Navigate("about:blank");
            }

            this.webBrowser1.Document.Write(html);
            this.webBrowser1.AllowNavigation = true;
        }

        /// <summary>
        /// Receiver callback.
        /// </summary>
        /// <param name="infoLink"></param>
        void ReceiverCallback(InfoLink infoLink)
        {
            // Check if this method is running on a different thread
            // than the thread that created the control.
            if (this.treeViewInfoLinks.InvokeRequired)
            {
                // It's on a different thread, so use Invoke.
                AddInfoLinkCallback d = 
                    new AddInfoLinkCallback(AddInfoLinkToTree);
                this.Invoke
                    (d, new object[] { infoLink });
            }
            else
            {
                // It's on the same thread, no need for Invoke
                AddInfoLinkToTree(infoLink);
            }
        }

        /// <summary>
        /// Adds an info link to the tree. Creates new category nodes
        /// as needed.
        /// </summary>
        /// <param name="infoLink">The info link object to add.</param>
        private void AddInfoLinkToTree(InfoLink infoLink)
        {
            lock (this)
            {
                // Validate info link
                if ((null == infoLink) ||
                    (string.IsNullOrEmpty(infoLink.Title)) ||
                    (null == infoLink.Uri) ||
                    (string.IsNullOrEmpty(infoLink.Uri.ToString())))
                {
                    this.toolStripStatusLabel1.Text = "Dropped invalid InfoLink.";
                    return;
                }

                // Get category
                string category = string.Empty;
                if (string.IsNullOrEmpty(infoLink.Category))
                {
                    category = "Uncategorized";
                }
                else
                {
                    category = infoLink.Category;
                }

                // Find category node
                TreeNode categoryNode = null;
                foreach (TreeNode node in this.treeViewInfoLinks.Nodes)
                {
                    if ((null != node) &&
                        (node.Name.Equals(category)))
                    {
                        categoryNode = node;
                        break;
                    }
                }

                // Maybe create new category node 
                // (adding spaces since we bold it and text may get clipped otherwise)
                if (null == categoryNode)
                {
                    categoryNode = new TreeNode();
                    categoryNode.Name = category;
                    categoryNode.Text = category + "      ";
                    categoryNode.Tag = null;
                    categoryNode.ToolTipText = "InfoLink Category: " + category;
                    categoryNode.NodeFont = new Font(this.treeViewInfoLinks.Font, FontStyle.Bold);
                    this.treeViewInfoLinks.Nodes.Add(categoryNode);
                }

                // Get sub-category
                string subCategory = string.Empty;
                if (string.IsNullOrEmpty(infoLink.SubCategory))
                {
                    subCategory = "Uncategorized";
                }
                else
                {
                    subCategory = infoLink.SubCategory;
                }

                // Find sub-category node
                TreeNode subCategoryNode = null;
                foreach (TreeNode node in categoryNode.Nodes)
                {
                    if ((null != node) &&
                        (node.Name.Equals(subCategory)))
                    {
                        subCategoryNode = node;
                        break;
                    }
                }

                // Maybe create new sub-category node
                if (null == subCategoryNode)
                {
                    subCategoryNode = new TreeNode();
                    subCategoryNode.Name = subCategory;
                    subCategoryNode.Text = subCategory;
                    subCategoryNode.Tag = null;
                    subCategoryNode.ToolTipText = "InfoLink Sub-Category: " + subCategory;
                    subCategoryNode.NodeFont = new Font(this.treeViewInfoLinks.Font, FontStyle.Italic);
                    categoryNode.Nodes.Add(subCategoryNode);
                }

                // Find info node
                TreeNode infoNode = null;
                foreach (TreeNode node in subCategoryNode.Nodes)
                {
                    // Check if we have this infoLink already
                    if (node.Name.Equals(infoLink.Title))
                    {
                        infoNode = node;
                        break;
                    }
                }

                // Maybe create new info node which stores tag
                if (null == infoNode)
                {
                    infoNode = new TreeNode();
                    infoNode.Name = infoLink.Title;
                    infoNode.Text = infoLink.Title;
                    infoNode.Tag = infoLink;
                    infoNode.ToolTipText = "InfoLink to: " + infoLink.Uri.ToString();
                    infoNode.NodeFont = new Font(this.treeViewInfoLinks.Font, FontStyle.Underline);
                    infoNode.ForeColor = Color.Blue;
                    subCategoryNode.Nodes.Add(infoNode);
                    categoryNode.Expand();
                    subCategoryNode.ExpandAll();
                }

                // Mark node as newer
                infoNode.BackColor = Color.LightGreen;
            }
        }

        /// <summary>
        /// Handle click in tree view node.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The received event.</param>
        private void treeView1_NodeMouseClick(
            object sender, 
            TreeNodeMouseClickEventArgs e)
        {
            TreeNode node = e.Node;

            if (null != node.Tag)
            {
                NavigateToUrl(((InfoLink)node.Tag).Uri);
                node.BackColor = Color.White;
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The received event.</param>
        private void toolStripButton1_Click(
            object sender, 
            EventArgs e)
        {
            if (this.webBrowser1.CanGoBack)
            {
                this.webBrowser1.GoBack();
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The received event.</param>
        private void toolStripButton2_Click(
            object sender, 
            EventArgs e)
        {
            this.webBrowser1.Refresh();
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The received event.</param>
        private void toolStripButton3_Click(
            object sender, 
            EventArgs e)
        {
            if (this.webBrowser1.CanGoForward)
            {
                this.webBrowser1.GoForward();
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The received event.</param>
        private void toolStripButton4_Click(
            object sender, 
            EventArgs e)
        {
            this.webBrowser1.Stop();
        }

        #region browser_event_handlers

        /// <summary>
        /// Updates the form title when the document title is changed.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The received event.</param>
        private void webBrowser1_DocumentTitleChanged(
            object sender,
            EventArgs e)
        {
            this.Text = "InfoBrowser";
            if (!string.IsNullOrEmpty(this.webBrowser1.DocumentTitle))
            {
                this.Text += " - " + this.webBrowser1.DocumentTitle;
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The received event.</param>
        private void webBrowser1_CanGoBackChanged(
            object sender, 
            EventArgs e)
        {
            this.toolStripButton1.Enabled = webBrowser1.CanGoBack;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The received event.</param>
        private void webBrowser1_CanGoForwardChanged(
            object sender, 
            EventArgs e)
        {
            this.toolStripButton3.Enabled = webBrowser1.CanGoForward;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The received event.</param>
        private void webBrowser1_Navigated(
            object sender, 
            WebBrowserNavigatedEventArgs e)
        {
            // Disable stop button
            this.toolStripButton4.Enabled = false;

            // Update url
            this.textBox1.Text = this.webBrowser1.Url.ToString();

            // Update reload status button
            this.toolStripButton2.Enabled =
                (!string.IsNullOrEmpty(this.webBrowser1.Url.ToString()));
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The received event.</param>
        private void webBrowser1_ProgressChanged(
            object sender, 
            WebBrowserProgressChangedEventArgs e)
        {
            // Update status message
            if (e.CurrentProgress == e.MaximumProgress)
            {
                // Disable stop button
                this.toolStripButton4.Enabled = false;

                // Set message
                this.toolStripStatusLabel1.Text = string.Empty;
            }
            else
            {
                // Enable stop button
                this.toolStripButton4.Enabled = true;

                // Set message
                this.toolStripStatusLabel1.Text = "Loading ...";
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The received event.</param>
        private void webBrowser1_Navigating(object sender, WebBrowserNavigatingEventArgs e)
        {
            // Nothing to do
        }

        #endregion

        /// <summary>
        /// 
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The received event.</param>
        private void textBox1_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                string url = this.textBox1.Text;

                if (string.IsNullOrEmpty(url))
                {
                    return;
                }

                // Add http
                if (!url.Equals("about:blank") &&
                    !url.StartsWith("http://") &&
                    !url.StartsWith("https://"))
                {
                    url = "http://" + url;
                }

                Uri uri;
                try
                {
                    uri = new Uri(url);
                }
                catch (System.UriFormatException)
                {
                    this.toolStripStatusLabel1.Text = "Error - invalid URL.";
                    return;
                }

                NavigateToUrl(uri);
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="url">The url</param>
        private void NavigateToUrl(Uri uri)
        {
            if (null == uri)
            {
                return;
            }

            // Browse
            this.webBrowser1.Navigate(uri);
        }

        private void InfoBrowserForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            this.infoLinkReceiver.StopListener();
            e.Cancel = false;
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        private void aboutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            MessageBox.Show(
                "InfoLink Receiver with Embedded Browser" + Environment.NewLine+
                "Receiving UDP packets on port: " + this.infoLinkReceiver.Port + Environment.NewLine +
                "(C) A. Schiffler, 2009" + Environment.NewLine,
                "InfoBrowser About");
        }

        private void saveInfoLinksAsHTMLToolStripMenuItem_Click(object sender, EventArgs e)
        {
            SaveFileDialog saveFileDialog = new SaveFileDialog();
            saveFileDialog.InitialDirectory = Environment.GetFolderPath(Environment.SpecialFolder.Desktop);
            saveFileDialog.FileName =
                "InfoLinks-" +
                DateTime.Now.ToString("yyyyMMMd") +
                ".htm";
            saveFileDialog.Filter = "HTML File|*.htm";
            saveFileDialog.Title = "Save InfoLinks as HTML";
            saveFileDialog.AddExtension = true;
            saveFileDialog.ShowDialog();

            if (saveFileDialog.FileName != "")
            {
                try
                {
                    FileStream fs = (FileStream)saveFileDialog.OpenFile();
                    SaveNodesAsHtmlPage(fs);
                    fs.Close();
                    this.toolStripStatusLabel1.Text = "Saved to file: " + saveFileDialog.FileName;
                }
                catch (Exception)
                {
                    this.toolStripStatusLabel1.Text = "Error - save failed.";
                }
            }
        }

        private void SaveNodesAsHtmlPage(FileStream fs)
        {
            StreamWriter writer = new StreamWriter(fs);
            writer.WriteLine("<html>");
            writer.WriteLine("<title>InfoLinks</title>");
            writer.WriteLine("<h1>InfoLinks Menu</h1>");
            foreach (TreeNode categoryNode in this.treeViewInfoLinks.Nodes)
            {
                writer.WriteLine("<dt><h3>" + categoryNode.Text + "</h3>");
                writer.WriteLine("<dl><p>");                
                foreach (TreeNode linkNode in categoryNode.Nodes)
                {
                    writer.WriteLine(
                        "<dt><a href=\"" + 
                        linkNode.Tag.ToString() + 
                        "\">" + linkNode.Text + "</a>");
                }
                writer.WriteLine("</dl><p>");
            }
            writer.WriteLine("</html>");
            writer.Flush();
        }
    }
}
