//-----------------------------------------------------------------------------
// MontyHallGameplayScreen.cs
// (C) A. Schiffler, 2009-2011
//-----------------------------------------------------------------------------

namespace MontyHallGame
{
    using System;
    using System.Threading;

    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Input;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Utilities;
    using NewGamePhysics.GraphicalElements;
    using NewGamePhysics.PhysicalElements;

    /// <summary>
    /// This screen implements the actual game logic. It is just a
    /// placeholder to get the idea across: you'll probably want to
    /// put some more interesting gameplay in here!
    /// </summary>
    class MontyHallGameplayScreen : GameScreen
    {
        #region Fields

        /// <summary>
        /// The content manager.
        /// </summary>
        private ContentManager contentManager;

        /// <summary>
        /// Text scroller for informational message.
        /// </summary>
        private ScrollingText scrollingInfoText;

        /// <summary>
        /// Open/close state for the doors.
        /// </summary>
        private int[] doorState;

        /// <summary>
        /// The current game phase:
        ///     1 = user can preselect a door
        ///     2 = monty can open a door
        ///     3 = user can open a door
        ///     4 = calculate game result
        /// </summary>
        private int gamePhase;

        /// <summary>
        /// The position of the selector.
        /// </summary>
        private int selectorPosition;

        /// <summary>
        /// The preselected door.
        /// </summary>
        private int preselectedDoor;

        /// <summary>
        /// The position of the prize.
        /// </summary>
        private int prizePosition;

        /// <summary>
        /// Ticks to wait until transitioning a game state.
        /// </summary>
        private long waitTicks;

        #endregion

        #region Initialization

        /// <summary>
        /// Constructor of the screen.
        /// </summary>
        public MontyHallGameplayScreen()
        {
            TransitionOnTime = TimeSpan.FromSeconds(1.5);
            TransitionOffTime = TimeSpan.FromSeconds(1.5);
        }

        /// <summary>
        /// Load graphics content for the game.
        /// </summary>
        public override void LoadContent()
        {
            if (this.contentManager == null)
            {
                this.contentManager = new ContentManager(ScreenManager.Game.Services, "Content");
            }

            // Reset doors
            doorState = new int[3];
            doorState[0] = 0;
            doorState[1] = 0;
            doorState[2] = 0;

            // Reset state
            gamePhase = 1;
            preselectedDoor = -1; // none
            selectorPosition = 1; // middle

            // Randomly selecy a prize position
            prizePosition = MontyHallGame.state.Rng.Next(0, 2);

            // Reset delay
            waitTicks = 0;

            // Reset game time
            ScreenManager.Game.ResetElapsedTime();

            // Reset play accumulator
            MontyHallGame.state.playTime = 0.0;

            // Create scrollers
            SpriteFont font = ScreenManager.Fonts["menu"];
            MontyHallGame game = (MontyHallGame)ScreenManager.Game;
            int width = game.GraphicsDevice.Viewport.Width;
            scrollingInfoText = new ScrollingText("", font, width, 30);
            scrollingInfoText.TextColor = Color.Black;

            // Play sound
            ScreenManager.Sounds["show"].Play();

            // Set initial text
            scrollingInfoText.Text = "Press [Left] or [Right], then [Space] to pre-select a door to Open later.";
        }

        /// <summary>
        /// Unload graphics content used by the game.
        /// </summary>
        public override void UnloadContent()
        {
            this.contentManager.Unload();
        }

        #endregion

        #region Update and Draw

        /// <summary>
        /// Updates the state of the game. This method checks the GameScreen.IsActive
        /// property, so the game will stop updating when the pause menu is active,
        /// or if you tab away to a different application.
        /// </summary>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus,
                                                       bool coveredByOtherScreen)
        {
            base.Update(gameTime, otherScreenHasFocus, coveredByOtherScreen);

            if (IsActive)
            {
                // Accumulate active game time
                MontyHallGame.state.playTime += gameTime.ElapsedRealTime.TotalSeconds;

                // Handle game phases
                switch (gamePhase)
                {
                    case 1:
                        // Nothing to do here, we just wait for the
                        // game phase to change based on user input.
                        break;
                    case 2:
                        // Switch the doors around, hidden to the user
                        NextDoor();
                        break;
                    case 3:
                        // Nothing to do here, we just wait for the
                        // game phase to change based on user input.
                        break;
                    case 4:
                        if (waitTicks == 0)
                        {
                            // Add a 1 sec delay to see the gold.
                            waitTicks = gameTime.TotalRealTime.Ticks + 1 * (1000000000/100);
                        }
                        else if (gameTime.TotalRealTime.Ticks > waitTicks)
                        {
                            // Determine if there was a winner and end game
                            if (doorState[prizePosition] == 1)
                            {
                                MontyHallGame.state.Wins++;

                                // Show game over screen
                                ScreenManager.AddScreen(
                                    new MontyHallGameOverScreen("*** Winner ***"), null);

                                // Play sound
                                ScreenManager.Sounds["money"].Play();
                            }
                            else
                            {
                                MontyHallGame.state.Losses++;

                                // Show game over screen
                                ScreenManager.AddScreen(
                                    new MontyHallGameOverScreen("Sorry - You Lost!"), null);

                                // Play sound
                                ScreenManager.Sounds["loser"].Play();
                            }
                        }
                        break;
                }

                // Update scrollers
                scrollingInfoText.Update(gameTime);
            }
        }

        /// <summary>
        /// Lets the game respond to player input. Unlike the Update method,
        /// this will only be called when the gameplay screen is active.
        /// </summary>
        public override void HandleInput(InputState input)
        {
            if (input == null)
            {
                throw new ArgumentNullException("input");
            }

            // Look up inputs for the active player profile.
            int playerIndex = (int)ControllingPlayer.Value;

            KeyboardState keyboardState = input.CurrentKeyboardStates[playerIndex];
            GamePadState gamePadState = input.CurrentGamePadStates[playerIndex];

            // The game pauses either if the user presses the pause button, or if
            // they unplug the active gamepad. This requires us to keep track of
            // whether a gamepad was ever plugged in, because we don't want to pause
            // on PC if they are playing with a keyboard and have no gamepad at all!
            bool gamePadDisconnected = !gamePadState.IsConnected &&
                                       input.GamePadWasConnected[playerIndex];

            if (input.IsInputCancel(ControllingPlayer) || gamePadDisconnected)
            {
                ScreenManager.AddScreen(new MontyHallPauseMenuScreen(), ControllingPlayer);
            }
            else
            {

                // Do doors
                if (input.IsNewInputRight(ControllingPlayer))
                {
                    // Play sound
                    ScreenManager.Sounds["click"].Play();

                    // Switc doors
                    NextDoor();
                }
                else if (input.IsNewInputLeft(ControllingPlayer))
                {
                    // Play sound
                    ScreenManager.Sounds["click"].Play();

                    // Switch doors
                    PreviousDoor();
                }

                // Open or select a door
                if (input.IsInputSelect(ControllingPlayer))
                {
                    switch (gamePhase)
                    {
                        case 1:
                            // Lock in the preselected position
                            preselectedDoor = selectorPosition;

                            // Switch Doors
                            NextDoor();

                            // Update Text
                            scrollingInfoText.Text = "Press [Space] and Monty will help by opening loosing door for you.";

                            // Play sound
                            ScreenManager.Sounds["crowd"].Play(0.33f, 0.0f, 0.0f);

                            // Next Game Phase
                            gamePhase++;
                            break;
                        case 2:
                            // Update selector so we do not reveal the prize
                            if (selectorPosition == prizePosition)
                            {
                                NextDoor();
                            }

                            // Open the door where the selector is placed                            
                            doorState[selectorPosition] = 1;

                            // Play sound
                            ScreenManager.Sounds["door"].Play();

                            // Set next door to the preselected one
                            selectorPosition = preselectedDoor;

                            // Update Text
                            scrollingInfoText.Text = "Stay or switch doors? Use [Left] or [Right] to select another door, then press [Space] try to get your prize.";

                            // Next Game Phase
                            gamePhase++;
                            break;
                        case 3:
                            // Open the door where the selector is placed
                            doorState[selectorPosition] = 1;

                            // Update number of door-switches
                            if (selectorPosition != preselectedDoor)
                            {
                                MontyHallGame.state.Switches++;
                            }

                            // Clear text
                            scrollingInfoText.Text = "";

                            // Play sound
                            ScreenManager.Sounds["door"].Play();

                            // Next Game Phase
                            gamePhase++;
                            break;
                    }
                }
            }
        }

        /// <summary>
        /// Draws the gameplay screen.
        /// </summary>
        public override void Draw(GameTime gameTime)
        {
            // Get drawing helpers and textures
            PrimitiveBatch primitiveBatch = ScreenManager.PrimitiveBatch;
            SpriteBatch spriteBatch = ScreenManager.SpriteBatch;
            Viewport viewport = ScreenManager.GraphicsDevice.Viewport;

            // Determine door texture
            string doorTextureName = "mdoor_";
            doorTextureName += (doorState[0] == 0) ? "c" : "o";
            doorTextureName += (doorState[1] == 0) ? "c" : "o";
            doorTextureName += (doorState[2] == 0) ? "c" : "o";
            Texture2D doorTexture = ScreenManager.Textures[doorTextureName];

            // Draw doors
            double aspectRatio = 1024.0 / 768.0;
            int displayWidth = (int)Math.Truncate((double)viewport.Height * aspectRatio);
            Rectangle dest = new Rectangle(
                (viewport.Width - displayWidth) / 2, 0,
                displayWidth, viewport.Height);
            spriteBatch.Begin(SpriteBlendMode.AlphaBlend);
            spriteBatch.Draw(doorTexture, dest, Color.White);
            spriteBatch.End();

            // Draw selector
            if ((gameTime.TotalRealTime.Milliseconds % 500) < 400)
            {
                switch (gamePhase)
                {
                    case 1:
                        dest = new Rectangle(
                        viewport.Width / 2 +
                            (int)((float)displayWidth * 0.29f * (float)(selectorPosition - 1))
                            - 60,
                        viewport.Height - 100,
                        100, 50);
                        spriteBatch.Begin(SpriteBlendMode.AlphaBlend);
                        spriteBatch.Draw(ScreenManager.Textures["opensign"], dest, Color.White);
                        spriteBatch.End();
                        break;
                    case 2:
                        for (int i = 0; i < 3; i++)
                        {
                            if (i != preselectedDoor)
                            {
                                dest = new Rectangle(
                                viewport.Width / 2 +
                                    (int)((float)displayWidth * 0.29f * (float)(i - 1))
                                    - 55,
                                viewport.Height - 100,
                                90, 90);
                                spriteBatch.Begin(SpriteBlendMode.AlphaBlend);
                                spriteBatch.Draw(ScreenManager.Textures["monty"], dest, Color.White);
                                spriteBatch.End();
                            }
                        }
                        break;
                    case 3:
                        dest = new Rectangle(
                        viewport.Width / 2 +
                            (int)((float)displayWidth * 0.29f * (float)(selectorPosition - 1))
                            - 60,
                        viewport.Height - 100,
                        100, 100);
                        spriteBatch.Begin(SpriteBlendMode.AlphaBlend);
                        spriteBatch.Draw(ScreenManager.Textures["key"], dest, Color.White);
                        spriteBatch.End();
                        break;
                }
            }

            // Draw pre-selection if set
            if (preselectedDoor >= 0)
            {
                dest = new Rectangle(
                viewport.Width / 2 +
                    (int)((float)displayWidth * 0.29f * (float)(preselectedDoor - 1))
                    - 30,
                (int)((float)viewport.Height * 0.25f),
                50, 25);
                spriteBatch.Begin(SpriteBlendMode.AlphaBlend);
                spriteBatch.Draw(ScreenManager.Textures["opensign"], dest, Color.White);
                spriteBatch.End();
            }

            // Draw prize if won
            if ((gamePhase == 4) && (doorState[prizePosition] != 0))
            {
                dest = new Rectangle(
                viewport.Width / 2 +
                    (int)((float)displayWidth * 0.29f * (float)(prizePosition - 1))
                    - 35,
                viewport.Height - 190,
                80, 80);
                spriteBatch.Begin(SpriteBlendMode.AlphaBlend);
                spriteBatch.Draw(ScreenManager.Textures["potogold"], dest, Color.White);
                spriteBatch.End();
            }

            // Draw scroller
            scrollingInfoText.Draw(gameTime, spriteBatch);

            // If the game is transitioning on or off, fade it out to black.
            if (TransitionPosition > 0)
            {
                ScreenManager.FadeBackBufferToBlack(255 - TransitionAlpha);
            }
        }

        #endregion

        /// <summary>
        /// Advance selector to next closed door
        /// which is not preselected.
        /// </summary>
        private void NextDoor()
        {
            do
            {
                selectorPosition = (selectorPosition + 1);
                if (selectorPosition > 2)
                {
                    selectorPosition = 0;
                }
            } while (((selectorPosition == preselectedDoor) && (gamePhase < 3)) ||
                (doorState[selectorPosition] != 0));
        }

        /// <summary>
        /// Advance selector to previous closed door
        /// which is not preselected.
        /// </summary>
        private void PreviousDoor()
        {
            do
            {
                selectorPosition = (selectorPosition - 1);
                if (selectorPosition < 0)
                {
                    selectorPosition = 2;
                }
            } while (((selectorPosition == preselectedDoor) && (gamePhase < 3)) ||
                (doorState[selectorPosition] != 0));
        }
    }
}
